package csbase.client.csdk.v2.filesystem;

import java.awt.Window;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import csbase.client.ClientLocalFile;
import csbase.client.csdk.v2.core.CSDKAbstractContext;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.logic.applicationservice.ApplicationRegistry;
import csdk.v2.api.filesystem.FileSelectionType;
import csdk.v2.api.filesystem.IFile;
import csdk.v2.api.filesystem.local.ILocalFileSystemContext;
import csdk.v2.api.filesystem.local.LocalFileSystemException;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;

/**
 * Implementao do contexto de sistema local de arquivos do CSDK, dando acesso
 * ao sistema de arquivos local do cliente CSBASE.
 */
public class CSDKLocalFileSystemContext extends CSDKAbstractContext implements
  ILocalFileSystemContext {

  /**
   * O registro da aplicao.
   */
  private ApplicationRegistry registry;

  /**
   * Construtor.
   *
   * @param registry o registro da aplicao.
   */
  public CSDKLocalFileSystemContext(ApplicationRegistry registry) {
    this.registry = registry;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getLocalFile(String[] path) throws LocalFileSystemException {
    String joinedPath = FileUtils.joinPath(path);
    File file = new File(joinedPath);
    if (file.exists()) {
      ClientLocalFile localFile = new ClientLocalFile(file);
      return new CSDKFile(localFile);
    }
    else {
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile createLocalFile(String[] path, String name, boolean isDir)
    throws LocalFileSystemException {
    File parent;
    if (path == null || path.length == 0) {
      parent = new File(".");
    }
    else {
      parent = new File(FileUtils.joinPath(path));
    }
    File file = new File(parent, name);
    if (file.exists()) {
      return null;
    }
    try {
      if (isDir) {
        if (!file.mkdir()) {
          return null;
        }
      }
      else {
        if (!file.createNewFile()) {
          return null;
        }
      }
    }
    catch (IOException e) {
      throw new LocalFileSystemException(e);
    }
    ClientLocalFile localFile = new ClientLocalFile(file);
    return new CSDKFile(localFile);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile browseLocalFileInOpenMode(String[] fileTypes, boolean allowAll,
    IFile currentDir, FileSelectionType selectionType, Window frame)
    throws LocalFileSystemException {
    ClientLocalFile currentDirectory = null;
    if (currentDir != null) {
      String[] currentPath = currentDir.getPath();
      String joinedCurrentPath = FileUtils.joinPath(currentPath);
      File localCurrentDir = new File(joinedCurrentPath);
      currentDirectory = new ClientLocalFile(localCurrentDir);
    }
    List<String> fileTypeList;
    if (fileTypes != null) {
      fileTypeList = Arrays.asList(fileTypes);
    }
    else {
      fileTypeList = Collections.emptyList();
    }
    ClientLocalFile file;
    if (selectionType == FileSelectionType.DIRECTORIES_ONLY) {
      file =
        ClientLocalFileChooserUtil.browseSingleDirectoryInOpenMode(frame,
          fileTypeList, registry.getApplicationName(LNG.getLocale()), allowAll,
          currentDirectory);
    }
    else {
      file =
        ClientLocalFileChooserUtil.browseSingleFileInOpenMode(frame,
          fileTypeList, registry.getApplicationName(LNG.getLocale()), allowAll,
          currentDirectory);
    }
    if (file == null) {
      return null;
    }
    return new CSDKFile(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] browseMultipleLocalFilesInOpenMode(String[] fileTypes,
    boolean allowAll, IFile currentDir, FileSelectionType selectionType,
    Window frame) throws LocalFileSystemException {
    ClientLocalFile currentDirectory = null;
    if (currentDir != null) {
      String[] currentPath = currentDir.getPath();
      String joinedCurrentPath = FileUtils.joinPath(currentPath);
      File localCurrentDir = new File(joinedCurrentPath);
      currentDirectory = new ClientLocalFile(localCurrentDir);
    }
    List<String> fileTypeList;
    if (fileTypes != null) {
      fileTypeList = Arrays.asList(fileTypes);
    }
    else {
      fileTypeList = Collections.emptyList();
    }
    List<ClientLocalFile> files;
    if (selectionType == FileSelectionType.DIRECTORIES_ONLY) {
      files =
        ClientLocalFileChooserUtil.browseMultipleDirectoriesInOpenMode(frame,
          fileTypeList, registry.getApplicationName(LNG.getLocale()), allowAll,
          currentDirectory);
    }
    else {
      files =
        ClientLocalFileChooserUtil.browseMultipleFilesInOpenMode(frame,
          fileTypeList, registry.getApplicationName(LNG.getLocale()), allowAll,
          currentDirectory);
    }
    if (files == null) {
      return null;
    }
    List<IFile> selectedFiles = new ArrayList<IFile>();
    for (ClientLocalFile file : files) {
      selectedFiles.add(new CSDKFile(file));
    }
    return selectedFiles.toArray(new IFile[selectedFiles.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile browseLocalFileInSaveMode(String[] fileTypes, boolean allowAll,
    IFile currentDir, Window frame)
    throws LocalFileSystemException {
    ClientLocalFile currentDirectory = null;
    if (currentDir != null) {
      String[] currentPath = currentDir.getPath();
      String joinedCurrentPath = FileUtils.joinPath(currentPath);
      File localCurrentDir = new File(joinedCurrentPath);
      currentDirectory = new ClientLocalFile(localCurrentDir);
    }
    List<String> fileTypeList;
    if (fileTypes != null) {
      fileTypeList = Arrays.asList(fileTypes);
    }
    else {
      fileTypeList = Collections.emptyList();
    }
    ClientLocalFile file =
      ClientLocalFileChooserUtil.browseSingleFileInSaveMode(frame,
        fileTypeList, null, registry.getApplicationName(LNG.getLocale()),
        allowAll, currentDirectory);
    if (file == null) {
      return null;
    }
    return new CSDKFile(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeLocalFile(String[] path) {
    String joinedPath = FileUtils.joinPath(path);
    File file = new File(joinedPath);
    return file.delete();
  }
}
