/*
 * $Id$
 */
package csbase.client.desktop;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Locale;
import java.util.Set;
import java.util.SortedSet;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.AbstractTableModel;

import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.ClientServerManager;
import csbase.client.remote.ClientRemoteMonitorListener;
import csbase.logic.LoginInfo;
import csbase.logic.MonitoredServerListener;
import csbase.logic.ServerURI;
import csbase.logic.server.ServerInfo;
import csbase.remote.ServerServiceInterface;

/**
 * Frame usado para fins de teste que possibilita visualizar os servidores onde
 * o cliente est conectado e tambm conectar explicitamente em um servidor
 * qualquer sendo possvel o login tradicional com login e senha ou usando o
 * servidor default como referncia. O login por referncia copia os atributos
 * de sesso do servidor de referncia. Tambm  possvel fazer logout e trocar
 * o servidor default que por consequncia repontera o locator para o servidor
 * escolhido.
 * 
 * @author Tecgraf/PUC-Rio
 * 
 */
public final class ServerStatusFrame extends DesktopComponentFrame {

  /**
   * Modelo.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class ServerTableModel extends AbstractTableModel {

    /**
     * Headers
     */
    String[] headers = new String[] { "URI", "Conectado?", "Padro?" };

    /**
     * Valores
     */
    Object[][] values;

    /**
     * Construtor
     */
    public ServerTableModel() {
      this.loadData();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getColumnCount() {
      return values[0].length;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getRowCount() {
      return values.length;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
      return values[rowIndex][columnIndex];
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setValueAt(Object value, int rowIndex, int columnIndex) {

      if (columnIndex == 2) {
        if ((Boolean) values[rowIndex][columnIndex] == false) {
          ClientServerManager.getInstance().setDefaultServer(
            (ServerURI) values[rowIndex][0]);
          this.loadData();
        }
      }
      else if (columnIndex == 1 && !(Boolean) values[rowIndex][2]) {
        final ServerURI sURI = (ServerURI) values[rowIndex][0];

        if ((Boolean) value) {
          login(sURI);
        }
        else {
          logout(sURI);
        }

        this.loadData();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName(int column) {
      return this.headers[column];
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {
      return getValueAt(0, columnIndex).getClass();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
      return (columnIndex == 2 || columnIndex == 1);
    }

    /**
     * 
     */
    protected void loadData() {
      Set<ServerURI> availableServers =
        ClientServerManager.getInstance().getManagedServers();
      this.values = new Object[availableServers.size()][headers.length];
      int c = 0;
      for (ServerURI sURI : availableServers) {
        this.values[c][0] = sURI;
        this.values[c][1] = ClientServerManager.getInstance().isAlive(sURI);
        this.values[c++][2] = ClientServerManager.getInstance().isDefault(sURI);
      }
      this.fireTableDataChanged();
    }

  }

  private ServerTableModel serverTableModel;
  private JCheckBox usePasswordCheckBox;
  private MonitoredServerListener monitoredServerListener;
  private static ServerStatusFrame frame;
  private JTextField jtfUser;
  private JPasswordField jtfPassword;
  private JCheckBox cloneServerSessionCheckBox;

  /**
   * Construtor
   * 
   * @param owner janela me.
   */
  private ServerStatusFrame(JFrame owner) {
    super("Servidores conectados");
    this.serverTableModel = new ServerTableModel();
    this.monitoredServerListener = new ClientRemoteMonitorListener(this);
    ServerStatusListener l = new ServerStatusListener();
    ClientServerManager.getInstance().addCommonListener(l);
    this.buildGui();
  }

  /**
   * Exibe o dilogo
   * 
   * @param owner A janela pai
   */
  public static void show(JFrame owner) {
    if (frame == null) {
      frame = new ServerStatusFrame(owner);
    }
    frame.setLocationRelativeTo(owner);
    frame.setVisible(true);
  }

  /**
   * @return painel.
   */
  private JPanel buildMainPanel() {
    SortableTable table = new SortableTable(this.serverTableModel);
    JPanel panel = new JPanel(new GridBagLayout());

    GridBagConstraints gbc = new GridBagConstraints();
    gbc.insets = new Insets(4, 4, 4, 4);
    gbc.fill = GridBagConstraints.BOTH;

    gbc.gridx = 0;
    gbc.gridy = 0;
    gbc.gridwidth = 3;
    panel.add(new JScrollPane(table), gbc);
    gbc.gridwidth = 1;

    final JComboBox jcbServers = new JComboBox();

    final RemoteTask<SortedSet<ServerInfo>> t =
      new RemoteTask<SortedSet<ServerInfo>>() {
        @Override
        protected void performTask() throws Exception {
          SortedSet<ServerInfo> serversInfos =
            ClientServerManager.getInstance().getService(
              ServerServiceInterface.class).getServersInfos();
          setResult(serversInfos);
        }
      };

    if (t.execute(this, "Inicializando", "Consultando servidor..")) {
      SortedSet<ServerInfo> servers = t.getResult();
      for (ServerInfo ts : servers) {
        if (!ts.isSuspended()) {
          jcbServers.addItem(ts.getURI());
        }
      }
    }

    final JButton btnConnectServer = new JButton("Conectar");

    if (jcbServers.getItemCount() == 0) {
      jcbServers.addItem("Nenhum servidor foi encontrado.");
      jcbServers.setEnabled(false);
      btnConnectServer.setEnabled(false);
    }
    else {
      btnConnectServer.setEnabled(true);
    }

    btnConnectServer.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        boolean usePassword =
          ServerStatusFrame.this.usePasswordCheckBox.isSelected();
        ServerURI sURI = (ServerURI) jcbServers.getSelectedItem();
        boolean copySession =
          ServerStatusFrame.this.cloneServerSessionCheckBox.isSelected();
        if (!usePassword) {
          loginByReference(sURI, copySession);
        }
        else {
          loginWithUserAndPassword(sURI);
        }

        serverTableModel.loadData();
      }
    });

    jtfUser = new JTextField();
    jtfPassword = new JPasswordField();
    usePasswordCheckBox = new JCheckBox("Usar usurio e senha");
    usePasswordCheckBox.setSelected(true);
    cloneServerSessionCheckBox =
      new JCheckBox(
        "Copiar sesso do servidor default (somente p/ login por referncia)");
    usePasswordCheckBox.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        boolean b = usePasswordCheckBox.isSelected();
        jtfUser.setEnabled(b);
        jtfPassword.setEnabled(b);
      }
    });

    gbc.gridx = 0;
    gbc.gridy = 1;
    gbc.gridwidth = 3;
    panel.add(cloneServerSessionCheckBox, gbc);
    gbc.gridwidth = 1;

    gbc.gridx = 0;
    gbc.gridy = 2;
    panel.add(usePasswordCheckBox, gbc);

    gbc.gridx = 1;
    gbc.gridy = 2;
    panel.add(jcbServers, gbc);

    gbc.gridx = 2;
    gbc.gridy = 2;
    panel.add(btnConnectServer, gbc);

    gbc.gridx = 0;
    gbc.gridy = 3;
    panel.add(new JLabel("Usurio:"), gbc);

    gbc.gridx = 1;
    gbc.gridy = 3;
    panel.add(jtfUser, gbc);

    gbc.gridx = 0;
    gbc.gridy = 4;
    panel.add(new JLabel("Senha:"), gbc);

    gbc.gridx = 1;
    gbc.gridy = 4;
    panel.add(jtfPassword, gbc);

    return panel;
  }

  /**
   * Cria o painel de botes.
   * 
   * @return painel
   */
  protected JPanel buildButtonsPanel() {
    final JPanel panel = new JPanel(new FlowLayout());

    final JButton closeButton = new JButton("Fechar");
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        frame.setVisible(false);
      }
    });

    panel.add(closeButton);
    return panel;
  }

  /**
   * Construo de GUI.
   */
  private void buildGui() {
    this.getContentPane().add(this.buildMainPanel(), BorderLayout.CENTER);
    this.getContentPane().add(this.buildButtonsPanel(), BorderLayout.SOUTH);
    this.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
    this.pack();
    this.setResizable(false);
  }

  /**
   * Listener para atualizar a tabela com o estado dos servidores
   * 
   * @author Tecgraf/PUC-Rio
   * 
   */
  private class ServerStatusListener implements MonitoredServerListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void notifyConnectionLost(ServerURI serverURI) {
      serverTableModel.loadData();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void notifyConnectionReestablished(ServerURI serverURI) {
      serverTableModel.loadData();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void notifyLoggedIn(ServerURI serverURI) {
      serverTableModel.loadData();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void notifyLoggedOut(ServerURI serverURI) {
      serverTableModel.loadData();
    }
  }

  /**
   * @param sURI
   */
  private final void login(final ServerURI sURI) {
    RemoteTask<Boolean> t = new RemoteTask<Boolean>(sURI) {
      @Override
      protected void handleError(Exception error) {
        StandardDialogs.showErrorDialog(ServerStatusFrame.this, "Erro", error
          .getMessage());
      }

      @Override
      protected void performTask() throws Exception {
        setResult(ClientServerManager.getInstance().login(sURI));
      }
    };

    if (t.execute(ServerStatusFrame.this, "Login", "Contactando servidor..")) {
      if (!t.getResult()) {
        StandardDialogs.showErrorDialog(ServerStatusFrame.this, "Erro",
          "Login ou senha invlidos.");
      }
    }
  }

  /**
   * Executa o login por referncia
   * 
   * @param copySession true copia sesso do servidor de referncia, false
   *        apenas cria a sesso nova
   * @param sURI
   */
  private final void loginByReference(final ServerURI sURI,
    final boolean copySession) {
    final String errTitle = "Erro";
    final String errMessage = "Login ou senha invlidos";
    RemoteTask<Boolean> t = new RemoteTask<Boolean>() {
      @Override
      protected void handleError(Exception error) {
        StandardDialogs.showErrorDialog(ServerStatusFrame.this, errTitle, error
          .getMessage());
      }

      @Override
      protected void performTask() throws Exception {
        setResult(ClientServerManager.getInstance().loginByReference(sURI,
          copySession));
      }
    };

    if (t.execute(ServerStatusFrame.this, "Login", "Contactando servidor..")) {
      if (!t.getResult()) {
        StandardDialogs.showErrorDialog(ServerStatusFrame.this, errTitle,
          errMessage);
      }
    }
  }

  /**
   * Executa login usando usurio e senha
   * 
   * @param sURI
   */
  private final void loginWithUserAndPassword(final ServerURI sURI) {

    String user = jtfUser.getText();
    String pass = new String(jtfPassword.getPassword());

    if (user == null || user.length() == 0 || pass.length() == 0) {
      StandardDialogs.showErrorDialog(this, "Erro",
        "Usurio e senha so obrigatrios");
    }
    else {
      final LoginInfo loginInfo =
        new LoginInfo(user, pass, Locale.getDefault());

      RemoteTask<Boolean> t = new RemoteTask<Boolean>() {
        @Override
        protected void handleError(Exception error) {
          StandardDialogs.showErrorDialog(ServerStatusFrame.this, "Erro", error
            .getMessage());
        }

        @Override
        protected void performTask() throws Exception {
          setResult(ClientServerManager.getInstance().loginWithUserPassword(
            sURI, loginInfo));
        }
      };

      if (t.execute(this, "Login", "Contactando servidor..")) {
        if (!t.getResult()) {
          StandardDialogs.showErrorDialog(ServerStatusFrame.this, "Erro",
            "Login ou senha invlidos");
        }
        else {
          serverTableModel.loadData();
        }
      }
    }

  }

  /**
   * Executa o logout sem remover da monitorao
   * 
   * @param sURI
   */
  private final void logout(final ServerURI sURI) {
    new RemoteTask<Void>(sURI) {
      @Override
      protected void performTask() {
        ClientServerManager.getInstance().logout(sURI, false);
      }
    }.execute(ServerStatusFrame.this, "Logout", "Contactando servidor..");
  }

}
