/*
 * IncludePasswordDialog.java
 * 
 * $Author: oikawa $ $Revision: 150777 $ - $Date: 2006-08-15 11:01:24 -0300
 * (Tue, 15 Aug 2006) $
 */
package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.security.DigestException;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.MDigest;
import csbase.logic.User;

/**
 * A classe <code>IncludePasswordDialog</code>  responsvel por cadastrar uma
 * nova senha atravs de uma janela de dilogo, modal e centrada sobre a janela
 * que a invocar. Caso o usurio no seja um administrador, ele dever entrar
 * com a senha antiga; caso contrrio, ele dever entrar apenas com a nova
 * senha.
 * 
 * @author $Author: oikawa $
 * @version $Revision: 150777 $
 */
public class IncludePasswordDialog {
  /** Janela de exibio do dilogo de senha */
  private DesktopComponentDialog window;
  /** Ttulo da janela */
  private String windowLabel;
  /** Campo para senha antiga do usurio (caso este no seja administrador) */
  private JTextField oldPasswordField;
  /** Campo para nova senha */
  private JTextField newPasswordField;
  /** Campo para confirmao da nova senha */
  private JTextField newPasswordConfirmField;
  /** Nova senha */
  private String newPassword;
  /** Indica se o usurio  novo ou est sendo modificado */
  private boolean isNew;
  /** Indica se quem est usando o sistema  o administrador */
  private boolean loggedUserIsAdmin;
  /** Usurio cuja senha ser criada/modificada */
  private User user;

  /**
   * Obtem a nova senha a ser cadastrada ou null caso o usurio tenha desistido
   * dessa operao.
   * 
   * @return a nova senha ou null caso no queira cadastrar ou modificar a senha
   *         do usurio
   */
  public String getNewPassword() {
    return newPassword;
  }

  /**
   * Cria e mostra a janela de cadastramento de senha de usurio.
   * 
   * @param owner Janela que invocou o dilogo de cadastramento de senha
   */
  private void display(DesktopComponentDialog owner) {
    JPanel pane = new JPanel(new BorderLayout());
    pane.add(makeTextPanel(), BorderLayout.NORTH);
    pane.add(makeButtonPanel(), BorderLayout.SOUTH);
    window = new DesktopComponentDialog(owner, windowLabel);
    window.getContentPane().add(pane);
    window.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        window.dispose();
      }
    });
    window.pack();
    window.center(owner);
    window.setVisible(true);
  }

  /**
   * Constri o painel de cadastramento de senha de usurio.
   * 
   * @return o painel principal da janela
   */
  private JPanel makeTextPanel() {
    // Fields
    oldPasswordField = new JPasswordField(12);
    newPasswordField = new JPasswordField(12);
    newPasswordConfirmField = new JPasswordField(12);
    // Monta os painis
    JPanel mainPanel = new JPanel();
    JPanel labelsPanel = new JPanel(new GridLayout(0, 1));
    JPanel fieldsPanel = new JPanel(new GridLayout(0, 1));
    mainPanel.add(labelsPanel);
    mainPanel.add(fieldsPanel);
    if (isNew) { // Senha para novo usurio
      labelsPanel.add(new JLabel(LNG.get("IAS_USER_NEW_PASSWORD")));
      fieldsPanel.add(newPasswordField);
      labelsPanel.add(new JLabel(LNG.get("IAS_USER_PASSWORD_CONFIRMATION")));
      fieldsPanel.add(newPasswordConfirmField);
    }
    else { // Alterao de senha de usurio existente
      if (loggedUserIsAdmin) {
        labelsPanel.add(new JLabel(LNG.get("IAS_USER_NEW_PASSWORD")));
        fieldsPanel.add(newPasswordField);
        labelsPanel.add(new JLabel(LNG
          .get("IAS_USER_NEW_PASSWORD_CONFIRMATION")));
        fieldsPanel.add(newPasswordConfirmField);
      }
      else {// No sendo administrador, pede-se a senha antiga
        labelsPanel.add(new JLabel(LNG.get("IAS_USER_CURRENT_PASSWORD")));
        fieldsPanel.add(oldPasswordField);
        labelsPanel.add(new JLabel(LNG.get("IAS_USER_NEW_PASSWORD")));
        fieldsPanel.add(newPasswordField);
        labelsPanel.add(new JLabel(LNG
          .get("IAS_USER_NEW_PASSWORD_CONFIRMATION")));
        fieldsPanel.add(newPasswordConfirmField);
      }
    }
    return mainPanel;
  }

  /**
   * Constri o painel com os botes da janela de cadastramento de senha de
   * usurios. As aes previstas so para confirmar a operao solicitada,
   * limpar os campos ou desistir da operao.
   * 
   * @return o painel de botes
   */
  private JPanel makeButtonPanel() {
    JButton okB = new JButton(LNG.get("IAS_CONFIRM"));
    okB.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        try {
          if (validateFields()) {
            window.dispose();
          }
          else {
            StandardErrorDialogs.showErrorDialog(window, windowLabel,
              LNG.get("IAS_USER_PASSWORD_ERROR"));
          }
        }
        catch (DigestException e) {
          StandardErrorDialogs.showErrorDialog(window, windowLabel,
            LNG.get("IAS_USER_DIGEST_PASSWORD_ERROR"), e);
        }
      }
    });
    JButton clearB = new JButton(LNG.get("IAS_CLEAR"));
    clearB.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        clearFields();
      }
    });
    JButton cancelB = new JButton(LNG.get("IAS_CANCEL"));
    cancelB.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        window.dispose();
      }
    });

    JPanel buttonP = new JPanel();
    buttonP.add(okB);
    buttonP.add(clearB);
    buttonP.add(cancelB);
    GUIUtils.matchPreferredSizes(new JComponent[] { okB, clearB, cancelB });

    return buttonP;
  }

  /**
   * Verifica se a senha  igual a confirmao da senha. A senha e a confirmao
   * devem estar preenchidas.
   * 
   * @return true se a senha e a confirmao so iguais ou false, caso contrrio
   */
  protected boolean newPasswordIsValid() {
    String password = newPasswordField.getText();
    String confirmation = newPasswordConfirmField.getText();
    return !password.equals("") && !confirmation.equals("")
      && password.equals(confirmation);
  }

  /**
   * Verifica se a senha atual do usurio que est fazendo a modificao de sua
   * prpria senha  vlida.
   * 
   * @return true se a senha atual  vlida ou false, caso contrrio
   * 
   * @throws DigestException Caso ocorra algum problema ao codificar a senha
   */
  protected boolean checkOldPassword() throws DigestException {
    String old = oldPasswordField.getText();
    String oldDigest = MDigest.getDigest(old);
    return user.getPasswordDigest().equals(oldDigest);
  }

  /**
   * Valida os campos de senha e confirmao de senha.
   * 
   * @return true caso os dados estejam corretos
   * 
   * @throws DigestException Caso ocorra algum problema ao codificar a senha
   */
  protected boolean validateFields() throws DigestException {
    // Se for inclusao de senha de usurio novo ou se quem est modificando  o
    // prprio administrador, no precisa da senha antiga.
    if (isNew || loggedUserIsAdmin) {
      if (newPasswordIsValid()) {
        newPassword = newPasswordField.getText();
        return true;
      }
      return false;
    }
    // Se for modificao de senha de usurio e quem est fazendo a modificao
    //  o prprio usurio, ento precisa da senha antiga.
    if (checkOldPassword() && newPasswordIsValid()) {
      newPassword = newPasswordField.getText();
      return true;
    }
    return false;
  }

  /**
   * Limpa os campos de nova senha e confirmao da nova senha.
   */
  protected void clearFields() {
    if (!isNew && (!loggedUserIsAdmin)) {
      oldPasswordField.setText("");
      oldPasswordField.requestFocus();
    }
    else {
      newPasswordField.requestFocus();
    }
    newPasswordField.setText("");
    newPasswordConfirmField.setText("");
  }

  /**
   * Cria uma janela para cadastramento de senha de um novo usurio.
   * 
   * @param owner janela pai desse dilogo
   */
  public IncludePasswordDialog(DesktopComponentDialog owner) {
    this.user = null;
    this.isNew = true;
    this.loggedUserIsAdmin = User.getLoggedUser().isAdmin();
    windowLabel = LNG.get("IAS_USER_PASSWORD_REGISTRATION_TITLE");
    display(owner);
  }

  /**
   * Cria uma janela para modificao de senha de usurio.
   * 
   * @param owner janela pai desse dilogo
   * @param user o usurio cuja senha vai ser cadastrada
   */
  public IncludePasswordDialog(DesktopComponentDialog owner, User user) {
    this.user = user;
    this.isNew = false;
    this.loggedUserIsAdmin = User.getLoggedUser().isAdmin();
    windowLabel =
      ClientUtilities.addSystemNameToTitle(LNG
        .get("ias.user.password_update_title"));
    display(owner);
  }
}
