package csbase.client.openbus;

import java.util.List;

import org.omg.CORBA.Any;
import org.omg.CORBA.BAD_OPERATION;

import scs.core.ComponentContext;
import tecgraf.openbus.session_service.v1_05.SessionEvent;
import tecgraf.openbus.session_service.v1_05.SessionEventSinkPOA;

/**
 * Recebedor de eventos do cliente CSBASE.
 * 
 * @author Tecgraf/PUC-Rio
 */
final public class OpenBusEventSink extends SessionEventSinkPOA {

  /**
   * Contexto.
   */
  final private ComponentContext context;

  /**
   * Consulta se valor {@code Any}  do tipo {@code String}.
   * 
   * @param value valor
   * @return indicativo
   */
  private boolean isValueString(Any value) {
    try {
      value.extract_string();
      return true;
    }
    catch (BAD_OPERATION e) {
      return false;
    }
  }

  /**
   * Consulta se valor {@code Any}  do tipo {@code Boolean}.
   * 
   * @param value valor
   * @return indicativo
   */
  private boolean isValueBoolean(Any value) {
    try {
      value.extract_boolean();
      return true;
    }
    catch (BAD_OPERATION e) {
      return false;
    }
  }

  /**
   * {@inheritDoc} TODO mjulia memberIdentifier: por enquanto, estou ignorando
   * esse novo parmetro.
   */
  @Override
  final public void disconnect(String memberIdentifier) {
  }

  /**
   * Faz loop dos receptores de eventos do cliente.
   * 
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param receivers receptores.
   * @param type tipo.
   * @return indicativo de tratamento.
   */
  private boolean loopReceiveOnNoValue(
    List<OpenBusEventReceiverInterface> receivers, String sourceId,
    String destId, final String type) {
    for (OpenBusEventReceiverInterface receiver : receivers) {
      final boolean treated =
        receiver.receiveEventWithNoValue(sourceId, destId, type);
      receiver.logReceivedEvent(treated, sourceId, destId, type, null);
      if (treated) {
        return true;
      }
    }
    return false;
  }

  /**
   * Faz loop dos receptores de eventos do cliente.
   * 
   * @param receivers receptores
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param type tipo
   * @param stringValue o valor
   * @return indicativo de tratamento.
   */
  private boolean loopReceiveOnStringValue(
    List<OpenBusEventReceiverInterface> receivers, String sourceId,
    String destId, final String type, final String stringValue) {
    for (OpenBusEventReceiverInterface receiver : receivers) {
      final boolean treated =
        receiver.receiveEventWithStringValue(sourceId, destId, type,
          stringValue);
      receiver.logReceivedEvent(treated, sourceId, destId, type, stringValue);
      if (treated) {
        return true;
      }
    }
    return false;
  }

  /**
   * Faz loop dos receptores de eventos do cliente.
   * 
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param receivers receptores.
   * @param type tipo.
   * @param booleanValue o valor.
   * @return indicativo de tratamento.
   */
  private boolean loopReceiveOnBooleanValue(
    List<OpenBusEventReceiverInterface> receivers, String sourceId,
    String destId, final String type, final boolean booleanValue) {
    for (OpenBusEventReceiverInterface receiver : receivers) {
      final boolean treated =
        receiver.receiveEventWithBooleanValue(sourceId, destId, type,
          booleanValue);
      receiver.logReceivedEvent(treated, sourceId, destId, type, booleanValue);
      if (treated) {
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc} TODO mjulia memberIdentifier: por enquanto, estou ignorando
   * esse novo parmetro.
   */

  @Override
  final public void push(String memberIdentifier, SessionEvent event) {
    final OpenBusAccessPoint oap = OpenBusAccessPoint.getInstance();
    List<OpenBusEventReceiverInterface> receivers =
      oap.getSession().getEventReceivers();

    final String rawType = event.type.trim();
    final String[] pathData = oap.getSession().extractPathTypeData(rawType);
    final String sourceId = pathData[0];
    final String destId = pathData[1];
    final String type = pathData[2];
    final Any anyValue = event.value;
    final Object value;

    final boolean treated;
    if (anyValue == null) {
      treated = false;
      value = "null any!!!";
    }
    else if (isValueString(anyValue)) {
      final String stringValue = anyValue.extract_string().trim();
      treated =
        loopReceiveOnStringValue(receivers, sourceId, destId, type, stringValue);
      value = stringValue;
    }
    else if (isValueBoolean(anyValue)) {
      final boolean booleanValue = anyValue.extract_boolean();
      treated =
        loopReceiveOnBooleanValue(receivers, sourceId, destId, type,
          booleanValue);
      value = booleanValue;
    }
    else {
      treated = loopReceiveOnNoValue(receivers, sourceId, destId, type);
      value = null;
    }
    oap.getSession()
      .signalEventReceived(treated, sourceId, destId, type, value);
  }

  /**
   * Construtor.
   */
  public OpenBusEventSink() {
    this.context = null;
  }

  /**
   * Construtor utilizado pelo mecanismo de criao de componentes do SCS.
   * 
   * @param context O contexto do componente ao qual esta faceta pertence.
   */
  public OpenBusEventSink(ComponentContext context) {
    this.context = context;
  }

  /**
   * Consulta o atributo (ver documentao de {@link #context}).
   * 
   * @return o atributo.
   */
  final public ComponentContext getContext() {
    return context;
  }
}
