package csbase.client.project;

import java.awt.BorderLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.io.File;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import csbase.client.ClientLocalFile;
import csbase.client.ClientSmartFile;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.client.project.ProjectFileTypeComboBox.Mode;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.ProjectFileType;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;

/**
 * A classe <code>DirectoryCreationDialog</code> mostra um dilogo para criao
 * de um sub-diretrio a partir de um <code>ClientFile</code>. O usurio pode
 * acionar o boto de criao, que confirma a operao, ou pode acionar o boto
 * de cancelamento, que desiste da operao. O boto de criao somente 
 * habilitado quando o usurio tiver informado um nome do diretrio. Ao
 * confirmar as operao, o servio  executado e o usurio  notificado de
 * qualquer problema que tenha ocorrido.
 *
 * @author Tecgraf/PUC-Rio
 */
public class DirectoryCreationDialog {
  /**
   * A janela de visualizao de informaes do projeto.
   */
  protected DesktopComponentDialog dialog;

  /**
   * A janela a partir da qual esse dilogo foi solicitado.
   */
  protected Window owner;

  /**
   * O campo de nome do diretrio
   */
  protected JTextField dirNameText;

  /**
   * O botao default.
   */
  protected JButton createButton;

  /**
   * O diretrio pai, onde o sub-diretrio est sendo criado.
   */
  protected ClientFile directory;

  private ProjectFileTypeComboBox dirTypeComboBox;

  /**
   * Mostra um dilogo que pede o nome do diretrio a ser criado.
   *
   * @param owner A janela a partir da qual esse dilogo foi solicitado.
   * @param directory O diretrio que vai ter o novo sub-diretrio.
   */
  public static void show(Window owner, ClientFile directory) {
    new DirectoryCreationDialog(owner, directory);
  }

  /**
   * Cria o dilogo modal, a partir de uma janela principal.
   *
   * @return O dilogo recm-criado.
   */
  protected DesktopComponentDialog makeDialog() {
    // Cria o painel principal da janela.
    JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.add(makeNamePanel(), BorderLayout.CENTER);
    mainPanel.add(makeButtonPanel(), BorderLayout.SOUTH);

    // Cria um dilogo modal
    final String title = getTitle();
    DesktopComponentDialog dlg = new DesktopComponentDialog(owner, title);
    dlg.getContentPane().add(mainPanel);
    dlg.pack();
    dlg.center(owner);
    dlg.getRootPane().setDefaultButton(createButton);
    dlg.setResizable(false);
    return dlg;
  }

  /**
   * Cria um painel com o campo de nome do diretrio. O campo de nome  um campo
   * de texto de apenas uma nica linha e possui um listener para que, quando um
   * nome for digitado, o boto principal seja habilitado.
   *
   * @return o painel recm-criado.
   */
  private JPanel makeNamePanel() {
    final JLabel dirNameLabel = new JLabel(LNG.get("PRJ_NAME"));
    dirNameText = new JTextField(20);
    final JLabel dirTypeLabel = new JLabel(LNG.get("PRJ_FILE_TYPE"));
    dirTypeComboBox = new ProjectFileTypeComboBox(Mode.DIRECTORY_ONLY, false);
    dirTypeComboBox.selectTypeCode(ProjectFileType.DIRECTORY_TYPE);
    JComponent[][] components = { { dirNameLabel, dirNameText }, { dirTypeLabel,
        dirTypeComboBox }, };

    dirNameText.addKeyListener(new KeyAdapter() {
      @Override
      public void keyReleased(KeyEvent e) {
        String name = dirNameText.getText().trim();
        if (name.isEmpty()) {
          createButton.setEnabled(false);
        }
        else {
          createButton.setEnabled(true);
        }
      }
    });
    return GUIUtils.createBasicGridPanel(components);
  }

  /**
   * Cria um painel com os botes que vo ser usados no dilogo de criao de
   * diretrio. Na criao de um diretrio, o usurio pode acionar o boto de
   * criao, que confirma a operao, ou pode acionar o boto de cancelamento,
   * que desiste da operao. O boto de criao somente  habilitado quando o
   * usurio tiver informado um nome do diretrio.
   *
   * @return O painel recm-criado.
   */
  protected JPanel makeButtonPanel() {
    JPanel panel = new JPanel();

    // Boto de criao de diretrio.
    createButton = new JButton(LNG.get("PRJ_CREATION"));
    createButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        handleAction();
      }
    });
    createButton.setEnabled(false);

    // Boto de cancelamento da operao de criao.
    final JButton cancelButton = new JButton(LNG.get("PRJ_CANCEL"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        dialog.close();
      }
    });

    final JComponent[] buttons = new JComponent[] { createButton,
        cancelButton };
    GUIUtils.matchPreferredSizes(buttons);
    panel.add(createButton);
    panel.add(cancelButton);
    return panel;
  }

  /**
   * Informa o ttulo que vai ser utilizado na janela de criao de projeto.
   *
   * @return O ttulo da janela.
   */
  protected String getTitle() {
    return ClientUtilities.addSystemNameToTitle(LNG.get(
      "project.directory_creation_title"));
  }

  /**
   * Executa a ao de criao de um diretrio. O servio  chamado no servidor.
   * Caso algum problema ocorra, o usurio  informado com uma mensagem em uma
   * outra janela de dilogo.
   */
  private void handleAction() {
    final String name = dirNameText.getText().trim();
    final String typeCode = dirTypeComboBox.getSelectedTypeCode();
    final String title = getTitle();
    final String prefix = getClass().getSimpleName();

    if (!ClientUtilities.isValidFileName(name)) {
      String err = LNG.get(prefix + ".dir.name.error");
      StandardErrorDialogs.showErrorDialog(owner, title, err);
      return;
    }

    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        switch (directory.getClientFileType()) {
          case LOCAL:
            ClientLocalFile parent = (ClientLocalFile) directory;
            File newDir = new File(parent.getStringPath(), name);
            if (!newDir.mkdir()) {
              String err = LNG.get(prefix + ".create.dir.error", new Object[] {
                  name });
              throw new ClientException(err);
            }
            break;
          case REMOTE:
            ((ClientProjectFile) directory).createFile(name, typeCode);
            break;
          case SMART:
            ((ClientSmartFile) directory).getClientProjectFile().createFile(
              name, typeCode);
            break;
          default:
            throw new IllegalArgumentException(
              "File type does not support directory creation");
        }
      }
    };

    String[] args = new String[] { name };
    String taskMessage = LNG.get(prefix + ".create.dir.task", args);
    if (task.execute(dialog, title, taskMessage)) {
      dialog.close();
    }
  }

  /**
   * Constri o dilogo e mostra para o usurio.
   *
   * @param owner A janela a partir da qual esse dilogo foi solicitado.
   * @param directory O diretrio que vai ter o novo sub-diretrio.
   */
  private DirectoryCreationDialog(Window owner, ClientFile directory) {
    this.owner = owner;
    this.directory = directory;
    dialog = makeDialog();
    dialog.setVisible(true);
  }
}
