/*
 * $Id: CommonProjectSelectDialog.java 95877 2009-09-04 21:57:56Z jnlopes $
 */
package csbase.client.project;

import java.awt.DefaultKeyboardFocusManager;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.KeyboardFocusManager;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.ButtonGroup;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.RowFilter;
import javax.swing.SortOrder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.text.Document;

import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.project.ProjectTableModel.RowData;
import csbase.client.project.ProjectTableModel.VisibleColumns;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.logic.UserProjectInfo;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Dilogo modal para abertura de um projeto do usurio logado, um projeto de
 * outros usurios (habilitado apenas para o admin), ou um projeto compartilhado
 * por outro usurio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectOpenDialog extends DesktopComponentDialog {

  /** Boto default da janela. */
  protected JButton defaultButton;

  /** Projeto selecionado. */
  protected CommonClientProject project;

  /** RadioButton que lista todos os projetos do prprio usurio. */
  protected JRadioButton choiceOpenUser;

  /** RadioButton que lista todos os projetos de todos os usurios. */
  protected JRadioButton choiceOpenShared;

  /**
   * RadioButton que lista todos os projetos compartilhados de outros usurios.
   */
  protected JRadioButton choiceOpenAny;

  /** Texto usado como filtro para a tabela. */
  private JTextField filterText;

  /** Boto para limpar o campo de filtro. */
  private JButton filterButton;

  /** Tabela de projetos compartilhados. */
  private ProjectTableSortableTable projectsTable;

  /**
   * Objeto que encapsula o projeto selecionado em uma linha da tabela.
   * 
   * @see ProjectTableModel.RowData
   */
  private RowData selectedProject;

  /** Lista de projetos compartilhados com o usurio logado. */
  private List<RowData> sharedProjectsList;

  /** Lista de projetos cujo usurio logado seja o dono. */
  private List<RowData> currentUserProjectsList;

  /**
   * Lista de todos os projetos (compartilhados ou no) de todos os usurios.
   */
  private List<RowData> allUsersProjectsList;

  /** Ao de abertura do projeto selecionado. */
  private AbstractAction openAction;

  /** Ao de cancelamento da tela. */
  private AbstractAction cancelAction;

  /** Ao de refresh a lista de projetos com os projetos read-only. */
  private AbstractAction refreshROProjAction;

  /**
   * Dispatcher para redirecionar eventos de teclado aplicvel apenas  este
   * dilogo.
   */
  private ProjectKeyboardFocusManager projectKeyboardFocusManager;

  /**
   * Contrutor.
   * 
   * @param owner janela a partir da qual esse dilogo foi chamado.
   */
  public ProjectOpenDialog(Window owner) {
    super(owner, LNG.get("ProjectOpenDialog.title.open"));
    makeDialog();
    setVisible(true);
  }

  /** Cria os componentes deste dilogo. */
  private void makeDialog() {
    initializeActions();

    /*
     * Cria o painel principal da janela.
     */
    JPanel mainPanel = new JPanel(new GridBagLayout());

    makeOpenOptionsPanel(mainPanel);
    createProjectPanel(mainPanel);

    GBC gbc = new GBC(0, 6).northwest().bottom(5).horizontal();
    mainPanel.add(makeButtonPanel(), gbc);

    KeyboardFocusManager keyboardFocusManager =
      KeyboardFocusManager.getCurrentKeyboardFocusManager();

    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        close();
      }
    });

    projectKeyboardFocusManager =
      new ProjectKeyboardFocusManager(this, projectsTable, filterText);

    getContentPane().add(mainPanel);
    setPreferredSize(new Dimension(450, 400));
    pack();
    center(getOwner());
    JRootPane dialogRootPane = getRootPane();

    //Definio do boto "Abrir" como padro deste dilogo
    dialogRootPane.setDefaultButton(defaultButton);

    InputMap iMap =
      dialogRootPane.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
    ActionMap aMap = dialogRootPane.getActionMap();

    //Associa a ao de cancelamento da tela  tecla ESC.
    KeyStroke escKeystroke = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);
    String actionMapKey = escKeystroke.toString();
    iMap.put(escKeystroke, actionMapKey);
    aMap.put(actionMapKey, cancelAction);

    //Adiciona o dispatcher de eventos de teclado ao KeyboardFocusManager.
    keyboardFocusManager.addKeyEventDispatcher(projectKeyboardFocusManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() {
    /*
     * Antes de fechar a janela, remover o dispatcher.
     */
    KeyboardFocusManager.getCurrentKeyboardFocusManager()
      .removeKeyEventDispatcher(projectKeyboardFocusManager);
    super.close();
  }

  /**
   * Inicializao das aes de abertura do projeto selecionado e de
   * cancelamento da tela.
   */
  private void initializeActions() {
    // Ao de abertura do projeto selecionado.
    openAction = new AbstractAction(LNG.get("ProjectOpenDialog.open")) {
      @Override
      public void actionPerformed(ActionEvent e) {
        handleOpenAction();
      }
    };

    // Ao de cancelamento da tela.
    cancelAction =
      new AbstractAction(LNG.get("CommonProjectSelectDialog.cancel")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          close();
        }
      };

    // Ao de refresh a lista de projetos read-only
    refreshROProjAction =
      new AbstractAction(LNG.get("ProjectOpenDialog.refresh.button")) {

        @Override
        public void actionPerformed(ActionEvent arg0) {
          handleRefreshAction();
        }
      };
  }

  /**
   * Adiciona opes no painel principal.
   * 
   * @param mainPanel painel principal.
   */
  protected void makeOpenOptionsPanel(JPanel mainPanel) {

    // Projetos do Usurio.
    choiceOpenUser = new JRadioButton(LNG.get("ProjectOpenDialog.option.user"));
    choiceOpenUser.setEnabled(userHasHisOwnProjects());

    GBC gbc = new GBC(0, 0).left(11).west().top(10).none();
    mainPanel.add(choiceOpenUser, gbc);

    // Projetos Compartilhados por Outros Usurios.
    choiceOpenShared =
      new JRadioButton(LNG.get("ProjectOpenDialog.option.others"));
    choiceOpenShared.setEnabled(userHasSharedProjects());

    gbc = new GBC(0, 1).left(11).west().none();
    mainPanel.add(choiceOpenShared, gbc);

    // Projetos de Todos os Usurios.
    if (User.getLoggedUser().isAdmin()) {
      choiceOpenAny = new JRadioButton(LNG.get("ProjectOpenDialog.option.any"));

      gbc = new GBC(0, 2).left(11).west().none();
      mainPanel.add(choiceOpenAny, gbc);
    }

    configureRadioButtons();
  }

  /**
   * Consulta se o usurio possui seus prprios projetos.
   * 
   * @return true se o usurio possui seus prprios projetos, false caso
   *         contrrio.
   */
  private boolean userHasHisOwnProjects() {
    /*
     * podemos usar o getter porque fazemos cache da lista (ela s  recuperada
     * do servidor uma nica vez)
     */
    List<RowData> ownProjects = getCurrentUserProjectsList(true);
    return ownProjects != null && !ownProjects.isEmpty();
  }

  /**
   * Consulta se existem projetos compartilhados com o usurio atual.
   * 
   * @return true se existem projetos compartilhados com o usurio corrente,
   *         false caso contrrio.
   */
  private boolean userHasSharedProjects() {
    /*
     * podemos usar o getter porque fazemos cache da lista (ela s  recuperada
     * do servidor uma nica vez)
     */
    List<RowData> sharedProjects = getSharedProjects(true);
    return sharedProjects != null && !sharedProjects.isEmpty();
  }

  /** Cria e configura os radio-buttons. */
  private void configureRadioButtons() {
    /*
     * definimos o estado inicial dos radio-buttons. Como ainda no adicionamos
     * os listeners, a tabela no vai ser atualizada
     */
    if (userHasHisOwnProjects()) {
      choiceOpenUser.setSelected(true);
    }
    else if (userHasSharedProjects()) {
      choiceOpenShared.setSelected(true);
    }
    else if (choiceOpenAny != null) {
      choiceOpenAny.setSelected(true);
    }

    ButtonGroup mainRadiosGroup = new ButtonGroup();

    //projetos do prprio usurio
    choiceOpenUser.addItemListener(new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent e) {
        if (e.getStateChange() == ItemEvent.SELECTED) {

          // Preencher projectList com todos os projetos do prprio usurio.
          List<RowData> projectList = getCurrentUserProjectsList(true);
          updateProjectTable(projectList, VisibleColumns.NAME);
        }
      }
    });
    mainRadiosGroup.add(choiceOpenUser);

    //projetos compartilhados com o usurio
    choiceOpenShared.addItemListener(new ItemListener() {

      @Override
      public void itemStateChanged(ItemEvent e) {
        /*
         * Preencher projectList com todos os projetos de outros usurios que
         * esto compartilhados.
         */
        if (e.getStateChange() == ItemEvent.SELECTED) {
          updateProjectTable(getSharedProjects(false),
            VisibleColumns.OWNER_NAME_PERMISSION);
        }
      }
    });
    mainRadiosGroup.add(choiceOpenShared);

    //projetos de todos os usurios (apenas para o admin)
    if (choiceOpenAny != null) {
      choiceOpenAny.addItemListener(new ItemListener() {
        @Override
        public void itemStateChanged(ItemEvent e) {
          /*
           * Preencher projectList com todos os projetos de todos os usurios.
           */
          if (e.getStateChange() == ItemEvent.SELECTED) {
            List<RowData> projectList = getAllUsersProjectsList(true);
            updateProjectTable(projectList, VisibleColumns.OWNER_NAME);
          }
        }
      });
      mainRadiosGroup.add(choiceOpenAny);
    }
  }

  /**
   * Cria painel com os projetos e adiciona no painel principal.
   * 
   * @param mainPanel painel principal.
   */
  private void createProjectPanel(JPanel mainPanel) {

    // inicializamos a tabela de acordo com a opo selecionada
    if (choiceOpenUser.isSelected()) {
      projectsTable = new ProjectTableSortableTable(
        getCurrentUserProjectsList(true), VisibleColumns.NAME);
    }
    else if (choiceOpenShared.isSelected()) {
      projectsTable = new ProjectTableSortableTable(getSharedProjects(true),
        VisibleColumns.OWNER_NAME_PERMISSION);
    }
    else if (choiceOpenAny != null && choiceOpenAny.isSelected()) {
      projectsTable = new ProjectTableSortableTable(
        getAllUsersProjectsList(true), VisibleColumns.OWNER_NAME);
    }
    else {
      throw new RuntimeException("opo desconhecida");
    }

    final ListSelectionModel listSelectionModel =
      projectsTable.getSelectionModel();

    listSelectionModel.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        if (e.getValueIsAdjusting()) {
          return;
        }
        selectedProject = projectsTable.getSelectedRowData();
      }
    });

    MouseListener mouseListener = new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
          handleOpenAction();
        }
      }

    };
    projectsTable.addMouseListener(mouseListener);

    // remapeamos a ao de ENTER da tabela para abrir o projeto selecionado
    InputMap im =
      projectsTable.getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
    ActionMap actionMap = projectsTable.getActionMap();
    im.put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
      "openProjectActionName");
    actionMap.put("openProjectActionName", openAction);

    projectsTable.sort(0, SortOrder.ASCENDING);
    projectsTable.setSelectedFirstRow();

    JPanel filterPanel = new JPanel(new GridBagLayout());

    JScrollPane scrollPane = new JScrollPane(projectsTable);

    GBC gbc = new GBC(0, 3).insets(10, 15, 10, 15).both();
    mainPanel.add(scrollPane, gbc);

    // Painel para botao refresh
    JPanel refreshPanel = new JPanel(new GridBagLayout());
    gbc = new GBC(0, 0).center().insets(0, 20, 0, 20);
    refreshPanel.add(new JButton(refreshROProjAction), gbc);
    gbc = new GBC(0, 4).none();
    mainPanel.add(refreshPanel, gbc);

    gbc = new GBC(0, 0).west().insets(10, 15, 10, 0);
    JLabel filterLabel =
      new JLabel(LNG.get("ProjectSelectonPanel.filter.label"));
    filterPanel.add(filterLabel, gbc);

    filterText = new JTextField();
    gbc = new GBC(1, 0).insets(10).horizontal().filly();
    filterPanel.add(filterText, gbc);

    filterButton = new JButton(LNG.get("ProjectSelectonPanel.filter.clear"));
    gbc = new GBC(2, 0).east().insets(10, 0, 10, 15);
    filterPanel.add(filterButton, gbc);

    gbc = new GBC(0, 5).horizontal().northwest();
    mainPanel.add(filterPanel, gbc);

    setupFilterControls();
  }

  /**
   * Configura os controles (textfield + boto) para filtragem da tabela com os
   * usurios.
   */
  private void setupFilterControls() {
    /*
     * Implementamos um DocumentListener para ativar o filtro quando o contedo
     * do campo for alterado de qualquer forma
     */
    final Document document = filterText.getDocument();

    document.addDocumentListener(new DocumentListener() {
      @Override
      public void changedUpdate(DocumentEvent e) {
        filterTableContent();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        filterTableContent();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        filterTableContent();
      }
    });

    /*
     * Queremos que o contedo do filtro seja todo selecionado quando o campo
     * ganhar o foco
     */
    filterText.addFocusListener(new FocusListener() {
      @Override
      public void focusGained(FocusEvent e) {
        filterText.selectAll();
      }

      @Override
      public void focusLost(FocusEvent e) {
        filterText.select(0, 0);
      }
    });

    /*
     * ao do boto "limpar"
     */
    filterButton.addActionListener(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        filterText.setText("");
        filterTableContent();
      }
    });
  }

  /**
   * Filtra o contedo da tabela a partir do contedo do campo "filtro". O texto
   * do campo  usado como <code>".*texto.*"</code>.
   */
  private void filterTableContent() {
    String text = filterText.getText();
    if (!text.isEmpty()) {

      //o "(?i)" define que a regex  case-insensitive
      projectsTable
        .setRowFilter(RowFilter.regexFilter("(?i).*" + text + ".*", 0, 1));
    }
    else {
      projectsTable.setRowFilter(null);
    }
    projectsTable.setSelectedFirstRow();
  }

  /**
   * Atualiza a tabela que faz a listagem dos projetos disponveis para serem
   * abertos.
   * 
   * @param projectList - Lista de projetos.
   * @param visibleColumns colunas visveis
   */
  public void updateProjectTable(List<RowData> projectList,
    VisibleColumns visibleColumns) {
    projectsTable.setVisibleColumns(visibleColumns);
    projectsTable.setNewValues(projectList);
    projectsTable.sort(0, SortOrder.ASCENDING);
    projectsTable.setSelectedFirstRow();
  }

  /**
   * Obtm a lista de todos os projetos (compartilhados ou no) de todos os
   * usurios. Disponvel apenas para o admin.
   * 
   * @param useCache <code>true</code> para reusar dados previamente coletados,
   *        <code>false</code> para buscar novos dados do servidor
   * @return Lista de todos os projetos (compartilhados ou no) de todos os
   *         usurios.
   */
  private List<RowData> getAllUsersProjectsList(boolean useCache) {
    if (useCache && allUsersProjectsList != null) {
      return allUsersProjectsList;
    }

    Task<List<RowData>> task = new RemoteTask<List<RowData>>() {

      @Override
      protected void performTask() throws Exception {
        List<RowData> projectsList = new ArrayList<RowData>();
        List<UserOutline> allUsers = User.getAllOutlines();
        for (UserOutline outline : allUsers) {
          List<RowData> projectListTemp = getAllProjectsByUser(outline);
          if ((projectListTemp != null) && (!projectListTemp.isEmpty())) {
            projectsList.addAll(projectListTemp);
          }
        }
        setResult(projectsList);
      }
    };

    if (!task.execute(getOwner(), getDialogTitle(),
      LNG.get("OpenAnyProjectDialog.getUsers"))) {
      return null;
    }

    allUsersProjectsList = task.getResult();
    return allUsersProjectsList;
  }

  /**
   * Obtm a lista de projetos do usurio corrente.
   * 
   * @param useCache <code>true</code> para reusar dados previamente coletados,
   *        <code>false</code> para buscar novos dados do servidor
   * @return Lista de projetos cujo usurio logado seja o dono.
   */
  private List<RowData> getCurrentUserProjectsList(boolean useCache) {
    if (useCache && currentUserProjectsList != null) {
      return currentUserProjectsList;
    }
    RemoteTask<List<RowData>> task = new RemoteTask<List<RowData>>() {
      @Override
      protected void performTask() throws Exception {
        setResult(getAllProjectsByUser(User.getLoggedUser().getOutline()));
      }
    };
    if (!task.execute(getOwner(), getDialogTitle(),
      LNG.get("OpenAnyProjectDialog.getUsers"))) {
      return null;
    }
    currentUserProjectsList = task.getResult();
    return currentUserProjectsList;
  }

  /**
   * @param outLine - Usurios cujos projetos sero listados.
   * @return - Lista de projetos por usurios, estejam os projetos
   *         compartilhados ou no.
   * @throws RemoteException
   */
  private List<RowData> getAllProjectsByUser(UserOutline outLine)
    throws RemoteException {
    List<RowData> projectList = new ArrayList<RowData>();
    UserOutline user = outLine;
    List<UserProjectInfo> projNames =
      CommonClientProject.getAllProjects(user.getId());
    if (projNames != null) {
      for (int i = 0; i < projNames.size(); i++) {
        RowData rowData = new RowData(user, projNames.get(i));
        projectList.add(rowData);
      }
    }
    return projectList;

  }

  /**
   * Obtm a lista de projetos compartilhados com o usurio corrente.
   * 
   * @param useCache <code>true</code> para reusar dados previamente coletados,
   *        <code>false</code> para buscar novos dados do servidor
   * @return Lista de projetos compartilhados com o usurio corrente.
   */
  private List<RowData> getSharedProjects(boolean useCache) {
    if (useCache && sharedProjectsList != null) {
      return sharedProjectsList;
    }

    final Object currentUserId = User.getLoggedUser().getId();

    Task<List<RowData>> task = new RemoteTask<List<RowData>>() {
      @Override
      protected void performTask() throws Exception {
        List<RowData> projectList = new ArrayList<RowData>();
        List<UserProjectInfo> allUserProjects =
          CommonClientProject.getAllUserProjectsFromOthers(currentUserId);

        for (UserProjectInfo userProjectInfo : allUserProjects) {
          Object userId = userProjectInfo.getOwnerId();
          User user = User.getUser(userId);
          if (user == null) {
            // TODO devemos fazer mais do que isso?
            System.err.println(String.format(
              "Usurio '%s' no est cadastrado mas possui projetos",
              userId.toString()));
            continue;
          }
          RowData rowData = new RowData(user.getOutline(), userProjectInfo);
          projectList.add(rowData);
        }
        setResult(projectList);
      }
    };

    String msg = LNG.get("CommonProjectSelectDialog.info.get.projects");

    sharedProjectsList = null;
    if (!task.execute(getOwner(), getDialogTitle(), msg)) {
      return null;
    }
    sharedProjectsList = task.getResult();
    return sharedProjectsList;
  }

  /**
   * Cria um painel com os botes que vo ser usados no dilogo requisitado.
   * Esse mtodo deve ser sobrescrito dependendo acao da selecao.
   * 
   * @return O painel recm-criado.
   */
  protected JPanel makeButtonPanel() {
    JPanel panel = new JPanel();
    defaultButton = (JButton) panel.add(new JButton(openAction));

    JButton cancelButton = (JButton) panel.add(new JButton(cancelAction));

    GUIUtils
      .matchPreferredSizes(new JComponent[] { defaultButton, cancelButton });
    return panel;
  }

  /**
   * Executa a ao de abrir o projeto selecionado e instalar um observador para
   * ele.
   */
  protected void handleOpenAction() {
    if (selectedProject == null) {
      StandardDialogs.showErrorDialog(getOwner(), getDialogTitle(),
        LNG.get("ProjectOpenDialog.error.no.selection"));
      return;
    }
    DesktopFrame desktop = DesktopFrame.getInstance();
    Object projectId = selectedProject.project.getProjectId();
    if (!desktop.checkOpenableProject(projectId)) {
      return;
    }

    // Fecha a janela antes de tentar abrir o novo projeto para evitar que esse 
    // dilogo fique em deadlock com outros dilogos de confirmao/erro.
    close();
    desktop.openProject(projectId);
  }

  /**
   * Executa a ao de refresh da lista de projetos
   */
  protected void handleRefreshAction() {
    /*
     * Dependendo do Radio selecionado, atualizamos a lista de projetos
     * especfica.
     */
    if (choiceOpenUser.isSelected()) {
      updateProjectTable(getCurrentUserProjectsList(false),
        VisibleColumns.NAME);
    }
    else if (choiceOpenShared.isSelected()) {
      updateProjectTable(getSharedProjects(false),
        VisibleColumns.OWNER_NAME_PERMISSION);
    }
    else if (choiceOpenAny.isSelected()) {
      updateProjectTable(getAllUsersProjectsList(false),
        VisibleColumns.OWNER_NAME);
    }

    /*
     * Seleciona o primeiro projeto da lista
     */
    projectsTable.setSelectedFirstRow();
  }

  /**
   * Informa o ttulo que vai ser utilizado na janela de dilogo. Esse mtodo
   * depende da ao que originou o dilogo.
   * 
   * @return O ttulo da janela.
   */
  public String getDialogTitle() {
    return LNG.get("ProjectOpenDialog.title.open");
  }

}

/**
 * Dispatcher para redirecionar eventos de teclado no dilogo de abertura de
 * projeto.<br>
 * Eventos cuja tecla associada  UP ou DOWN so redirecionados para a tabela de
 * projetos. Eventos cuja tecla associada  BACKSPACE, DELETE ou um caracter que
 * pode fazer parte do nome de um usurio ou projeto so redirecionados para o
 * filtro da tabela.
 * 
 * @author Tecgraf
 */
class ProjectKeyboardFocusManager extends DefaultKeyboardFocusManager {

  /** Dilogo de abertura de projeto. */
  private ProjectOpenDialog dialog;

  /** Tabela de projetos. */
  private ProjectTableSortableTable projectsTable;

  /** Campo de texto usado como filtro para a tabela. */
  private JTextField filterText;

  /**
   * Construtor
   * 
   * @param dialog dilogo de abertura de projeto.
   * @param projectsTable tabela de projetos
   * @param filterText campo de texto usado como filtro para a tabela.
   */
  ProjectKeyboardFocusManager(ProjectOpenDialog dialog,
    ProjectTableSortableTable projectsTable, JTextField filterText) {
    super();
    this.dialog = dialog;
    this.projectsTable = projectsTable;
    this.filterText = filterText;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean dispatchKeyEvent(KeyEvent e) {
    if (!dialog.isFocused()) {
      // Se o dilogo de abertura de projeto no for o detentor do foco, este
      // dispatcher no deve tratar o evento de teclado.
      return false;
    }

    int keyCode = e.getKeyCode();
    if (keyCode == KeyEvent.VK_UP || keyCode == KeyEvent.VK_DOWN) {
      redispatchEvent(projectsTable, e);
      return true;
    }

    if (isValidToFilterText(e)) {
      redispatchEvent(filterText, e);
      return true;
    }
    return false;
  }

  /**
   * Verifica se a tecla digitada deve ser redirecionada para o textfield de
   * filto para a tabela.
   * 
   * @param e evento de teclado que gerado pela digitao de uma tecla.
   * @return true se a tecla digitada deve ser redirecionada para o filtro da
   *         tabela, false caso contrrio.
   */
  private boolean isValidToFilterText(KeyEvent e) {
    if (e.getKeyCode() == KeyEvent.VK_DELETE
      || e.getKeyCode() == KeyEvent.VK_BACK_SPACE) {
      return true;
    }
    return isValidChar(e.getKeyChar());
  }

  /**
   * Verifica se o caracter digitado  vlido para um nome de usurio ou
   * projeto.
   * 
   * @param c caracter digitado.
   * @return true se o caracter digitado  vlido para um nome de usurio ou
   *         projeto, false caso contrrio.
   */
  private boolean isValidChar(char c) {
    if (Character.isLetterOrDigit(c) || c == '.' || c == '_') {
      return true;
    }
    return false;
  }
}