/*
 * $Id:$
 */

package csbase.client.remote.manager.server;

import java.rmi.RemoteException;
import java.security.cert.Certificate;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.SortedSet;

import csbase.client.desktop.RemoteTask;
import csbase.logic.User;
import csbase.logic.server.ServerInfo;
import csbase.logic.server.ServerInfoAddEvent;
import csbase.logic.server.ServerInfoData;
import csbase.logic.server.ServerInfoEvent;
import csbase.logic.server.ServerInfoEventType;
import csbase.logic.server.ServerInfoModifyEvent;
import csbase.logic.server.ServerInfoRemoveEvent;
import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;

/**
 * <p>
 * Representa um gerenciador de informaes de servidores.
 * </p>
 * 
 * <p>
 *  responsvel por verificar atualizaes no repositrio de informaes de
 * servidores localizado no servidor central.
 * </p>
 * 
 * <p>
 * Componentes do cliente que precisem estar atualizados com o repositrio de
 * informaes de servidores devem se cadastrar como ouvintes neste gerenciador
 * para receber as devidas notificaes.
 * </p>
 * 
 * <p>
 * Os mtodos desta classe normalmente faro acessos ao servidor. Portanto,
 * chamadas  mtodos desta classe devem ser realizadas com o uso de
 * {@link RemoteTask}. Os mtodos que no fazem acesso ao servidor informaro
 * explicitamente esse comportamento.
 * </p>
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ServerInfoManager {
  /**
   * A instncia nica do gerenciador de informaes de servidores.
   */
  private static ServerInfoManager instance;

  /**
   * Representa o estado do gerenciador de informaes de servidores.
   */
  private ServerInfoManagerState state;

  /**
   * Representa o estado habilitado do gerenciador de informaes de servidores.
   */
  private static final ServerInfoManagerEnabledState ENABLED_STATE =
    new ServerInfoManagerEnabledState();

  /**
   * Representa o estado desabilitado do gerenciador de informaes de
   * servidores.
   */
  private static final ServerInfoManagerDisabledState DISABLED_STATE =
    new ServerInfoManagerDisabledState();

  /**
   * Cria um gerenciador de informaes de servidores locais.
   */
  private ServerInfoManager() {
    if (this.canEnable()) {
      this.state = ENABLED_STATE;
    }
    else {
      this.state = DISABLED_STATE;
    }
  }

  /**
   * Verifica se pode ser habilitado ou no.
   * 
   * @return true, caso possa ser habilitado, ou false, caso contrrio.
   */
  private boolean canEnable() {
    if (ClientRemoteLocator.serverService != null) {
      if (User.getLoggedUser().isAdmin()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Obtm a instncia nica do gerenciador de informaes de servidores locais.
   * 
   * @return A instncia nica do gerenciador de informaes de servidores
   *         locais.
   */
  public static ServerInfoManager getInstance() {
    if (instance == null) {
      instance = new ServerInfoManager();
    }
    return instance;
  }

  /**
   * Inicia o monitoramento ao servio de informaes de servidores locais.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public void start() throws RemoteException {
    this.state.start();
  }

  /**
   * Para o monitoramento ao servio de informaes de servidores locais.
   */
  public void stop() {
    this.state.stop();
  }

  /**
   * Obtm um conjunto ordenado (por nome) com informaes dos servidores locais
   * pertencentes ao repositrio.
   * 
   * @return Informaes sobre os servidores locais.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public SortedSet<ServerInfo> getServersInfos() throws RemoteException {
    return this.state.getServersInfos();
  }

  /**
   * Adiciona informaes de um servidor local ao repositrio.
   * 
   * @param localServer Informaes de servidor local a serem adicionadas.
   * 
   * @return true, caso as informaes tenham o sido adicionadas, ou false, caso
   *         j existam informaes de um servidor local com o mesmo nome.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public boolean addServerInfo(ServerInfoData localServer)
    throws RemoteException {
    return this.state.addServerInfo(localServer);
  }

  /**
   * Adiciona o certificado a um servidor.
   * 
   * @param serverName O nome do servidor a receber o certificado.
   * @param cert A Instncia do certificado
   * @return true se o certificado foi adicionado com sucesso, false caso
   *         contrrio
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public boolean addCertificate(String serverName, Certificate cert)
    throws RemoteException {
    return this.state.addCertificate(serverName, cert);
  }

  /**
   * Remove um certificado do servidor.
   * 
   * @param serverName O nome do servidor
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public void removeCertificate(String serverName) throws RemoteException {
    this.state.removeCertificate(serverName);
  }

  /**
   * Modifica as informaes de um servidor local no repositrio.
   * 
   * @param oldLocalServer Informaes do servidor local no seu estado antigo.
   * @param newLocalServer Informaes do servidor local em seu estado atual.
   * 
   * @return true, caso as informaes tenham sido modificadas, ou false, caso
   *         j exista informaes de um outro servidor local com o mesmo nome.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public boolean modifyServerInfo(ServerInfo oldLocalServer,
    ServerInfoData newLocalServer) throws RemoteException {
    return this.state.modifyServerInfo(oldLocalServer, newLocalServer);
  }

  /**
   * Remove as informaes de um servidor local.
   * 
   * @param localServer Informaes do servidor local a serem removidas.
   * 
   * @throws RemoteException Caso ocorra algum erro na comunicao com o
   *         servio.
   */
  public void removeServerInfo(ServerInfo localServer) throws RemoteException {
    this.state.removeServerInfo(localServer);
  }

  /**
   * Adiciona um ouvinte  lista de ouvintes.
   * 
   * @param listener O ouvinte a ser adicionado.
   */
  public void addServerInfoManagerListener(ServerInfoManagerListener listener) {
    this.state.addServerInfoManagerListener(listener);
  }

  /**
   * Remove um ouvinte da lista de ouvintes.
   * 
   * @param listener O ouvinte a ser removido.
   */
  public void removeServerInfoManagerListener(ServerInfoManagerListener listener) {
    this.state.removeServerInfoManagerListener(listener);
  }

  /**
   * Verifica se o gerenciador de informaes de servidores locais est
   * habilitado ou no.
   * 
   * @return true, caso esteja habilitado, ou false, caso contrrio.
   */
  public boolean isEnabled() {
    return this.state.equals(ENABLED_STATE);
  }

  /**
   * Representa um estado de um gerenciador de informaes de servidores.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private interface ServerInfoManagerState {

    /**
     * Inicia o gerente
     * 
     * @throws RemoteException em caso de falha de comunicao
     * @throws UnsupportedOperationException no caso da operao no ser
     *         suportada.
     */
    void start() throws RemoteException, UnsupportedOperationException;

    /**
     * Finaliza o gerente
     * 
     * @throws UnsupportedOperationException no caso da operao no ser
     *         suportada.
     */
    void stop() throws UnsupportedOperationException;

    /**
     * Consulta um conjunto de informaes de servidores.
     * 
     * @return o conjunto
     * @throws RemoteException em caso de falha de comunicao
     * @throws UnsupportedOperationException no caso da operao no ser
     *         suportada.
     */
    SortedSet<ServerInfo> getServersInfos() throws RemoteException,
      UnsupportedOperationException;

    /**
     * Adicioa informaes de um servidor local.
     * 
     * @param localServer servidor local.
     * @return indicativo de sucesso.
     * @throws RemoteException em caso de falha de comunicao
     * @throws UnsupportedOperationException no caso da operao no ser
     *         suportada.
     */
    boolean addServerInfo(ServerInfoData localServer) throws RemoteException,
      UnsupportedOperationException;

    /**
     * Modifiac dados de um servidor local.
     * 
     * @param oldLocalServer dados antigos do servidor local.
     * @param newLocalServer dados novos do servidor local.
     * @return indicativo de sucesso.
     * @throws RemoteException em caso de falha de comunicao
     * @throws UnsupportedOperationException no caso da operao no ser
     *         suportada.
     */
    boolean modifyServerInfo(ServerInfo oldLocalServer,
      ServerInfoData newLocalServer) throws RemoteException,
      UnsupportedOperationException;

    /**
     * @param localServer o servidor local
     * @throws RemoteException em caso de falha de comunicao
     * @throws UnsupportedOperationException no caso da operao no ser
     *         suportada.
     */
    void removeServerInfo(ServerInfo localServer) throws RemoteException,
      UnsupportedOperationException;

    /**
     * @param listener o listener
     * @throws UnsupportedOperationException no caso da operao no ser
     *         suportada.
     */
    void addServerInfoManagerListener(ServerInfoManagerListener listener)
      throws UnsupportedOperationException;

    /**
     * @param listener o listener
     * @throws UnsupportedOperationException no caso da operao no ser
     *         suportada.
     */
    void removeServerInfoManagerListener(ServerInfoManagerListener listener)
      throws UnsupportedOperationException;

    /**
     * Adiciona certificado.
     * 
     * @param serverName nome do servidor
     * @param cert certificado.
     * @return indicativo de sucesso.
     * @throws RemoteException em caso de exceo de comunicao
     * @throws UnsupportedOperationException no caso de operao no suportada
     */
    boolean addCertificate(String serverName, Certificate cert)
      throws RemoteException, UnsupportedOperationException;

    /**
     * Remove certificado.
     * 
     * @param serverName nome do servidor
     * @throws RemoteException em caso de exceo de comunicao
     * @throws UnsupportedOperationException no caso de operao no suportada
     */
    void removeCertificate(String serverName) throws RemoteException,
      UnsupportedOperationException;
  }

  /**
   * Representa o estado desabilitado do gerenciador de informaes de
   * servidores locais.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private static final class ServerInfoManagerDisabledState implements
    ServerInfoManagerState {
    /**
     * {@inheritDoc}
     */
    @Override
    public void start() throws UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void stop() throws UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public SortedSet<ServerInfo> getServersInfos()
      throws UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean addServerInfo(ServerInfoData localServer)
      throws RemoteException, UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean modifyServerInfo(ServerInfo oldLocalServer,
      ServerInfoData newLocalServer) throws RemoteException,
      UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeServerInfo(ServerInfo localServer)
      throws RemoteException, UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addServerInfoManagerListener(ServerInfoManagerListener listener)
      throws UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeServerInfoManagerListener(
      ServerInfoManagerListener listener) throws UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean addCertificate(String serverName, Certificate cert)
      throws UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeCertificate(String serverName) throws RemoteException,
      UnsupportedOperationException {
      throw new UnsupportedOperationException();
    }
  }

  /**
   * Representa o estado habilitado do gerenciador de informaes de servidores
   * locais.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private static final class ServerInfoManagerEnabledState implements
    ServerInfoManagerState, Observer {
    /**
     * Lista com os ouvintes interessados no contedo do repositrio de
     * servidores locais.
     */
    private List<ServerInfoManagerListener> listenerList;

    /**
     * O observador do servio de servidores locais.
     */
    private ServerServiceRemoteObserver remoteObserver;

    /**
     * Construtor
     */
    ServerInfoManagerEnabledState() {
      this.listenerList = new LinkedList<ServerInfoManagerListener>();
    }

    /**
     * Verifica se o observador do servio de servidores locais j foi iniciado.
     * 
     * @return true, caso o observador j tenha sido iniciado, ou false, caso
     *         contrrio.
     */
    private boolean wasStarted() {
      return (this.remoteObserver != null);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void start() throws RemoteException {
      if (this.wasStarted()) {
        throw new IllegalStateException(
          "Tentativa de iniciar o gerenciador de informaes de servidores locais que j foi iniciado.");
      }
      this.remoteObserver = new ServerServiceRemoteObserver();
      this.remoteObserver.addObserver(this);
      this.remoteObserver.install();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void stop() {
      if (!this.wasStarted()) {
        throw new IllegalStateException(
          "Tentativa de para o gerenciador de informaes de servidores locais sem ter sido iniciado.");
      }
      this.remoteObserver.deleteObserver(this);
      this.remoteObserver = null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public SortedSet<ServerInfo> getServersInfos() throws RemoteException {
      return ClientRemoteLocator.serverService.getServersInfos();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean addServerInfo(ServerInfoData localServer)
      throws RemoteException {
      return ClientRemoteLocator.serverService.addServerInfo(localServer);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean modifyServerInfo(ServerInfo oldLocalServer,
      ServerInfoData newLocalServer) throws RemoteException {
      return ClientRemoteLocator.serverService.modifyServerInfo(oldLocalServer,
        newLocalServer);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeServerInfo(ServerInfo localServer) throws RemoteException {
      ClientRemoteLocator.serverService.removeServerInfo(localServer);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addServerInfoManagerListener(ServerInfoManagerListener listener) {
      if (!this.wasStarted()) {
        throw new IllegalStateException(
          "Tentativa de adicionar ouvinte ao gerenciador de informaes de servidores locais que no foi iniciado.");
      }
      this.listenerList.add(listener);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeServerInfoManagerListener(
      ServerInfoManagerListener listener) {
      if (!this.wasStarted()) {
        throw new IllegalStateException(
          "Tentativa de remover ouvinte do gerenciador de informaes de servidores locais que no foi iniciado.");
      }
      this.listenerList.remove(listener);
    }

    /**
     * Tratador de evento de adio de informaes de um servidor local.
     * 
     * @param event O evento de adio de informaes de um servidor local.
     */
    private void handleEvent(ServerInfoAddEvent event) {
      this.fireWasAddedLocalServer(event.getServerInfo());
    }

    /**
     * Tratador de evento de modificao de informaes de um servidor local.
     * 
     * @param event O evento de modificao de informaes de um servidor local.
     */
    private void handleEvent(ServerInfoModifyEvent event) {
      this.fireWasModifiedLocalServer(event.getOldServerInfo(), event
        .getNewServerInfo());
    }

    /**
     * Tratador de evento de remoo de informaes de um servidor local.
     * 
     * @param event O evento de remoo de informaes de um servidor local.
     */
    private void handleEvent(ServerInfoRemoveEvent event) {
      this.fireWasRemovedLocalServer(event.getServerInfo());
    }

    /**
     * Informa a todos os ouvintes que informaes de um servidor local foram
     * adicionadas ao repositrio.
     * 
     * @param localServer Informaes que foram adicionadas.
     */
    private void fireWasAddedLocalServer(ServerInfo localServer) {
      Iterator<ServerInfoManagerListener> listenerIterator =
        this.listenerList.iterator();
      while (listenerIterator.hasNext()) {
        ServerInfoManagerListener listener = listenerIterator.next();
        listener.wasAddedServerInfo(localServer);
      }
    }

    /**
     * Notifica aos ouvintes interessados que um informaes de servidor local
     * foram modificadas.
     * 
     * @param oldLocalServer Informaes do servidor local em seu estado antigo
     *        (antes da modificao).
     * @param newLocalServer Informaes do servidor local em seu estado atual
     *        (j modificado).
     */
    private void fireWasModifiedLocalServer(ServerInfo oldLocalServer,
      ServerInfo newLocalServer) {
      Iterator<ServerInfoManagerListener> listenerIterator =
        this.listenerList.iterator();
      while (listenerIterator.hasNext()) {
        ServerInfoManagerListener listener = listenerIterator.next();
        listener.wasModifiedServerInfo(oldLocalServer, newLocalServer);
      }
    }

    /**
     * Notifica aos ouvintes interessados que informaes de um servidor local
     * foram removidas.
     * 
     * @param localServer Informaes removidas.
     */
    private void fireWasRemovedLocalServer(ServerInfo localServer) {
      Iterator<ServerInfoManagerListener> listenerIterator =
        this.listenerList.iterator();
      while (listenerIterator.hasNext()) {
        ServerInfoManagerListener listener = listenerIterator.next();
        listener.wasRemovedServerInfo(localServer);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void update(Observable o, Object arg) {
      ServerInfoEvent localServerEvent = (ServerInfoEvent) arg;
      if (localServerEvent.getType().equals(ServerInfoEventType.ADD)) {
        handleEvent((ServerInfoAddEvent) localServerEvent);
      }
      else if (localServerEvent.getType().equals(ServerInfoEventType.MODIFY)) {
        handleEvent((ServerInfoModifyEvent) localServerEvent);
      }
      else if (localServerEvent.getType().equals(ServerInfoEventType.REMOVE)) {
        handleEvent((ServerInfoRemoveEvent) localServerEvent);
      }
      else {
        throw new IllegalStateException(
          "Recebido um evento de um tipo desconhecido: "
            + localServerEvent.getType());
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean addCertificate(String serverName, Certificate cert)
      throws RemoteException {
      return ClientRemoteLocator.serverService.addCert(serverName, cert);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeCertificate(String serverName) throws RemoteException {
      ClientRemoteLocator.serverService.removeCert(serverName);
    }
  }

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      @Override
      public void restart() {
        ServerInfoManager.instance = null;
      }
    });
  }
}
