/*
 * $Id: DiskUsageProxy.java 150777 2014-03-19 14:16:56Z oikawa $
 */

package csbase.client.remote.srvproxies;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import javax.swing.JFrame;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommonClientProject;
import csbase.logic.diskusageservice.DiskOccupation;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.DiskUsageServiceInterface;

/**
 * Classe que modela um proxy Cliente/DiskUsageService.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DiskUsageProxy {

  /**
   * Consulta um texto de internacionalizao.
   * 
   * @param tag o tag
   * @return o texto
   */
  private static String getString(final String tag) {
    final String className = DiskUsageProxy.class.getSimpleName();
    final String newTag = className + "." + tag;
    return LNG.get(newTag);
  }

  /**
   * Monta texto traduzido com base em um identificador de rea de disco no
   * servidor (monitorada).
   * 
   * @param id identificador
   * @return texto
   */
  public static String translateAreaId(final String id) {
    final String prefix = DiskUsageProxy.class.getSimpleName();
    final String key = prefix + "." + id + ".name";
    final String idText = "[Id: " + id + "]";
    final String text = !LNG.hasKey(key) ? idText : LNG.get(key);
    return text;
  }

  /**
   * Obtm o uso de disco por usurios.
   * 
   * @return tabela (hash) contendo uso de disco (em Mb) para cada usurio
   */
  public static Hashtable<Object, Double> getUsersList() {
    RemoteTask<Hashtable<Object, Double>> task =
      new RemoteTask<Hashtable<Object, Double>>() {
        @Override
        public void performTask() throws Exception {
          final DiskUsageServiceInterface dskService =
            ClientRemoteLocator.diskUsageService;
          if (dskService == null) {
            return;
          }
          setResult(dskService.getUsersList());
        }
      };

    final DesktopFrame mainFrame = DesktopFrame.getInstance();
    if ((mainFrame == null)) {
      return null;
    }

    final JFrame desktop = mainFrame.getDesktopFrame();
    final String PRE = "disk.request";

    final String title = getString(PRE + ".title");
    final String message = getString(PRE + ".message");
    if (task.execute(desktop, title, message)) {
      Hashtable<Object, Double> info = task.getResult();
      if (info != null) {
        return info;
      }
    }

    final String errTitle = getString(PRE + ".error.title");
    final String errMessage = getString(PRE + ".error.message");
    StandardErrorDialogs.showErrorDialog(desktop, errTitle, errMessage);
    return null;
  }

  /**
   * Obtm o uso de disco de um projeto do usurio
   * 
   * @param ownerId o identificador de owner.
   * @param projName o identificador de projeto.
   * @return o valor em (Mb)
   */
  public static double getUsedSpaceForProjectMb(final Object ownerId,
    final String projName) {
    RemoteTask<Double> task = new RemoteTask<Double>() {
      @Override
      public void performTask() throws Exception {
        final DiskUsageServiceInterface dskService =
          ClientRemoteLocator.diskUsageService;
        if (dskService == null) {
          return;
        }
        setResult(dskService.getUsedSpaceForProjectMb(ownerId, projName));
      }
    };
    final String PRE = "project.usage";
    final JFrame desktop = DesktopFrame.getInstance().getDesktopFrame();
    final String title = getString(PRE + ".title");
    final String message = getString(PRE + ".message");
    if (task.execute(desktop, title, message)) {
      final double used = task.getResult();
      return used;
    }

    final String errTitle = getString(PRE + ".error.title");
    final String errMessage = getString(PRE + ".error.message");
    StandardErrorDialogs.showErrorDialog(desktop, errTitle, errMessage);
    return 0.0;
  }

  /**
   * Obtm informao do servio de ocupao de espao em disco. Em caso de erro
   * na obteno da informao (valor -1 informado pelo servio) exibe janela de
   * erro.
   * 
   * @param task a tarefa de consulta ao servidor
   * @return informao solicitada 0 em caso de erro
   */
  private static DiskOccupation getInfo(RemoteTask<DiskOccupation> task) {
    final JFrame desktop = DesktopFrame.getInstance().getDesktopFrame();

    /*
     * O servio de ocupao de disco no lana excees por erro na obteno da
     * informao (retorna -1). Falhas na chamada a execute ento resultam de
     * erros de comunicao (j sinalizado ao usurio) ou de cancelamento (no
     * necessita sinalizao)
     */
    final String title = LNG.get("DISK_REQUEST_TITLE");
    final String msg = LNG.get("DISK_REQUEST_MESSAGE");
    task.setProgressDialogDelay(20);
    final boolean executed = task.execute(desktop, title, msg);
    final DiskOccupation dummy = new DiskOccupation();
    if (executed) {
      final DiskOccupation result = task.getResult();
      if (result == null) {
        return dummy;
      }
      return result;
    }
    return dummy;
  }

  /**
   * Mtodo para obter o espao livre/ocupado no disco do projeto corrente.
   * 
   * @return espao livre/ocupado.
   */
  public static DiskOccupation getCurrentProjectOccupation() {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    if (project == null) {
      return new DiskOccupation();
    }
    RemoteTask<DiskOccupation> task = new RemoteTask<DiskOccupation>() {
      @Override
      public void performTask() throws Exception {
        final DiskUsageServiceInterface diskUsageService =
          ClientRemoteLocator.diskUsageService;
        final Object owner = project.getUserId();
        final String projectName = project.getName();
        final DiskOccupation serverOccupation =
          diskUsageService.getSingleProjectOccupation(owner, projectName);
        setResult(serverOccupation);
      }

      /**
       * {@inheritDoc}
       */
      @Override
      protected void handleError(Exception error) {
        setResult(null);
      }
    };
    return getInfo(task);
  }

  /**
   * Mtodo para obter o espao livre no disco do servidor.
   * 
   * @param id identificador de rea.
   * @return espao livre em um disco ou 0 em caso de erro
   */
  public static DiskOccupation getAreaOccupation(final String id) {
    RemoteTask<DiskOccupation> task = new RemoteTask<DiskOccupation>() {
      @Override
      public void performTask() throws Exception {
        final DiskUsageServiceInterface diskUsageService =
          ClientRemoteLocator.diskUsageService;
        final DiskOccupation serverOccupation =
          diskUsageService.getAreaOccupation(id);
        setResult(serverOccupation);
      }
    };
    return getInfo(task);
  }

  /**
   * Mtodo para obter o espao livre no disco do servidor.
   * 
   * @return espao livre em um disco ou 0 em caso de erro
   */
  public static List<String> getAllAreasIds() {
    RemoteTask<List<String>> task = new RemoteTask<List<String>>() {
      @Override
      public void performTask() throws Exception {
        final DiskUsageServiceInterface diskUsageService =
          ClientRemoteLocator.diskUsageService;
        final List<String> list = diskUsageService.getAllAreasIds();
        setResult(list);
      }
    };

    final String PRE = "project.usage";
    final JFrame desktop = DesktopFrame.getInstance().getDesktopFrame();
    final String message = getString(PRE + ".message");
    if (task.execute(desktop, null, message)) {
      final List<String> result = task.getResult();
      return result;
    }

    final String errMessage = getString(PRE + ".error.message");
    StandardErrorDialogs.showErrorDialog(desktop, null, errMessage);
    final List<String> emptyList = new ArrayList<String>();
    return emptyList;
  }

  /**
   * Mtodo para obter o espao livre no disco do servidor.
   * 
   * @return espao livre em um disco ou 0 em caso de erro
   */
  public static DiskOccupation getServerOccupation() {
    final String id = DiskUsageServiceInterface.DIR_SERVER_ID;
    final DiskOccupation occ = getAreaOccupation(id);
    return occ;
  }

  /**
   * Mtodo para obter o espao livre em no disco de algoritmos.
   * 
   * @return espao livre em um disco ou 0 em caso de erro
   */
  public static DiskOccupation getAlgorithmOccupation() {
    final String id = DiskUsageServiceInterface.DIR_ALGORITHM_ID;
    return getAreaOccupation(id);
  }

  /**
   * Mtodo para obter o espao livre no disco de projetos.
   * 
   * @return espao livre em um disco ou 0 em caso de erro
   */
  public static DiskOccupation getProjectOccupation() {
    final String id = DiskUsageServiceInterface.DIR_PROJECT_ID;
    return getAreaOccupation(id);
  }
}
