package csbase.client.rest.resources.v1;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import csbase.client.Client;
import csbase.client.rest.RestController;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import io.swagger.annotations.Authorization;
import tecgraf.javautils.version.VersionNumber;

/**
 * Responsvel pelo atendimento REST de verso do cliente.
 *
 * @author Tecgraf/PUC-Rio
 */
@Path("v1/")
@Api(value = "Base", authorizations = {
    @Authorization(value = "Bearer", scopes = {}) })
public class BaseRestService {

  /**
   * Encapsulador de verso
   *
   * @author Tecgraf/PUC-Rio
   */
  public static class VersionWrapper {
    /**
     * Texto de verso
     */
    final private VersionNumber versionNumber;

    /**
     * Consulta de texto de verso.
     *
     * @return texto
     */
    public String getVersion() {
      return versionNumber.toString();
    }

    /**
     * Conultado de nmero major de verso.
     *
     * @return nmero
     */
    public int getMajor() {
      return versionNumber.getMajor();
    }

    /**
     * Conultado de nmero minor de verso.
     *
     * @return nmero
     */

    public int getMinor() {
      return versionNumber.getMinor();
    }

    /**
     * Conultado de nmero patch de verso.
     *
     * @return nmero
     */
    public int getPatch() {
      return versionNumber.getPatch();
    }

    /**
     * Construtor
     *
     * @param versionText texto de verso consultado do sistema.
     */
    public VersionWrapper(String versionText) {
      VersionNumber vn = VersionNumber.fromString(versionText);
      if (vn == null) {
        vn = VersionNumber.ZERO_VERSION;
      }
      this.versionNumber = vn;
      this.versionNumber.freeze();
    }
  }

  /**
   * Consulta a verso do sistema (client-side)
   *
   * @return um texto com a verso.
   */
  @GET
  @Path("version")
  @Produces(MediaType.APPLICATION_JSON)
  @ApiOperation(value = "Get current client version.", notes = "This can only be done by the logged in user.", response = VersionWrapper.class)
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation", response = VersionWrapper.class), })
  public Response getVersion() {
    final String version = Client.getInstance().getVersion();
    final VersionWrapper versionWrapper = new VersionWrapper(version);
    return Response.status(Status.OK).entity(versionWrapper).build();
  }

  /**
   * Consulta a verso do sistema (client-side)
   *
   * @return um texto com a verso.
   */
  @GET
  @Path("token")
  @Produces(MediaType.TEXT_PLAIN)
  @ApiOperation(value = "Get current authorization token in text plain.", notes = "This can only be done by the logged in user.", response = VersionWrapper.class)
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation"), })
  public Response getAuthToken() {
    return Response.status(Status.OK).entity(RestController.getInstance()
      .getAuthToken()).build();
  }
}
