package csbase.client.util.table;

import java.awt.Component;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

/**
 * Clula de <i>ComboBoxes</i>: a clula que exibe uma {@link JComboBox}.
 */
public final class ComboBoxCellView extends AbstractCellView {

  /**
   * Os tens disponveis para seleo.
   */
  private List<ComboBoxCellViewItem> availableItems;
  /**
   * A <i>combobox</i>.
   */
  private JComboBox comboBox;
  private DefaultComboBoxModel comboBoxModel;

  /**
   * Cria uma clula de <i>combobox</i>.
   * 
   * @param availableItems A relao de tens disponveis para seleo (No
   *        aceita {@code null} e no pode estar vazia).
   */
  public ComboBoxCellView(List<ComboBoxCellViewItem> availableItems) {
    createComboBox();
    setAvailableItems(availableItems);
  }

  /**
   * Atribui os tens disponveis para seleo nesta clula.
   *
   * @param availableItems Os tens disponveis (No aceita {@code null}.
   */
  public void setAvailableItems(List<ComboBoxCellViewItem> availableItems) {
    if (availableItems == null) {
      throw new IllegalArgumentException(
        "O parmetro availableItems est nulo.");
    }

    this.availableItems = new ArrayList<ComboBoxCellViewItem>(availableItems);
    Collections.sort(this.availableItems);
    comboBoxModel.removeAllElements();
    for (ComboBoxCellViewItem item : availableItems) {
      comboBoxModel.addElement(item);
    }
  }

  @Override
  public String getTip() {
    ComboBoxCellViewItem item = getSelectedItem();
    if (item == null) {
      return null;
    }
    return item.getTip();
  }

  public ComboBoxCellViewItem getSelectedItem() {
    ComboBoxCellViewItem item = (ComboBoxCellViewItem) comboBox
      .getSelectedItem();
    return item;
  }

  @Override
  public Object getValue() {
    ComboBoxCellViewItem item = getSelectedItem();
    if (item == null) {
      return null;
    }
    return item.getValue();

  }

  @Override
  public JComponent getView() {
    return comboBox;
  }

  @Override
  protected void keepValue(Object value) {
    if (value == null) {
      comboBoxModel.setSelectedItem(null);
    }
    else {
      for (ComboBoxCellViewItem item : availableItems) {
        if (item.getValue().equals(value)) {
          comboBoxModel.setSelectedItem(item);
          return;
        }
      }
      // Se o item no estiver disponvel, criamos um item.
      // O validador da combobox indicar o erro.
      ComboBoxCellViewItem unknownItem = new ComboBoxCellViewItem(value
        .toString(), null, value);
      unknownItem.setValid(false);
      comboBoxModel.setSelectedItem(unknownItem);
    }
  }

  /**
   * Cria a <i>combobox</i>.
   */
  private void createComboBox() {
    comboBoxModel = new DefaultComboBoxModel();
    comboBox = new JComboBox();
    comboBox.setModel(comboBoxModel);
    comboBox.setRenderer(createListCellRenderer());
    comboBox.getModel().addListDataListener(createListDataListener());
  }

  /**
   * Cria um {@link ListDataListener}.
   *
   * @return .
   */
  private ListDataListener createListDataListener() {
    return new ListDataListener() {

      @Override
      public void intervalRemoved(ListDataEvent e) {
      }

      @Override
      public void intervalAdded(ListDataEvent e) {
      }

      @Override
      public void contentsChanged(ListDataEvent e) {
        ComboBoxCellViewItem selectedItem = getSelectedItem();
        if (selectedItem != null) {
          comboBox.setToolTipText(selectedItem.getTip());
        }
        else {
          comboBox.setToolTipText(null);
        }
        fireValueWasChanged();
      }
    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getValidationErrorMessage() {
    if (getSelectedItem() == null) {
      return null;
    }
    if (!availableItems.contains(getSelectedItem())) {
      return String.format(
        "O item %s no est presente na lista de itens vlidos (%s).",
        getSelectedItem(), availableItems);
    }
    return null;
  }

  /**
   * Cria o <i>renderer</i> da <i>combobox</i>.
   * 
   * @return O renderer.
   */
  private DefaultListCellRenderer createListCellRenderer() {
    return new DefaultListCellRenderer() {

      @Override
      public Component getListCellRendererComponent(JList list, Object value,
        int index, boolean isSelected, boolean cellHasFocus) {
        Component component = super.getListCellRendererComponent(list, value,
          index, isSelected, cellHasFocus);
        ComboBoxCellViewItem item = (ComboBoxCellViewItem) value;
        if (item != null) {
          setText(item.getLabel());
          if (item.getTip() != null) {
            setToolTipText(item.getTip());
          }
        }
        return component;
      }
    };
  }
}
