package csbase.client.algorithms.commands.newview;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.StatusBar;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent.Type;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.kernel.ClientException;
import csbase.client.util.gui.log.AutoReloadable;
import csbase.client.util.gui.log.LogPanelReloader;
import csbase.client.util.gui.log.tab.ConsolidatedLogTab;
import csbase.client.util.gui.log.tab.ReloadableTab;
import csbase.client.util.gui.log.tab.SimpleLogTab;
import csbase.client.util.gui.log.tab.Tab;
import csbase.client.util.gui.log.tab.AbstractTab.TabType;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.CommandViewPermission;
import csbase.logic.ProgressData;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmConfigurator;

/**
 * Base para implementao de uma viso do comando com suporte a abas.
 */
abstract class AbstractCommandView implements CommandView {

  /**
   * Barra de status.
   */
  protected StatusBar statusBar;

  /**
   * Aba selecionada no momento.
   */
  protected Tab selected;

  /**
   * Painel que contm as abas.
   */
  protected JTabbedPane tabbedPane;

  /**
   * Comando sendo visualizado.
   */
  protected CommandInfo command;

  /**
   * Configurador do algoritmo executado pelo comando.
   */
  protected AlgorithmConfigurator configurator;

  /**
   * Componente dono da viso.
   */
  protected DesktopComponentFrame owner;

  /**
   * Tipo {@link CommandViewType} da viso.
   */
  protected CommandViewType viewType;

  /**
   * Painel principal da viso.
   */
  protected JPanel mainPanel;

  /**
   * Lista de abas estticas.
   */
  protected final Map<String, Tab> staticTabs;

  /**
   * Lista de abas recarregveis (de log).
   */
  protected final Map<String, ReloadableTab> reloadableTabs;

  /**
   * Nome da propriedade do {@link ConfigurationManager} que determina o
   * identificador de uma aba de log.
   */
  private static final String LOG_TAB_ID_PROPERTY_NAME = "tab.{0}.id";

  /**
   * Nome da propriedade do {@link ConfigurationManager} que determina o ttulo
   * de uma aba de log.
   */
  private static final String LOG_TAB_TITLE_PROPERTY_NAME = "tab.{0}.title.{1}";

  /**
   * Nome da propriedade do {@link ConfigurationManager} que determina o padro
   * de nome de arquivo a ser mostrado na aba de log.
   */
  private static final String LOG_TAB_FILE_PATTERN_PROPERTY_NAME =
    "tab.{0}.log.file.pattern.{1}";

  /**
   * Nome da propriedade do {@link ConfigurationManager} que determina se a aba
   * tem acesso restrito a alguns usurios.
   */
  private static final String LOG_TAB_RESTRICTED_PROPERTY_NAME =
    "tab.{0}.restricted";

  /**
   * Tag que pode ser utilizada no padro para adicionar um identificador no
   * padro de nome de arquivo de log.
   */
  private static final String ID_REPLACEMENT_TAG = "$ID";

  /**
   * Identificador que ir substituir o {@link #ID_REPLACEMENT_TAG} (se
   * existente) no padro.
   */
  protected String idForLogPattern;

  /**
   * Construtor.
   * 
   * @param viewType Tipo {@link CommandViewType} da viso.
   * @param configurator O configurador do algoritmo (No aceita {@code null}).
   * @param command O comando (No aceita {@code null}).
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  public AbstractCommandView(CommandViewType viewType, CommandInfo command,
    AlgorithmConfigurator configurator) throws ClientException {
    if (command == null) {
      throw new IllegalArgumentException("O parmetro command est nulo.");
    }
    if (configurator == null) {
      throw new IllegalArgumentException("O parmetro configurator est nulo.");
    }
    if (viewType == null) {
      this.viewType = CommandViewType.SIMPLE;
    }
    else {
      this.viewType = viewType;
    }
    this.idForLogPattern = null;
    this.command = command;
    this.configurator = configurator;
    this.reloadableTabs = new HashMap<String, ReloadableTab>();
    this.staticTabs = new HashMap<String, Tab>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getMainComponent(DesktopComponentFrame window)
    throws ClientException {
    if (mainPanel == null) {
      this.owner = window;
      initUI();
    }
    return mainPanel;
  }

  /**
   * Inicializa a viso do comando.
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  private void initUI() throws ClientException {
    mainPanel = new JPanel();
    mainPanel.setLayout(new BorderLayout());

    statusBar = new StatusBar();
    statusBar.showStatusBar();

    List<LogTabConfiguration> tabConfiguration;
    try {
      tabConfiguration = getLogTabsConfiguration();
    }
    catch (ConfigurationManagerException e) {
      throw new ClientException(getString("error.loading.tab.configuration"), e);
    }
    tabbedPane = new JTabbedPane();
    createTabs(tabConfiguration);
    tabbedPane.addChangeListener(new ChangeListener() {
      /**
       * {@inheritDoc}
       */
      @Override
      public void stateChanged(ChangeEvent e) {
        updateSelection();
      }
    });

    mainPanel.add(this.tabbedPane, BorderLayout.CENTER);

    JPanel southPane = new JPanel(new BorderLayout());
    southPane.add(statusBar, BorderLayout.SOUTH);

    mainPanel.add(southPane, BorderLayout.SOUTH);

    /** Atualiza a barra de estado com o estado inicial do comando. */
    updateStatusBar();
  }

  /**
   * Atualiza o estado de seleo das abas.
   */
  private void updateSelection() {
    Tab currrentSelection = selected;

    int index = tabbedPane.getSelectedIndex();
    String title = tabbedPane.getTitleAt(index);

    // Se uma aba diferente tiver sido selecionada, remove a seleo.
    if (currrentSelection != null
      && !currrentSelection.getTitle().equals(title)) {
      currrentSelection.setSelected(false);
      selected = null;
    }

    // Busca a nova aba selecionada no caso de no haver seleo prvia ou a 
    // seleo tiver sido diferente.
    if (selected == null) {
      selected = getTabByTitle(title);
    }

    // Se houver uma aba selecionada
    if (selected != null) {
      selected.setSelected(true);
      updateStatusBar();
    }
  }

  /**
   * Indica se essa viso possui abas de log (configurvel por sistema via
   * {@link ConfigurationManager}).
   * 
   * @return verdadeiro se a viso possui abas de log ou falso, caso contrrio.
   */
  private boolean hasLogsTab() {
    ConfigurationManager cnfManager = ConfigurationManager.getInstance();
    final boolean defaultValue = false;
    if (cnfManager == null) {
      return defaultValue;
    }
    try {
      Class<LogTabConfiguration> propClass = LogTabConfiguration.class;
      Configuration cnf = cnfManager.getConfiguration(propClass);
      String propName = "enabled";
      Boolean isEnabled =
        cnf.getOptionalBooleanProperty(propName, defaultValue);
      return isEnabled;
    }
    catch (Exception e) {
      return defaultValue;
    }
  }

  /**
   * Obtm o progresso da execuoo do comando.
   * 
   * @return o progresso da execuo do comando ou nulo caso no seja possvel
   *         obter essa informao.
   */
  protected ProgressData getCommandProgress() {
    return command.getProgressData();
  }

  /**
   * Cria no painel de abas, as abas de parmetro do configurador e as
   * recarregveis (de log).
   * 
   * @param tabConfiguration configurao das abas recarregveis.
   * 
   * @throws ClientException Erro ao criar as vises das abas.
   */
  private void createTabs(List<LogTabConfiguration> tabConfiguration)
    throws ClientException {

    tabbedPane.removeAll();
    Map<String, Tab> statTabs = createStaticTabs();
    if (statTabs != null) {
      for (Entry<String, Tab> entry : statTabs.entrySet()) {
        String title = entry.getKey();
        Tab tab = entry.getValue();
        if (tab != null) {
          tabbedPane.addTab(title, tab.getMainComponent());
        }
        this.staticTabs.put(title, tab);
      }
    }

    Map<String, ReloadableTab> relTabs = createReloadableTabs(tabConfiguration);
    if (relTabs != null) {
      for (LogTabConfiguration conf : tabConfiguration) {
        String title = conf.getTitle();
        ReloadableTab tab = relTabs.get(title);
        if (tab != null) {
          tabbedPane.addTab(title, tab.getMainComponent());
        }
        this.reloadableTabs.put(title, tab);
      }
    }

    Dimension tabsDimension = new Dimension(600, 300);
    mainPanel.setPreferredSize(tabsDimension);
  }

  /**
   * Cria as abas estticas da viso.
   * 
   * @return a lista de abas de logs.
   * @throws ClientException em caso de erro na criao da viso das abas.
   */
  protected Map<String, Tab> createStaticTabs() throws ClientException {
    Map<String, Tab> tabs = new HashMap<String, Tab>();
    Tab detailsTab = createDetailsTab();
    if (detailsTab != null) {
      tabs.put(detailsTab.getTitle(), detailsTab);
    }
    return tabs;
  }

  /**
   * Cria a aba de detalhes do comando.
   * 
   * @return a aba de detalhes.
   * @throws ClientException em caso de erro na criao da viso da aba.
   */
  protected Tab createDetailsTab() throws ClientException {
    String title = getString("tab.details.title");
    return new ParametersTab(configurator, title, owner);
  }

  /**
   * Retorna a string localizada a partir de uma chave.
   * 
   * @param key A chave.
   * @return A string localizada.
   */
  private String getString(final String key) {
    return LNG.getAnyOf(getClass().getSimpleName() + "." + key,
      AbstractCommandView.class.getSimpleName() + "." + key);
  }

  /**
   * Retorna a string localizada a partir de uma chave e seus parmetros.
   * 
   * @param key A chave.
   * @param keyArgs Os parmetros.
   * @return A string localizada.
   */
  private String getString(final String key, final Object... keyArgs) {
    return MessageFormat.format(getString(key), keyArgs);
  }

  /**
   * Cria as abas recarregveis da viso.
   * 
   * @param tabConfiguration configurao das abas recarregveis.
   * 
   * @return a lista de abas de logs.
   */
  protected Map<String, ReloadableTab> createReloadableTabs(
    List<LogTabConfiguration> tabConfiguration) {
    if (hasLogsTab()) {
      return createLogTabs(tabConfiguration);
    }
    else {
      return Collections.emptyMap();
    }
  }

  /**
   * Cria as abas de logs (configurveis por sistema).
   * 
   * @param tabConfiguration configurao das abas recarregveis.
   * 
   * @return a lista de abas de logs.
   */
  protected Map<String, ReloadableTab> createLogTabs(
    List<LogTabConfiguration> tabConfiguration) {
    Map<String, ReloadableTab> logTabs = new HashMap<String, ReloadableTab>();
    Map<LogTabConfiguration, Set<ClientProjectFile>> matchMap =
      getMatchingLogFiles(tabConfiguration);
    if (matchMap != null) {
      for (LogTabConfiguration conf : tabConfiguration) {
        String title = conf.getTitle();
        ReloadableTab logTab;
        Set<ClientProjectFile> matchingFiles = matchMap.get(conf);
        /*
         * Aba deve ser recarregvel automaticamente somente se o comando ainda
         * no estiver terminado.
         */
        boolean autoReloadTab = !isCommandFinished();
        if (matchingFiles == null || matchingFiles.size() == 0) {
          logTab = null;
        }
        else if (matchingFiles.size() > 1) {
          logTab =
            new ConsolidatedLogTab(title, matchingFiles, statusBar, owner,
              autoReloadTab);
        }
        else {
          logTab =
            new SimpleLogTab(title, matchingFiles.iterator().next(), statusBar,
              owner, autoReloadTab);
        }
        if (logTab != null) {
          logTabs.put(title, logTab);
        }
      }
    }
    return logTabs;
  }

  /**
   * Retorna a lista de configuraes das abas de log, determinando quais
   * arquivos devem aparecer em cada aba, de acordo com o que foi configurado
   * via {@link ConfigurationManager}.
   * 
   * @return A lista de configuraes.
   * @throws ConfigurationManagerException em caso de erro na leitura das
   *         configuraes.
   */
  protected List<LogTabConfiguration> getLogTabsConfiguration()
    throws ConfigurationManagerException {
    ConfigurationManager cnfManager = ConfigurationManager.getInstance();
    if (cnfManager == null) {
      return Collections.emptyList();
    }
    List<LogTabConfiguration> configurations =
      new ArrayList<LogTabConfiguration>();
    Class<LogTabConfiguration> propClass = LogTabConfiguration.class;
    Configuration cnf = cnfManager.getConfiguration(propClass);
    if (cnf == null) {
      return Collections.emptyList();
    }
    User loggedUser = User.getLoggedUser();
    boolean isAdmin = loggedUser.isAdmin();
    boolean continueReading = true;
    for (int i = 1; continueReading; i++) {
      String tabId =
        cnf.getOptionalProperty(MessageFormat.format(LOG_TAB_ID_PROPERTY_NAME,
          i));
      if (tabId != null) {

        // Ttulo especificado para a aba
        String title =
          cnf.getMandatoryProperty(MessageFormat.format(
            LOG_TAB_TITLE_PROPERTY_NAME, tabId, LNG.getLocale()));
        // Para cada aba, podem ser definidos vrios padres de nomenclatura
        List<String> filePatterns =
          cnf.getMandatoryListProperty(MessageFormat.format(
            LOG_TAB_FILE_PATTERN_PROPERTY_NAME, tabId,
            viewType.getPropertyName()));

        if (idForLogPattern != null) {
          List<String> replacedPatterns = new ArrayList<String>();
          for (String pattern : filePatterns) {
            String replacedPattern =
              pattern.replace(ID_REPLACEMENT_TAG, idForLogPattern);
            replacedPatterns.add(replacedPattern);
          }
          filePatterns = replacedPatterns;
        }

        boolean isRestricted =
          cnf.getOptionalBooleanProperty(
            MessageFormat.format(LOG_TAB_RESTRICTED_PROPERTY_NAME, tabId),
            false);

        boolean hasPermission;
        if (isRestricted && !isAdmin) {
          try {
            hasPermission =
              CommandViewPermission.checkCommandViewPermission(loggedUser,
                tabId);
          }
          catch (Exception e) {
            hasPermission = false;
            StandardDialogs.showErrorDialog(this.mainPanel.getParent(),
              getTitle(),
              getString("error.checking.permission", command.getId()));
          }
        }
        else {
          hasPermission = true;
        }

        if (hasPermission) {
          configurations.add(new LogTabConfiguration(tabId, title,
            filePatterns, isRestricted));
        }
      }
      else {
        continueReading = false;
      }
    }

    return configurations;
  }

  /**
   * Retorna uma mapa de arquivos de log do comando de acordo com a configurao
   * das abas.
   * 
   * @param tabConfiguration configurao das abas.
   * 
   * @return A lista de configuraes.
   */
  protected Map<LogTabConfiguration, Set<ClientProjectFile>> getMatchingLogFiles(
    List<LogTabConfiguration> tabConfiguration) {
    String[] logsDir = command.getPersistencyPath();
    MatchLogTabConfigurationsTask task =
      new MatchLogTabConfigurationsTask(logsDir, tabConfiguration);

    if (task.execute(owner, getTitle(), getString("message.loading.logs"))) {
      return task.getResult();
    }
    else {
      StandardDialogs.showErrorDialog(this.mainPanel.getParent(), getTitle(),
        getString("error.loading.logs", command.getId()));
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTitle() {
    return command.getId();
  }

  /**
   * Atualiza a barra de status com o estado do comando.
   */
  private void updateStatusBar() {
    String statusMessage;
    switch (command.getStatus()) {
      case SCHEDULED:
        statusMessage = getString("message.status.scheduled");
        break;
      case FINISHED:
        statusMessage = getFinalizationTypeDescription();
        break;
      case SYSTEM_FAILURE:
        statusMessage = getString("message.status.system_failure");
        break;
      case EXECUTING:
      case INIT:
      case UPLOADING:
      case DOWNLOADING:
        if (command.isValid()) {
          if (command.isQueued()) {
            statusMessage = getString("message.status.queued");
          }
          else {
            ProgressData data = getCommandProgress();
            if (data != null) {
              statusMessage = getString("message.status.running.progress",
                data.getDescription());
            }
            else {
              statusMessage = getString("message.status.running");
            }
          }
        }
        else {
          statusMessage = getString("message.status.disconnected");
        }
        break;
      default:
        statusMessage = getString("message.status.unknown");
        break;
    }
    statusBar.setStatus(statusMessage);
  }

  /**
   * Retorna a descrio do tipo de finalizao do comando.
   * 
   * @return A descrio do tipo de finalizao do comando.
   */
  protected String getFinalizationTypeDescription() {
    CommandFinalizationInfo finalizationInfo = command.getFinalizationInfo();
    CommandFinalizationType finalizationType =
      finalizationInfo.getFinalizationType();
    Integer exitCode = finalizationInfo.getExitCode();
    String description =
      getFinalizationTypeDescription(finalizationType, exitCode);
    if (command.hasWarnings()) {
      description += " " + getString("message.status.warning");
    }
    return description;
  }

  /**
   * Retorna a descrio textual do tipo de finalizao.
   * 
   * @param finalizationType Tipo de finalizao.
   * @param exitCode Cdigo de retorno.
   * @return a descrio.
   */
  protected String getFinalizationTypeDescription(
    CommandFinalizationType finalizationType, Integer exitCode) {
    switch (finalizationType) {
      case END:
        return getString("message.status.finished");
      case EXECUTION_ERROR:
        if (exitCode != null) {
          return getString("message.status.finished.error.code",
            new Object[] { exitCode });
        }
        else {
          return getString("message.status.finished.error");
        }
      case NO_EXIT_CODE:
        return (getString("message.status.finished.no_code"));
      case SUCCESS:
        return (getString("message.status.finished.success"));
      case FAILED:
        return (getString("message.status.finished.failed"));
      case KILLED:
        return (getString("message.status.finished.killed"));
      case LOST:
        return (getString("message.status.finished.lost"));
      default:
        return (getString("message.status.finished"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public StatusBar getStatusBar() {
    return this.statusBar;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void commandUpdated(Type type, CommandInfo cmd) {
    this.command = cmd;
    reload();
  }

  /**
   * Atualiza o contedo das abas recarregveis.
   */
  private void reload() {
    final boolean commandFinished = isCommandFinished();
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        try {
          Collection<ReloadableTab> tabs = reloadableTabs.values();
          for (ReloadableTab tab : tabs) {
            if (tab != null) {
              if (tab instanceof AutoReloadable) {
                if (commandFinished) {
                  LogPanelReloader reloader =
                    ((AutoReloadable) tab).getReloader();
                  reloader.stop();
                }
              }
              else {
                tab.reload();
              }
            }
          }
          updateStatusBar();
        }
        catch (Exception e) {
          statusBar.setError(getString("error.loading.logs", command.getId()));
        }
      }
    });
  }

  /**
   * Indica se o comando sendo visualizado j terminou sua execuo.
   * 
   * @return verdadeiro se o comando estiver terminado ou falso, caso contrrio.
   */
  protected boolean isCommandFinished() {
    return command.getStatus().equals(CommandStatus.FINISHED);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void selectTab(TabType tabType) throws ClientException {
    if (tabType != null) {
      Tab tab = null;
      switch (tabType) {
        case STATIC:
          if (!staticTabs.isEmpty()) {
            tab = getNextTab(staticTabs.values());
          }
          break;
        case RELOADABLE:
          if (!reloadableTabs.isEmpty()) {
            tab = getNextTab(reloadableTabs.values());
          }
          break;
      }
      if (tab != null) {
        tabbedPane.setSelectedComponent(tab.getMainComponent());
        updateSelection();
      }
      else {
        selectPreferredTab();
      }
      selectPreferredTab();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void selectTab(String title) throws ClientException {
    if (title != null) {
      Tab tab = getTabByTitle(title);
      if (tab != null) {
        tabbedPane.setSelectedComponent(tab.getMainComponent());
        updateSelection();
      }
      else {
        selectPreferredTab();
      }
    }
    else {
      selectPreferredTab();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void selectPreferredTab() throws ClientException {
    Tab tab = null;
    if (!staticTabs.isEmpty()) {
      tab = getNextTab(staticTabs.values());
    }
    else if (!reloadableTabs.isEmpty()) {
      tab = getNextTab(reloadableTabs.values());
    }
    if (tab != null) {
      tabbedPane.setSelectedComponent(tab.getMainComponent());
      updateSelection();
    }
    // se tab for nulo, no tem nenhuma aba a selecionar.
  }

  /**
   * Retorna a primeira aba no nula da coleo de abas.
   * 
   * @param collection a coleo de abas.
   * @return a primeira aba no nula encontrada.
   */
  private Tab getNextTab(Collection<? extends Tab> collection) {
    Tab chosenTab = null;
    for (Tab tab : collection) {
      if (tab != null) {
        chosenTab = tab;
        break;
      }
    }
    return chosenTab;
  }

  /**
   * Retorna a aba da viso que tem o ttulo especificado.
   * 
   * @param title o ttulo.
   * @return a aba com o ttulo especificado ou nulo, caso no exista aba com o
   *         ttulo.
   */
  private Tab getTabByTitle(String title) {
    Tab tab = staticTabs.get(title);
    if (tab == null) {
      tab = reloadableTabs.get(title);
    }
    return tab;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Tab getSelected() {
    return selected;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() {
    if (selected != null) {
      selected.setSelected(false);
    }
    statusBar.shutdownTimer();
  }

}
