package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.swing.Icon;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy.AlgorithmOperation;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;

/**
 * Painel com as informaes das categorias do algoritmo selecionado para
 * edio.
 * 
 */
public class AlgorithmCategoryInfoPanel extends CommonInfoEditPanel {
  /** Painel com as informaes estticas do algoritmo */
  private AlgorithmStaticInfoPanel algorithmHeaderPanel;

  /**
   * Painel de associao de algoritmos a categorias
   */
  private AlgorithmsAndCategoriesBindPanel bindPanel;

  /** Painel principal */
  private JPanel mainPanel;

  /** Indica se o painel est com informaes sendo alteradas */
  private boolean isEditing;

  /**
   * Classe que representa o dado de categoria a ser gerenciado no painel de
   * associaes de algoritmos e categorias.
   * 
   */
  private class CategoryData implements DataInterface, Comparable<CategoryData> {
    /** Informaes da categoria */
    private Category category;

    /**
     * Construtor.
     * 
     * @param category informaes da categoria
     */
    CategoryData(Category category) {
      this.category = category;
    }

    /**
     * Obtm a categoria.
     * 
     * @return a categoria
     */
    public Category getCategory() {
      return category;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getName() {
      return category.getFullName();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getId() {
      return category.getId();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Icon getIcon() {
      return ApplicationImages.ICON_FOLDER_16;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getFullName() {
      return getName();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int compareTo(CategoryData categoryData) {
      return category.getId().compareToIgnoreCase(categoryData.getId());
    }
  }

  /**
   * Constri o painel com as informaes de categorias do algoritmo.
   * 
   * @param algoPane painel com os dados de atualizao de um algoritmo
   */
  public AlgorithmCategoryInfoPanel(AlgorithmEditDataPanel algoPane) {
    super(algoPane);
  }

  /**
   * Obtm o painel global de edio de dados de algoritmo.
   * 
   * @return o painel global de edio de dados de algoritmo
   */
  private AlgorithmEditDataPanel getAlgorithmEditDataPanel() {
    return (AlgorithmEditDataPanel) getEditPanel();
  }

  /**
   * Salva efetivamente as informaes no servidor.
   * 
   * @return retorna true se as informaes foram salvas com sucesso, caso
   *         contrrio, retorna false
   */
  protected boolean saveInfo() {
    bindPanel.bindData();
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initializeData() {
    if (getSelectedAlgorithm() == null) {
      return;
    }
    if (!isEditing) {
      algorithmHeaderPanel.setSelectedAlgorithm(getSelectedAlgorithm());
      bindPanel.initializeData(getSelectedAlgorithm(),
        getAvailableCategories(), getAlgorithmCategories());
      revalidate();
    }
    changeOperationsState(isEditing);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel buildMainInfoPanel() {
    if (mainPanel == null) {
      mainPanel = new JPanel(new GridBagLayout());
      algorithmHeaderPanel = new AlgorithmStaticInfoPanel(getApplication());

      bindPanel = getDataBindingPanel();
      mainPanel.add(algorithmHeaderPanel, new GBC(0, 0).horizontal().west()
        .insets(0, 0, 0, 0));
      mainPanel.add(bindPanel, new GBC(0, 1).both().west().insets(0, 0, 0, 0));
    }
    return mainPanel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void apply() {
    isEditing = false;
    if (!saveInfo()) {
      return;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void cancel() {
    if (confirmCancelling()) {
      initializeData();
    }
  }

  /**
   * Confirma com o usurio se a operao de cancelamento deve ser efetivada, j
   * que perder todas as alteraes feitas sobre o dado.
   * 
   * @return retorna true, se a alterao do dado deve ser cancelada, caso
   *         contrrio, retorna false
   */
  private boolean confirmCancelling() {
    int confirm =
      getApplication().showOptionDialog(
        getString("AlgorithmCategoryInfoPanel.msg.cancel.confirm"),
        new String[] {
            getString("AlgorithmCategoryInfoPanel.msg.cancel.confirm.yes"),
            getString("AlgorithmCategoryInfoPanel.msg.cancel.confirm.no") });
    if (confirm == 0) {
      isEditing = false;
      return true;
    }
    return false;
  }

  /**
   * Obtm o painel que realiza a associao dos dados.
   * 
   * @return o painel
   */
  private AlgorithmsAndCategoriesBindPanel getDataBindingPanel() {
    if (bindPanel == null) {
      bindPanel =
        new AlgorithmsAndCategoriesBindPanel(getApplication(),
          getSelectedAlgorithm(), getAvailableCategories(),
          getAlgorithmCategories(),
          AlgorithmsAndCategoriesBindPanel.ItemType.CATEGORY_ITEM);
      bindPanel.addBindListener(wasModified -> {
        isEditing = wasModified;
        changeOperationsState(wasModified);
      });
    }
    return bindPanel;
  }

  /**
   * Obtm os filhos disponveis para associao com os ns selecionados. Deve
   * incluir todos os filhos existentes que ainda no estejam associados.
   * 
   * @return os filhos disponveis para a associao
   */
  private SortedSet<DataInterface> getAvailableCategories() {
    SortedSet<DataInterface> availableData = new TreeSet<>();
    availableData = getAllCategories();

    if (getSelectedAlgorithm() == null) {
      return availableData;
    }

    // Por enquanto, tratar o caso de um nico n pai selecionado
    SortedSet<Category> categorySortedSet =
      getAlgorithmCategories(getSelectedAlgorithm().getItem());
    if (categorySortedSet == null) {
      return availableData;
    }

    // ** Exclui da lista de disponveis os filhos que j pertecem ao pai
    // selecionado

    Iterator<DataInterface> availableIterator = availableData.iterator();

    while (availableIterator.hasNext()) {
      CategoryData item = (CategoryData) availableIterator.next();
      if (categorySortedSet.contains(item.getCategory())) {
        availableIterator.remove();
      }
    }
    return availableData;
  }

  /**
   * Obter todas as categorias disponveis no servidor, inclusive as
   * subcategorias.
   * 
   * @return as categorias disponveis no servidor, inclusive as subcategorias
   */
  private SortedSet<DataInterface> getAllCategories() {
    SortedSet<Category> allCategoryData =
      getApplication().getAllCategories(true);
    return getCategoryDataInterfaceSet(allCategoryData);
  }

  /**
   * Constri, a partir das informaes dos algorimtos, um conjunto de dados com
   * informao comum que o painel de associaes requer.
   * 
   * @param categories conjunto de algoritmos
   * @return um conjunto de dados com as informaes dos algoritmos
   */
  private SortedSet<DataInterface> getCategoryDataInterfaceSet(
    SortedSet<Category> categories) {
    SortedSet<DataInterface> categoryDataSet = new TreeSet<>();
    if (categories != null) {
      for (Category category : categories) {
        categoryDataSet.add(new CategoryData(category));
      }
    }
    return categoryDataSet;
  }

  /**
   * Obtm os filhos previamente associados com os ns selecionados. Deve
   * incluir todos os filhos que j estejam associados.
   * 
   * @return os filhos previamente associados com os ns
   */
  private SortedSet<DataInterface> getAlgorithmCategories() {
    SortedSet<DataInterface> selectedData = new TreeSet<>();
    if (getSelectedAlgorithm() != null) {
      SortedSet<Category> categories =
        getAlgorithmCategories(getSelectedAlgorithm().getItem());
      selectedData = getCategoryDataInterfaceSet(categories);
    }
    return selectedData;
  }

  /**
   * Obtm as categorias de um algoritmo, cujo usurio tem permisso de
   * gerenciamento sobre o algoritmo nas categorias.
   * 
   * @param algoInfo informaes do algoritmo procurado
   * @return retorna o conjunto ordenado de categorias de um algoritmo, cujo
   *         usurio tem permisso de gerenciamento
   */
  private SortedSet<Category> getAlgorithmCategories(AlgorithmInfo algoInfo) {
    SortedSet<Category> categorySortedSet =
      AlgorithmManagementProxy.getAlgorithmCategories(getApplication()
        .getApplicationFrame(), AlgorithmOperation.ADMIN_ALGORITHM,
        getSelectedAlgorithm().getItem());
    return categorySortedSet;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDataChanged() {
    isEditing = false;
    changeOperationsState(isEditing);
  }

  /**
   * Obtm o algoritmo selecionado para edio.
   * 
   * @return o algoritmo selecionado para edio
   */
  private AlgorithmListItem getSelectedAlgorithm() {
    return getAlgorithmEditDataPanel().getSelectedAlgorithm();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean wasModified() {
    if (bindPanel.wasModified()) {
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getTitle() {
    return getString("AlgorithmCategoryInfoPanel.tab.category");
  }
}
