package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.DefaultComboBoxModel;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;

import tecgraf.javautils.gui.GBC;
import csbase.client.algorithms.tasks.OutdatedFlowInfo;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.remote.ClientRemoteLocator;

/**
 * Dilogo de relatrio de fluxos desatualizados
 *
 *
 * @author Tecgraf/PUC-Rio
 */
public class OutdatedFlowDialog extends JDialog {

  /**
   * O identificar de mensagens enviadas com configuradores para o aplicativo.
   */
  public static final String CONFIGURATOR_MESSAGE = "CONFIGURATOR_MESSAGE";

  /** Painel na janela principal para sincronia */
  AlgorithmSelectionPanel selectionPanel;

  /** A aplicao */
  Application app;

  /** Lista de fluxos */
  JList<OutdatedFlowInfo> jList = new JList<>();

  /** rea de detalhes */
  JTextArea infoArea = new JTextArea();

  /** Boto de fechar a janela */
  JButton closeButton = new JButton();

  /** Boto de sincronizar listas */
  JButton sincButton = new JButton();

  /** Boto de abrir editor de fluxos */
  JButton flowEditorButton = new JButton();

  /**
   * Cosntrutor
   *
   * @param outdatedFlows Lista de fluxos a serem mostrados
   * @param algorithmSelectionPanel O painel na janela principal
   */
  public OutdatedFlowDialog(Vector<OutdatedFlowInfo> outdatedFlows,
    AlgorithmSelectionPanel algorithmSelectionPanel) {
    super(algorithmSelectionPanel.getApplication().getApplicationFrame());

    addEscListener();

    this.selectionPanel = algorithmSelectionPanel;
    this.app = algorithmSelectionPanel.getApplication();
    Class<?> myClass = this.getClass();

    jList.setModel(new DefaultComboBoxModel<>(outdatedFlows));
    infoArea.setEditable(false);

    JScrollPane listScroll = new JScrollPane(jList);
    JScrollPane infoScroll = new JScrollPane(infoArea);

    setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    closeButton.setText(app.getClassString(myClass, "close.button"));
    closeButton.addActionListener(e -> closeDialog());

    sincButton.setIcon(ApplicationImages.ICON_LINK_16);
    sincButton.setToolTipText(app
      .getClassString(myClass, "sinc.button.tooltip"));
    sincButton.addActionListener(e -> sincSelection());

    flowEditorButton.setIcon(ApplicationImages.ICON_FLOW_16);
    flowEditorButton.setToolTipText(app.getClassString(myClass,
      "flow.button.tooltip"));
    flowEditorButton.addActionListener(e -> openFlowEditor());

    JToolBar toolbar = new JToolBar(JToolBar.VERTICAL);
    toolbar.setFloatable(false);
    toolbar.add(sincButton);
    toolbar.add(flowEditorButton);

    JPanel listPanel = new JPanel(new GridBagLayout());
    listPanel.add(listScroll, new GBC(0, 0).insets(0, 0, 0, 6).both().west());
    listPanel.add(toolbar, new GBC(1, 0).insets(0, 6, 0, 0).vertical()
      .northwest());

    this.setLayout(new GridBagLayout());
    add(new JLabel(app.getClassString(myClass, "outdated.flow.label")),
      new GBC(0, 0).insets(12, 12, 6, 12).horizontal());
    add(listPanel, new GBC(0, 1).insets(6, 12, 12, 12).both().weighty(3));

    add(new JLabel(app.getClassString(myClass, "detail.label")), new GBC(0, 2)
    .insets(12, 12, 6, 12).horizontal());
    add(infoScroll, new GBC(0, 3).insets(6, 12, 6, 12).both().weighty(1));
    add(closeButton, new GBC(0, 4).center().none().insets(6, 12, 12, 12)
      .weighty(1));
    jList.setPreferredSize(new Dimension(400, 400));
    infoArea.setPreferredSize(new Dimension(450, 100));
    Font font = infoArea.getFont();
    infoArea.setFont(new Font("Courier", font.getStyle(), font.getSize()));
    this.setTitle(app.getClassString(myClass, "title"));

    this.pack();
    this.setLocationRelativeTo(app.getApplicationFrame());

    jList.addListSelectionListener(e -> updateInfoArea());
    jList.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent evt) {
        if (evt.getClickCount() == 2) {
          sincSelection();
        }
      }
    });

    if (outdatedFlows.size() > 0) {
      jList.setSelectedIndex(0);
    }
  }

  /**
   * Sincroniza a seleo com a seleo da janela principal
   */
  private void sincSelection() {
    OutdatedFlowInfo info = jList.getSelectedValue();
    if (info == null) {
      return;
    }
    if (selectionPanel != null) {
      selectionPanel.selectAlgorithm(info.getFlow());
    }
  }

  /**
   * Abre o editor de fluxo
   */
  private void openFlowEditor() {

    ApplicationManager am = ApplicationManager.getInstance();

    RemoteTask<FlowAlgorithmConfigurator> task =
      new RemoteTask<FlowAlgorithmConfigurator>() {
      @Override
      protected void performTask() throws Exception {
        OutdatedFlowInfo info = jList.getSelectedValue();
        FlowAlgorithmConfigurator configurator =
          (FlowAlgorithmConfigurator) ClientRemoteLocator.algorithmService
          .createAlgorithmConfigurator(info.getFlow().getName(), info
            .getFlow().getLastVersion().getId());
        setResult(configurator);
      }
    };

    ApplicationFrame window = app.getApplicationFrame();
    String title = DesktopFrame.getInstance().getTitle();
    String message =
      app.getClassString(this.getClass(), "creating.configurator");

    try {
      task.execute(window, title, message);
    }
    catch (Exception e) {
      JOptionPane.showMessageDialog(window, app.getClassString(this.getClass(),
        "error.creating.configurator"), "error.title",
        JOptionPane.PLAIN_MESSAGE);
      return;
    }

    FlowAlgorithmConfigurator configurator = task.getResult();

    try {
      ApplicationType app = am.runApplication("flowapplication");
      app.sendMessage(CONFIGURATOR_MESSAGE, configurator, app.getName());
    }
    catch (Exception exception) {
      JOptionPane.showMessageDialog(window, app.getClassString(this.getClass(),
        "error.opening.editor"), "error.title", JOptionPane.PLAIN_MESSAGE);
      return;
    }
  }

  /**
   * fecha o dilogo
   */
  protected void closeDialog() {
    this.dispose();
  }

  /**
   * Atualiza as informaes do fluxo selecionado
   */
  protected void updateInfoArea() {
    OutdatedFlowInfo flow = jList.getSelectedValue();
    if (flow == null) {
      infoArea.setText("");
      return;
    }

    if (flow.getError() != null) {
      infoArea.setText(flow.getError().getMessage());
    }
    else {
      String text = "";

      for (int i = 0; i < flow.getLastVersionNodes().size(); i++) {
        AlgorithmInfo node = flow.getLastVersionNodes().get(i);
        AlgorithmVersionInfo outdatedNode =
          flow.getOutdatedVersionNodes().get(i);
        text +=
          node.getName()
            + "  "
            + app.getClassString(this.getClass(), "can.be.updated",
              new String[] { outdatedNode.toString(),
                  node.getLastVersion().toString() }) + "\n";
      }
      infoArea.setText(text);
    }
  }

  /**
   * Define o painel da janela principal
   * 
   * @param selectionPanel o painel de seleo.
   */
  public void setSelectionPanel(AlgorithmSelectionPanel selectionPanel) {
    this.selectionPanel = selectionPanel;
  }

  /**
   * Define o contedo do dilogo
   * 
   * @param outdatedFlows Lista de fluxos
   * @param selectionPanel O painel da janela principal
   */
  public void setData(Vector<OutdatedFlowInfo> outdatedFlows,
    AlgorithmSelectionPanel selectionPanel) {
    this.selectionPanel = selectionPanel;
    this.jList.setModel(new DefaultComboBoxModel<>(
            outdatedFlows));

  }

  /**
   * Ajusta a tecla esc para fechar o dilogo.
   */
  private void addEscListener() {
    final AbstractAction cancelAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        closeDialog();
      }
    };

    final int mode = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    final InputMap inputMap = rootPane.getInputMap(mode);
    final ActionMap actionMap = rootPane.getActionMap();
    final int esc = KeyEvent.VK_ESCAPE;
    final KeyStroke cancelStroke = KeyStroke.getKeyStroke(esc, 0);
    final String actionMapKey = cancelStroke.toString();
    inputMap.put(cancelStroke, actionMapKey);
    actionMap.put(actionMapKey, cancelAction);
  }
}
