package csbase.client.applications.flowapplication.graph;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Iterator;
import java.util.Set;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import csbase.client.desktop.DesktopComponentDialog;
import csbase.exception.algorithms.ParameterNotFoundException;
import tecgraf.javautils.core.lng.LNG;

/**
 * Visualizador de Parmetros de N de Verso de Algoritmo.
 * 
 * @author lmoreira
 * 
 */
public final class NodeParameterViewer {
  /**
   * O dalogo utilizado neste visualizador.
   */
  private DesktopComponentDialog dialog;

  /**
   * O n.
   */
  private GraphNode node;

  /**
   * Cria o visualizador.
   * 
   * @param node O n (No aceita {@code null}).
   */
  public NodeParameterViewer(final GraphNode node) {
    setNode(node);
  }

  /**
   * Executa o visualizador.
   */
  public void execute() {
    close();
    buildGui();
  }

  /**
   * Ajusta a largura da tabela e das colunas para que a tabela exiba o contedo
   * de cada clula por completo.
   * 
   * @param table A tabela.
   */
  private void adjustTableAndColumnWidths(final JTable table) {
    final int tableWidth = computeTableWidth(table);
    final int tableHeight = table.getPreferredSize().height;
    table.setPreferredScrollableViewportSize(new Dimension(tableWidth,
      tableHeight));
  }

  /**
   * Constri a GUI.
   */
  private void buildGui() {
    final Window window = this.node.getGraph().getParentWindow();
    this.dialog = new DesktopComponentDialog(window);
    this.dialog.setTitle(createTitle());
    final JButton closeButton = new JButton(getString("closeButton"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        close();
      }
    });

    final JPanel buttonPanel = new JPanel();
    buttonPanel.add(closeButton);
    this.dialog.getRootPane().setDefaultButton(closeButton);
    final JPanel contentPane = new JPanel(new BorderLayout());
    contentPane.add(new JScrollPane(createTable()), BorderLayout.CENTER);
    contentPane.add(buttonPanel, BorderLayout.SOUTH);
    this.dialog.setContentPane(contentPane);
    this.dialog.pack();
    this.dialog.center(window);
    this.dialog.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(final WindowEvent e) {
        close();
      }
    });
    this.dialog.setModal(false);
    this.dialog.setVisible(true);
    this.dialog.getRootPane().getDefaultButton().requestFocus();
  }

  /**
   * Fecha o dilogo.
   */
  private void close() {
    if (this.dialog != null) {
      this.dialog.setVisible(false);
      this.dialog.dispose();
      this.dialog = null;
    }
  }

  /**
   * Calcula a largura do contedo de uma coluna (todas as clulas exceto o
   * cabealho).
   * 
   * @param table A tabela (No aceita {@code null}).
   * @param column A coluna (No aceita {@code null}).
   * 
   * @return A largura.
   */
  private int computeColumnContentWidth(final JTable table,
    final TableColumn column) {
    final int columnIndex = column.getModelIndex();
    final TableModel model = table.getModel();
    final Class<?> columnClass = model.getColumnClass(columnIndex);
    final TableCellRenderer renderer = table.getDefaultRenderer(columnClass);
    int maxCellWidth = 0;
    for (int rowIndex = 0; rowIndex < table.getRowCount(); rowIndex++) {
      final Object value = table.getValueAt(rowIndex, columnIndex);
      final Component component = renderer.getTableCellRendererComponent(table,
        value, false, false, rowIndex, columnIndex);
      final int cellWidth = component.getPreferredSize().width;
      maxCellWidth = Math.max(maxCellWidth, cellWidth);
    }
    return maxCellWidth;
  }

  /**
   * Calcula a largura do cabealho de uma coluna.
   * 
   * @param table A tabela (No aceita {@code null}).
   * @param column A coluna (No aceita {@code null}).
   * 
   * @return A largura.
   */
  private int computeColumnHeaderWidth(final JTable table,
    final TableColumn column) {
    final JTableHeader tableHeader = table.getTableHeader();
    final Object headerValue = column.getHeaderValue();
    final TableCellRenderer headerRenderer = tableHeader.getDefaultRenderer();
    final Component component = headerRenderer.getTableCellRendererComponent(
      null, headerValue, false, false, 0, column.getModelIndex());
    final int headerWidth = component.getPreferredSize().width;
    return headerWidth;
  }

  /**
   * Calcula a largura da coluna.
   * 
   * @param table A tabela (No aceita {@code null}).
   * @param column A coluna (No aceita {@code null}).
   * 
   * @return A largura da coluna.
   */
  private int computeColumnWidth(final JTable table, final TableColumn column) {
    final int columnHeaderWidth = computeColumnHeaderWidth(table, column);
    final int columnContentWidth = computeColumnContentWidth(table, column);
    final int columnWidth = Math.max(columnHeaderWidth, columnContentWidth);
    return columnWidth;
  }

  /**
   * Calcula a largura da tabela.
   * 
   * @param table A tabela (No aceita {@code null}).
   * 
   * @return A largura da tabela.
   */
  private int computeTableWidth(final JTable table) {
    int tableWidth = 0;
    tableWidth += table.getIntercellSpacing().width;
    int intercellWidth = 0;
    final TableColumnModel columnModel = table.getColumnModel();
    for (int columnIndex = 0; columnIndex < table
      .getColumnCount(); columnIndex++) {
      final TableColumn column = columnModel.getColumn(columnIndex);
      final int columnWidth = computeColumnWidth(table, column);
      tableWidth += intercellWidth;
      tableWidth += columnWidth;
      column.setPreferredWidth(columnWidth);
      intercellWidth = table.getIntercellSpacing().width;
    }
    tableWidth += table.getIntercellSpacing().width;
    return tableWidth;
  }

  /**
   * Cria a tabela de parmetros.
   * 
   * @return a tabela.
   */
  private JTable createTable() {
    final JTable table = new JTable(createTableModel());
    adjustTableAndColumnWidths(table);
    return table;
  }

  /**
   * Cria o modelo da tabela.
   * 
   * @return o modelo da tabela.
   */
  private TableModel createTableModel() {
    return new AbstractTableModel() {
      /**
       * A quantidade de colunas.
       */
      private final int COLUMN_COUNT = 3;
      /**
       * Os nomes das colunas.
       */
      private String[] columnNames;
      /**
       * Os valores da tabela.
       */
      private String[][] values;

      {
        this.columnNames = new String[] { getString("labelColumn"), getString(
          "valueColumn"), getString("nameColumn") };
        final Set<String> parameterNames = node.getParameterNames();
        this.values = new String[parameterNames.size()][COLUMN_COUNT];
        final Iterator<String> parameterNameIterator = parameterNames
          .iterator();
        try {
          for (int i = 0; i < this.values.length; i++) {
            final String parameterName = parameterNameIterator.next();
            final String parameterLabel = node.getParameterLabel(parameterName);
            this.values[i][0] = parameterLabel;
            final String parameterValue = node.getParameterValue(parameterName);
            final String value = (parameterValue == null ? "" : parameterValue);
            this.values[i][1] = value;
            this.values[i][2] = parameterName;
          }
        }
        catch (final ParameterNotFoundException e) {
          final IllegalStateException illegalStateException =
            new IllegalStateException();
          illegalStateException.initCause(e);
          throw illegalStateException;
        }
      }

      @Override
      public int getColumnCount() {
        return COLUMN_COUNT;
      }

      @Override
      public String getColumnName(final int columnIndex) {
        return this.columnNames[columnIndex];
      }

      @Override
      public int getRowCount() {
        return this.values.length;
      }

      @Override
      public String getValueAt(final int rowIndex, final int columnIndex) {
        return this.values[rowIndex][columnIndex];
      }

      @Override
      public boolean isCellEditable(final int rowIndex, final int columnIndex) {
        return false;
      }
    };
  }

  /**
   * Cria o ttulo do dilogo.
   * 
   * @return o ttulo.
   */
  private String createTitle() {
    String title = getString("title");
    String label = this.node.getLabel();
    if (label != null) {
      title = String.format(title, label + " [" + this.node.getAlgorithmName()
        + "]", this.node.getAlgorithmVersionId());
    }
    else {
      title = String.format(title, this.node.getAlgorithmName(), this.node
        .getAlgorithmVersionId());
    }
    return title;
  }

  /**
   * Obtm uma mensagem no {@link LNG mecanismo de internacionalizao}.
   * 
   * @param keySuffix O sufixo da chave. A chave  o nome da classe acrescido de
   *        '.' e o sufixo.
   * 
   * @return A mensagem.
   */
  private String getString(final String keySuffix) {
    return LNG.get(getClass().getName() + "." + keySuffix);
  }

  /**
   * Atribui o n a este visualizador.
   * 
   * @param node O n (No aceita {@code null}).
   */
  private void setNode(final GraphNode node) {
    if (node == null) {
      throw new IllegalArgumentException("O parmetro node est nulo.");
    }
    this.node = node;
  }
}