package csbase.client.applications.flowapplication.zoom;

import javax.swing.DefaultBoundedRangeModel;

/**
 * <p>
 * Modelo de zoom para slider.
 * </p>
 * 
 * <p>
 * Classe que adequa a modelos de zoom para ser usados como modelos para
 * sliders.
 * </p>
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ZoomSliderModel extends DefaultBoundedRangeModel {
  /**
   * Modelo.
   */
  private final ZoomModel zoomModel;

  /**
   * Cria um modelo de zoom para slider.
   * 
   * @param model O modelo de zoom.
   */
  public ZoomSliderModel(final ZoomModel model) {
    super(toSliderValue(model.getValue()), 0,
      toSliderValue(model.getMinValue()), toSliderValue(model.getMaxValue()));
    this.zoomModel = model;
    this.zoomModel.addListener(new ZoomSliderModelListener());
  }

  /**
   * Obtm o major tick usado no slider.
   * 
   * @return O valor do major tick.
   */
  public int getMajorTicks() {
    return 5 * getMinorTicks();
  }

  /**
   * Obtm o minor tick usado no slider.
   * 
   * @return O valor do minor tick.
   */
  public int getMinorTicks() {
    return toSliderValue(this.zoomModel.getBlockIncrement());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getValue() {
    return toSliderValue(this.zoomModel.getValue());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValue(final int sliderValue) {
    this.zoomModel.setValue(fromSliderValue(sliderValue));
  }

  /**
   * Converte o valor vindo do slider para o valor utilizado no modelo de zoom.
   * 
   * @param sliderValue O valor do slider.
   * 
   * @return O valor convertido.
   */
  private static double fromSliderValue(final int sliderValue) {
    return sliderValue / 100.0;
  }

  /**
   * Converte o valor do modelo de zoom para o valor utilizado no slider.
   * 
   * @param zoomValue O valor de zoom.
   * 
   * @return O valor convertido.
   */
  private static int toSliderValue(final double zoomValue) {
    return (int) Math.round(zoomValue * 100.0);
  }

  /**
   * O listener de zoom que comunica ao modelo de zoom para sliders sobre
   * mudanas de zoom.
   */
  private class ZoomSliderModelListener implements ZoomListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void wasChanged(final ZoomModel model) {
      fireStateChanged();
    }
  }
}
