package csbase.client.preferences.types;

import java.awt.Color;

import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.editors.PVColorEditor;
import csbase.client.preferences.util.PreferenceBundle;

/**
 * Preferncia que encapsula um objeto do tipo {@link Color}. Este valor deve
 * ser usado da seguinte forma:<br>
 * 
 * \@Value(type=PVColor.class, defaultValue="10, 20, 15")
 * 
 * No campo 'defaultValue' devemos informar os inteiros separados por virgula
 * que correspondem aos valores RGB.
 * 
 * @see PreferenceValue PreferenceCategory PreferencePolicy
 * 
 * @author Tecgraf
 */
public class PVColor extends PreferenceValue<Color> {

  /**
   * Construtor usado na instanciao desta classe por reflexo.
   * 
   * @param name nome da preferncia que possui este valor.
   * @param value valor da preferncia.
   * @param defaultValue valor default da preferncia.
   * @param policy poltica de visibilidade do valor.
   * @param preferenceBundle objeto responsvel pela internacionalizao.
   */
  public PVColor(PreferenceDefinition name, String value, String defaultValue,
    PreferencePolicy policy, PreferenceBundle preferenceBundle) {
    super(name, value, defaultValue, policy, preferenceBundle);

    setPreferenceEditorClass(PVColorEditor.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getClassName() {
    return getClass().getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Color toValue(String value) {
    if ("".equals(value)) {
      return Color.GRAY;
    }

    String[] values = value.split(" *, *");

    int r = Integer.parseInt(values[0].trim());
    int g = Integer.parseInt(values[1].trim());
    int b = Integer.parseInt(values[2].trim());

    return new Color(r, g, b);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    builder.append(getValue().getRed());
    builder.append(",");
    builder.append(getValue().getGreen());
    builder.append(",");
    builder.append(getValue().getBlue());
    return builder.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PreferenceValue<Color> clone() {
    return new PVColor(name, this.toString(), defaultValue, policy,
      preferenceBundle);
  }
}
