/*
 * Detalhes da ltima alterao:
 *
 * $Author: tatimf $ $Date: 2016-08-17 13:59:27 -0300 (Wed, 17 Aug 2016) $
 * $Revision: 175498 $
 */
package csbase.client.remote;

import java.lang.reflect.InvocationTargetException;
import java.rmi.RemoteException;
import java.util.Locale;

import csbase.client.ClientServerManager;
import csbase.client.login.ReloginDialog;
import csbase.client.openbus.OpenBusAuthenticator;
import csbase.exception.CSBaseException;
import csbase.logic.EncryptedPassword;
import csbase.logic.LoginInfo;
import csbase.logic.ServerURI;
import csbase.logic.Session;
import csbase.logic.SingleServerManager;
import csbase.logic.User;
import csbase.logic.openbus.OpenBusLoginToken;
import csbase.remote.Authenticator;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.PreLoginAuthenticator;
import csbase.remote.UserPasswordAuthenticator;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * Representa uma classe que ser usada por um cliente para monitorar o estado
 * da conexo com um servidor.
 *
 * @author Tecgraf/PUC-Rio
 */
public class ClientRemoteMonitor extends SingleServerManager {
  /** A instncia que monitora a comunicao cliente/servidor. */
  private static ClientRemoteMonitor instance;

  /** Indica se o mtodo de autenticao do servidor  local ou no. */
  private boolean localLogin;

  /** Indica se o servidor aceita troca de senhas. */
  private boolean passwordChangeAllowed;

  /** Informaes de login do usurio logado */
  private LoginInfo loginInfo;

  /**
   * O autenticador do usurio no servidor.
   */
  private Authenticator authenticator;

  /**
   * Busca o autenticador. Consulta o atributo (ver documentao de
   * {@link #authenticator}).
   *
   * @return o atributo.
   */
  public final Authenticator getAuthenticator() {
    return authenticator;
  }

  /**
   * Cria um monitorador remoto de servidor que ser usado por um cliente.
   *
   * @param appName O nome da aplicao (sistema) que est iniciando o
   *        monitoramento.
   * @param serverURI A String que representa a URI CSBase do servidor
   * @param locator A classe com o localizador de servios. Ser a classe
   *        {@link ClientRemoteLocator} ou um de seus descendentes.
   */
  public ClientRemoteMonitor(String appName, String serverURI,
    Class<?> locator) {
    super(ServerURI.parse(serverURI), locator,
      ClientServerManager.getWindowSize());
    instance = this;
  }

  /**
   * @return A Instncia nica desta classe
   */
  public static final ClientRemoteMonitor getInstance() {
    return instance;
  }

  /**
   * Mtodo de depurao (stderr)
   *
   * @param message mensagem
   */
  protected final void warning(String message) {
    System.err.println(message);
  }

  /**
   * Mtodo de depurao.
   *
   * @param t exceo
   */
  protected final void error(Throwable t) {
    t.printStackTrace();
  }

  /**
   * Solicita login e senha atravs de um dilogo.
   *
   * @return Instncia de {@link LoginInfo} contendo as informaes do usurio
   *
   */
  private LoginInfo askLogin() {
    final class AskLoginTask implements Runnable {
      LoginInfo info;

      @Override
      public void run() {
        ReloginDialog dialog = new ReloginDialog(getReloginImagePath());
        dialog.setVisible(true);
        info = dialog.getLoginInfo();
      }

      public LoginInfo getLoginInfo() {
        return info;
      }
    }

    AskLoginTask task = new AskLoginTask();
    try {
      SwingThreadDispatcher.invokeAndWait(task);
    }
    catch (InterruptedException ie) {
      throw new RuntimeException(ie);
    }
    catch (InvocationTargetException te) {
      throw (RuntimeException) te.getCause();
    }
    return task.getLoginInfo();
  }

  /**
   * Retorna o login do usurio.
   *
   * @return .
   */
  public String getLogin() {
    return loginInfo.getLoginName();
  }

  /**
   * Retorna a senha do usurio.
   *
   * @return .
   */
  public String getPassword() {
    return loginInfo.getPassword();
  }

  /**
   * Altera a senha do usurio.
   *
   * @param newPassword .
   */
  public void setPassword(String newPassword) {
    loginInfo.setPassword(newPassword);
  }

  /**
   * Indica se o mtodo de autenticao do servidor  local ou no.
   *
   * @return <code>true</code> se os usurios forem autenticados localmente,
   *         <code>false</code> caso contrrio.
   */
  public final boolean isLocalLogin() {
    return this.localLogin;
  }

  /**
   * Informa se o servidor aceite troca de senhas. A troca pode no ser aceita
   * quando o mtodo de autenticao utilizado pelo servidor acesse um servidor
   * externo. Atualmente, este  o caso da autenticao via LDAP.
   *
   * @return verdadeiro caso o servidor aceite trocar senhas, falso caso
   *         contrrio.
   */
  public final boolean canChangePasswords() {
    return this.passwordChangeAllowed;
  }

  /**
   * Inicia o monitor. Dados login, senha e o locale escolhido pelo usurio, o
   * monitor vai tentar estabelecer a conexo com o servidor, autenticar o
   * usurio e, a partir da, ir monitorar constantemente a comunicao.
   *
   * @param login login
   * @param password senha
   * @param locale locale
   *
   * @return indicativo
   *
   * @throws CSBaseException em caso de falha.
   * @throws IllegalArgumentException Caso algum dos parmetros esteja nulo.
   */
  public final boolean start(String login, String password, Locale locale)
    throws CSBaseException {
    if (login == null || password == null || locale == null) {
      throw new IllegalArgumentException(
        "login: " + login + ", password: " + password + ", locale: " + locale);
    }

    this.loginInfo = new LoginInfo(login, password, locale);
    this.authenticator = new UserPasswordAuthenticator(login, password, locale);
    return this.start();
  }

  /**
   * Inicia o monitor. Dado um token que identica um pre-login do usurio, o
   * monitor vai tentar estabelecer a conexo com o servidor, autenticar o
   * usurio a partir desse token e, a partir da, ir monitorar constantemente
   * a comunicao.
   *
   * @param token Token de identificao do usurio.
   *
   * @return indicativo
   *
   * @throws CSBaseException em caso de falha
   * @throws IllegalArgumentException Caso a chave de identificao da sesso do
   *         usurio seja nula.
   */
  public final boolean start(String token) throws CSBaseException {
    if (token == null) {
      throw new IllegalArgumentException("token: " + token);
    }

    this.authenticator = new PreLoginAuthenticator(token);
    this.loginInfo = null;
    return this.start();
  }

  /**
   * Inicia o monitor. Dada a credencial do usurio no OpenBus, o monitor vai
   * tentar estabelecer a conexo com o servidor, autenticar o usurio a partir
   * dessa credencial e, a partir da, ir monitorar constantemente a
   * comunicao.
   *
   * @param token O token que permite ao servidor conectar-se ao barramento
   *        usando a conexo do cliente e validar o usurio
   * @param locale locale
   *
   * @return indicativo
   *
   * @throws CSBaseException em caso de falha
   * @throws IllegalArgumentException Caso a chave de identificao da sesso do
   *         usurio seja nula.
   */
  public final boolean start(OpenBusLoginToken token, Locale locale)
    throws CSBaseException {
    this.authenticator = new OpenBusAuthenticator(token, locale);
    return this.start();
  }

  /**
   * Mtodo de incio.
   *
   * @return indicativo
   * @throws CSBaseException em caso de erro
   */
  protected boolean start() throws CSBaseException {
    boolean r = false;
    try {
      r = this.lookup();
      if (r) {
        this.startMonitoring();
      }
    }
    catch (RemoteException e) {
      e.printStackTrace();
    }
    return r;
  }

  /**
   * Faz o login para um usurio via login e senha ou token dependendo de como o
   * monitor foi iniciado. Faz tambm o registro do usurio logado.
   * <p>
   * Havendo uma queda do servidor, e o monitor tiver sido iniciado atravs de
   * um token, ser solicitado ao usurio que entre com login e senha.
   * </p>
   *
   * @return A sesso do usurio.
   *
   * @throws CSBaseException Caso ocorra algum problema no acesso ao servidor.
   * @throws RemoteException Em caso de falha na comunicao
   *
   * @see csbase.remote.ServerEntryPoint#preLogin(String, String, Locale)
   * @see csbase.remote.ServerEntryPoint#login(String)
   * @see csbase.remote.ServerEntryPoint#login(String, EncryptedPassword,
   *      Locale)
   */
  @Override
  protected Session login() throws CSBaseException, RemoteException {
    this.passwordChangeAllowed = this.getServer().canChangePasswords();
    this.localLogin = this.getServer().isLocalLogin();
    Session session;
    session = this.authenticator.authenticate(this.getServer());
    if (session == null) {
      return null;
    }
    User user = session.getUser();
    User.registerLogin(session.getKey(), user);
    return session;
  }

  /**
   * Retorna as informaes de login corrente. Se no h ainda nenhuma
   * informao de chave e senha,  solicitado ao usurio que informe a mesma.
   *
   * @return .
   */
  public LoginInfo getLoginInfo() {
    if (this.loginInfo == null) {
      this.loginInfo = askLogin();
    }
    return this.loginInfo;
  }

  /**
   * Obtem o caminho da imagem da tela de relogin.
   *
   * @return reloginImagePath o caminho da imagem.
   */
  protected String getReloginImagePath() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String lang(String key) {
    return LNG.get(key);
  }

  /**
   * Retorna true se loginInfo no foi inicializado. til, por exemplo, quando o
   * login foi feito por token (login as) durante a gerao de menus do desktop
   *
   * @return true se loginInfo no foi inicializado
   */
  public boolean isLoginInfoInitialized() {
    return loginInfo != null;
  }
}
