/*
 * $Id$
 */
package csbase.client.applicationmanager;

import java.awt.Window;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Locale;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Vector;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JMenuItem;

import csbase.client.Client;
import csbase.client.desktop.DesktopPref;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.types.PVList;
import csbase.client.remote.srvproxies.EventLogProxy;
import csbase.logic.FileInfo;
import csbase.logic.RemoteFileInputStream;
import csbase.logic.applicationservice.AppPropertyResourceBundle;
import csbase.logic.applicationservice.ApplicationCategory;
import csbase.logic.applicationservice.ApplicationRegistry;
import csbase.remote.ApplicationServiceInterface;
import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartManager;
import csdk.v2.helper.application.Message;
import csdk.v2.helper.application.MessageSender;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNGKeys;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Estrura interna
 *
 * @author Tecgraf/PUC-Rio
 */
class InternalStructure {
  /**
   * Link
   */
  final private ApplicationLink link;

  /**
   * Ao
   */
  final private AbstractAction action;

  /**
   * @return o campo link
   */
  final ApplicationLink getLink() {
    return link;
  }

  /**
   * @return o campo action
   */
  final AbstractAction getAction() {
    return action;
  }

  /**
   * Construtor
   *
   * @param action ao
   * @param link link
   */
  public InternalStructure(AbstractAction action, ApplicationLink link) {
    this.action = action;
    this.link = link;

  }
}

/**
 * Classe que implementa o gerente de aplicaes do cliente, que ser
 * representado por um singleton.
 *
 * @author Tecgraf
 */
public final class ApplicationManager {

  /**
   * Pool de thread para tratamento de mensagens assncronas. Inicia-se com uma
   * thread, e aumenta de acordo com a demanda. Caso uma thread fique inativa
   * por mais de 60 segundos  destruda.
   *
   */
  private static final ExecutorService threadPool = new ThreadPoolExecutor(1,
    Integer.MAX_VALUE, 60L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>());

  /**
   * Nome da fila de eventos cliente bsica de aplicativos
   */
  private final String appQueueName = "Applications";

  /**
   * Fila de eventos cliente bsica de aplicativos
   */
  private final String[] appQueue = { appQueueName };

  /**
   * Instncia do singleton para o gerente de servios.
   */
  private static ApplicationManager instance = null;

  /**
   * Lista de tipos de aplicaes registradas.
   */
  private final Hashtable<String, ApplicationRegistry> registries;

  /**
   * Lista de instncias de aplicaes em execuo.
   */
  final private Hashtable<String, Vector<ApplicationType>> runningApplications =
    new Hashtable<>();

  /** Controla as aplicaes externas */
  final private ExternalApplicationManager externalAppsManager;

  /**
   * Seleo de idioma do usurio.
   */
  private Locale locale = null;

  /**
   * Associa os tipos de arquivos com as aplicaes
   */
  private Hashtable<String, Vector<String>> fileTypeAssociations = null;

  /**
   * Listeners
   */
  private final ArrayList<ApplicationManagerListener> listeners =
    new ArrayList<>();

  /**
   * Armazenamento de estrutura por identificador de aplicao.
   */
  final private Hashtable<String, InternalStructure> structures =
    new Hashtable<>();

  /**
   * Categorias de aplicaes.
   */
  final private Hashtable<String, ApplicationCategory> categories;

  /*
   * Inicializador esttico.
   */
  static {
    RestartManager.getInstance().addListener(() -> ApplicationManager.instance =
      null);
  }

  /**
   * Mtodo que retorna a instncia do gerente.
   *
   * @return a instncia.
   */
  public static ApplicationManager getInstance() {
    return ApplicationManager.instance;
  }

  /**
   * Ajuste da instncia
   *
   * @param locale o Locale desejado.
   * @throws RemoteException em caso de falha.
   */
  public static void setInstance(final Locale locale) throws RemoteException {
    if (ApplicationManager.instance == null) {
      ApplicationManager.instance = new ApplicationManager(locale);
    }
  }

  /**
   * Adio de listener
   *
   * @param listener o listener
   */
  final public void addListener(final ApplicationManagerListener listener) {
    listeners.add(listener);
  }

  /**
   * Consulta estrutura interna
   *
   * @param id identificador da aplicao
   * @return estrutura
   */
  private InternalStructure getInternalStructure(final String id) {
    return structures.get(id);
  }

  /**
   * Constroi o mapeamento dos tipos de arquivos com as aplicaes.
   */
  private void buildFileTypesAssociations() {
    if (this.registries == null) {
      return;
    }
    this.fileTypeAssociations = new Hashtable<>();
    for (final Enumeration<ApplicationRegistry> e = this.registries
      .elements(); e.hasMoreElements();) {
      final ApplicationRegistry reg = e.nextElement();
      final List<String> types = reg.getFileTypes();
      if (types == null) {
        continue;
      }
      for (final String type : types) {
        // Verifica se j tem o tipo com alguma aplicao
        final Vector<String> apps = this.fileTypeAssociations.get(type);
        if (apps == null) {
          final Vector<String> newApps = new Vector<>();
          newApps.add(reg.getId());
          this.fileTypeAssociations.put(type, newApps);
        }
        else {
          apps.add(reg.getId());
        }
      }
    }
  }

  /**
   * Consulta a possibilidade de habilitar a aplicao.
   *
   * @param hasProject se existe projeto aberto.
   * @param reg registro da aplicao.
   * @return um flag indicativo.
   */
  private boolean canEnableApplication(final boolean hasProject,
    final ApplicationRegistry reg) {
    if (!reg.isEnabled()) {
      return false;
    }
    if (!reg.isRunnable()) {
      return false;
    }
    if (reg.requireProject()) {
      return hasProject;
    }
    return true;
  }

  /**
   * Construo de uam ao que dispare a aplicao
   *
   * @param aid o identificador da aplicao.
   * @return a ao.
   */
  public AbstractAction createApplicationAction(final String aid) {
    final ApplicationRegistry reg = getApplicationRegistry(aid);
    if (reg == null) {
      return null;
    }
    final InternalStructure structure = getInternalStructure(aid);
    final AbstractAction action = structure.getAction();
    return action;
  }

  /**
   * Construo de umitem de menu que dispara uma aplicao.
   *
   * @param aid o identificador da aplicao.
   * @param text .
   * @return um item de menu.
   */
  public JMenuItem createApplicationMenuItem(final String aid,
    final String text) {
    final ApplicationRegistry reg = getApplicationRegistry(aid);
    if (reg == null) {
      return null;
    }
    final AbstractAction action = createApplicationAction(aid);
    final JMenuItem mi = new JMenuItem(action);
    if (text != null) {
      mi.setText(text);
    }
    return mi;
  }

  /**
   * Cria a aplicao.
   *
   * @param id O identificador.
   * @return A aplicao.
   * @throws ApplicationException Em caso de erro.
   */
  @SuppressWarnings("unchecked")
  private ApplicationType createApplicationType(final String id)
    throws ApplicationException {
    try {
      final ApplicationRegistry currentRegistry = getApplicationRegistry(id);
      if (currentRegistry == null) {
        final String message = MessageFormat.format(LNG.get(
          "application.manager.no.registry"), id);
        throw new ApplicationException(message);
      }
      if (currentRegistry.isSingleton()) {
        final Vector<ApplicationType> running = runningApplications.get(id);
        if (running != null && running.size() > 0) {
          return running.get(0);
        }
      }

      /*
       * Caso a aplicao no esteja em cache, verificamos se existe uma nova
       * verso no servidor (se o registro do cliente est desatualizado). O
       * usurio deve escolher se tenta a atualizao, mesmo podendo gerar
       * inconsistncias no Desktop.
       */
      ApplicationRegistry registry = currentRegistry;
      if (!ApplicationCache.isApplicationVersionInCache(currentRegistry)) {
        final ApplicationRegistry newRegistry = getLatestRegistry(
          currentRegistry);
        if (newRegistry != null) {
          String message = LNG.get("application.manager.update.program");
          String updateOption = LNG.get("application.manager.update.option");
          final Object[] options = { updateOption, LNG.get(LNGKeys.CANCEL) };
          String title = registry.getApplicationName(locale);
          if (StandardDialogs.showOptionDialog(null, title, message,
            options) == 0) {
            updateRegistry(newRegistry);
            registry = newRegistry;
          }
          else {
            throw new ApplicationUpdateException();
          }
        }
      }

      ClassLoader applicationClassLoader = createApplicationClassLoader(
        registry);
      final String className = registry.getClassName();
      final Class<?> cls = Class.forName(className, true,
        applicationClassLoader);
      if (loadBundles(registry, cls)) {
        if (ApplicationType.class.isAssignableFrom(cls)) {
          return createApplicationType(id,
            (Class<? extends ApplicationType>) cls);
        }
        else if (csdk.v1_0.api.application.IApplication.class.isAssignableFrom(
          cls)) {
          return createCSDK1_0Application(id,
            (Class<? extends csdk.v1_0.api.application.IApplication>) cls);
        }
        else if (csdk.v2.api.application.IApplication.class.isAssignableFrom(
          cls)) {
          return createCSDKv2Application(id,
            (Class<? extends csdk.v2.api.application.IApplication>) cls);
        }
        else {
          final Object[] args = { registry.getApplicationName(locale) };
          String msg = LNG.get("application.manager.invalid.class", args);
          throw new ApplicationException(msg);
        }
      }
      else {
        final Object[] args = { registry.getApplicationName(locale), id,
            className };
        final String msg = LNG.get("application.manager.bundle.errors", args);
        throw new ApplicationException(msg);
      }
    }
    catch (final OutOfMemoryError e) {
      final String message = MessageFormat.format(LNG.get(
        "application.manager.out.of.memory"), id);
      throw new ApplicationException(message, e);
    }
    catch (final ClassNotFoundException e) {
      final String message = MessageFormat.format(LNG.get(
        "application.manager.no.class"), id);
      throw new ApplicationException(message, e);
    }
  }

  /**
   * Cria o classloader da aplicao, obtendo as bibliotecas do servidor, se
   * necessrio.
   *
   * @param reg o registro da aplicao.
   * @return o classloader da aplicao.
   * @throws ApplicationException em caso de erro na carga das bibliotecas.
   */
  private ClassLoader createApplicationClassLoader(ApplicationRegistry reg)
    throws ApplicationException {
    FileInfo[] libs = reg.getApplicationLibs();
    ClassLoader defaultClassLoader = this.getClass().getClassLoader();
    if (libs != null && libs.length > 0) {
      File libsDir = ApplicationCache.getApplicationCache(reg);
      if (libsDir == null) {
        throw new ApplicationCacheCorrupted();
      }
      URL[] libURLs = loadApplicationLibs(libsDir, libs);
      return new ApplicationClassLoader(libURLs, defaultClassLoader, reg
        .getClassLoaderWhiteList(), reg.getClassLoaderBlackList());
    }
    else {
      return defaultClassLoader;
    }
  }

  /**
   * Obtm o registro da aplicao mais atualizado no servidor, caso exista.
   *
   * @param registry o registro local da aplicao.
   * @return um registro mais atualizado, se houver.
   */
  private ApplicationRegistry getLatestRegistry(ApplicationRegistry registry) {
    final ApplicationServiceInterface appService =
      ClientRemoteLocator.applicationService;
    if (appService == null) {
      return null;
    }
    RemoteTask<ApplicationRegistry> task =
      new RemoteTask<ApplicationRegistry>() {
        @Override
        protected void performTask() throws Exception {
          ApplicationRegistry serverRegistry = appService
            .getApplicationRegistry(registry.getId());
          setResult(serverRegistry);
        }
      };
    String title = registry.getApplicationName(locale);
    String description = LNG.get("application.manager.update.registry.message");
    ApplicationRegistry serverRegistry = null;
    if (task.execute(null, title, description)) {
      serverRegistry = task.getResult();
    }
    if (serverRegistry == null) {
      return null;
    }
    Long serverTimestamp = serverRegistry.getTimestamp();
    long localTimestamp = registry.getTimestamp();
    if (serverTimestamp != localTimestamp) {
      return serverRegistry;
    }
    return null;
  }

  /**
   * Obtm o conjunto de bibliotecas da aplicao do cache de local do cliente.
   *
   * @param libDir o diretrio local de bilbiotecas da aplicao.
   * @param infos o conjunto de bibliotecas da aplicao.
   * @return o conjunto de URLs para as bibliotecas armazenadas localmente.
   * @throws ApplicationCacheCorrupted caso alguma das bibliotecas no seja
   *         encontrada no cache local.
   * @throws ApplicationException em caso de erro ao obter as URLS locais das
   *         bibliotecas.
   */
  private URL[] loadApplicationLibs(File libDir, FileInfo[] infos)
    throws ApplicationException {
    List<URL> urls = new ArrayList<>();
    for (final FileInfo info : infos) {
      File libFile = new File(libDir, info.getName());
      if (!libFile.exists()) {
        throw new ApplicationCacheCorrupted("Application library not found: "
          + libFile.getPath());
      }
      try {
        URL url = libFile.toURI().toURL();
        urls.add(url);
      }
      catch (MalformedURLException e) {
        throw new ApplicationException(e);
      }
    }
    return urls.toArray(new URL[urls.size()]);
  }

  /**
   * Carrega os pacotes de internacionalizao da aplicao.
   *
   * @param registry o registro da aplicao.
   * @param mainClass classe principal da aplicao.
   * @return verdadeiro se carga foi feita com sucesso ou falso, caso contrrio.
   */
  private boolean loadBundles(final ApplicationRegistry registry,
    final Class<?> mainClass) {
    AppPropertyResourceBundle resourceBundle = registry.getResourceBundle();
    if (resourceBundle != null) {
      return true;
    }

    // prefixo que indica a URL-Base para obteno dos bundles externos
    final String urlPrefix = Client.getCodeBaseStr();
    final RemoteTask<Boolean> loadBundleTask = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws RemoteException {
        boolean bundleOk = true;
        if (registry.isBundleRequired()) {
          bundleOk = false;
          if (LNG.getNativeLocale() != null && !locale.equals(LNG
            .getNativeLocale())) {
            bundleOk |= registry.loadClientBundles(mainClass, LNG
              .getNativeLocale());
          }
          bundleOk |= registry.loadClientBundles(mainClass, locale);
          if (LNG.getNativeLocale() != null && !locale.equals(LNG
            .getNativeLocale())) {
            bundleOk |= registry.loadServerBundles(
              ClientRemoteLocator.applicationService, LNG.getNativeLocale());
          }
          bundleOk |= registry.loadServerBundles(
            ClientRemoteLocator.applicationService, locale);
        }
        if (LNG.getNativeLocale() != null && !locale.equals(LNG
          .getNativeLocale())) {
          registry.loadURLBundles(urlPrefix, LNG.getNativeLocale());
        }
        registry.loadURLBundles(urlPrefix, locale);
        setResult(bundleOk);
      }
    };
    if (loadBundleTask.execute(null, null, LNG.get(
      "application.manager.task.bundle"))) {
      return loadBundleTask.getResult();
    }
    return false;
  }

  /**
   * Cria uma aplicao CSBase nativa.
   *
   * @param id O identificador.
   * @param appClass A classe que implementa a aplicao.
   * @return A aplicao.
   * @throws ApplicationException Em caso de erro.
   */
  private ApplicationType createApplicationType(final String id,
    Class<? extends ApplicationType> appClass) throws ApplicationException {
    try {
      final Class<?>[] argClasses = new Class[] { String.class };
      final Constructor<? extends ApplicationType> constructor = appClass
        .getConstructor(argClasses);
      final Object[] pObjs = new Object[] { id };
      final Object application = constructor.newInstance(pObjs);
      return (ApplicationType) application;
    }
    catch (final InvocationTargetException e) {
      final Throwable appException = e.getTargetException();
      final String message = MessageFormat.format(LNG.get(
        "application.manager.no.creation"), id, appException
          .getLocalizedMessage());
      throw new ApplicationException(message, appException);
    }
    catch (final NoSuchMethodException e) {
      final String message = MessageFormat.format(LNG.get(
        "application.manager.bad.configuration"), id);
      throw new ApplicationException(message, e);
    }
    catch (final InstantiationException e) {
      final String message = MessageFormat.format(LNG.get(
        "application.manager.bad.instanciation"), id);
      throw new ApplicationException(message, e);
    }
    catch (final IllegalAccessException e) {
      final String message = MessageFormat.format(LNG.get(
        "application.manager.bad.implementation"), id);
      throw new ApplicationException(message, e);
    }
  }

  /**
   * Cria uma aplicao baseada no CSDK 1.0.
   *
   * @param id O identificador.
   * @param appClass A classe que implementa a aplicao.
   * @return A aplicao.
   * @throws ApplicationException Em caso de erro.
   */
  private csbase.client.csdk.v1_0.application.CSDKApplication createCSDK1_0Application(
    final String id,
    Class<? extends csdk.v1_0.api.application.IApplication> appClass)
    throws ApplicationException {
    try {
      ApplicationRegistry reg = getApplicationRegistry(id);
      csbase.client.csdk.v1_0.application.CSDKApplication csdkApplication =
        new csbase.client.csdk.v1_0.application.CSDKApplication(reg, appClass);
      return csdkApplication;
    }
    catch (final OutOfMemoryError e) {
      final String message = MessageFormat.format(LNG.get(
        "application.manager.out.of.memory"), id);
      throw new ApplicationException(message, e);
    }
  }

  /**
   * Cria uma aplicao baseada no CSDK 2.0.
   *
   * @param id O identificador.
   * @param appClass A classe que implementa a aplicao.
   * @return A aplicao.
   * @throws ApplicationException Em caso de erro.
   */
  private csbase.client.csdk.v2.application.CSDKApplication createCSDKv2Application(
    final String id,
    Class<? extends csdk.v2.api.application.IApplication> appClass)
    throws ApplicationException {
    try {
      ApplicationRegistry reg = getApplicationRegistry(id);
      csbase.client.csdk.v2.application.CSDKApplication csdkApplication =
        new csbase.client.csdk.v2.application.CSDKApplication(reg, appClass);
      return csdkApplication;
    }
    catch (final OutOfMemoryError e) {
      final String message = MessageFormat.format(LNG.get(
        "application.manager.out.of.memory"), id);
      throw new ApplicationException(message, e);
    }
  }

  /**
   * Remoo de listener
   *
   * @param listener o listener
   */
  final public void delListener(final ApplicationManagerListener listener) {
    if (listeners.contains(listener)) {
      listeners.remove(listener);
    }
  }

  /**
   * Mtodo que termina todas as aplicaes registradas no ApplicationManager.
   * 
   * @return booleano que indica se as aplicaes foram fechadas.
   */
  public boolean finishAllApplications() {
    for (final Enumeration<String> appKeys = this.runningApplications
      .keys(); appKeys.hasMoreElements();) {
      final String key = appKeys.nextElement();
      if (key != null) {
        final Vector<ApplicationType> running = this.runningApplications.get(
          key);
        final int i = 0;
        while (running.size() > 0) {
          final ApplicationType app = running.get(i);
          if (!app.closeApplication()) {
            return false;
          }
        }
      }
    }
    return true;
  }

  /**
   * Obtm a lista completa dos registros de aplicao.
   *
   * @return a lista.
   */
  final public ArrayList<ApplicationRegistry> getAllApplicationRegistries() {
    final ArrayList<ApplicationRegistry> list = new ArrayList<>();
    for (final Enumeration<String> e = this.registries.keys(); e
      .hasMoreElements();) {
      final String aid = e.nextElement();
      final ApplicationRegistry reg = getApplicationRegistry(aid);
      if (reg != null) {
        list.add(reg);
      }
    }
    return list;
  }

  /**
   * Busca de um array para um event-log
   *
   * @param id id
   * @return a nova lista de strings
   */
  final public String[] getApplicationEventQueue(final String id) {
    return new String[] { appQueueName, id };

  }

  /**
   * Obtem a aplicao correspondente ao tipo de arquivo.
   *
   * @param fileType o tipo de arquivo.
   * @return o identificador da aplicao.
   */
  public List<String> getApplicationsFromType(final String fileType) {
    if (fileType == null) {
      return null;
    }
    final Vector<String> apps = this.fileTypeAssociations.get(fileType);
    return apps;
  }

  /**
   * Busca do cone configurada para a aplicao.
   *
   * @param id identificador da aplicao.
   * @return .
   */
  public ImageIcon getApplicationIcon(final String id) {
    final ApplicationRegistry reg = getApplicationRegistry(id);
    if (reg == null) {
      return null;
    }

    final byte[] icDef = reg.getIconDefinition();
    if (icDef == null) {
      return null;
    }
    return new ImageIcon(icDef);
  }

  /**
   * Busca de um array para um event-log (do manager)
   *
   * @return a nova lista de strings
   */
  private String[] getApplicationManagerStartStopQueue() {
    return appQueue;
  }

  /**
   * Mtodo para retorno do registry de uma aplicao.
   *
   * @param id o identificador de aplicao.
   * @return o registro.
   */
  public ApplicationRegistry getApplicationRegistry(final String id) {
    if (registries == null) {
      return null;
    }
    return this.registries.get(id);
  }

  /**
   * Verfica se existe registro para aplicao com o id definido.
   *
   * @param id o id.
   * @return verdeiro se o registro existe ou falso, caso contrrio.
   */
  public boolean hasApplicationRegistry(final String id) {
    if (registries == null) {
      return false;
    }
    return this.registries.containsKey(id);
  }

  /**
   * Mtodo para retorno categoria de aplicaes
   *
   * @param id o identificador
   * @return a categoria.
   */
  public ApplicationCategory getApplicationCategory(final String id) {
    if (categories == null) {
      return null;
    }
    return this.categories.get(id);
  }

  /**
   * Retorna a hash table de categorias.
   *
   * @return a tabela.
   */
  public Hashtable<String, ApplicationCategory> getAllApplicationCategories() {
    return categories;
  }

  /**
   * Busca de um array para um event-log
   *
   * @param application aplicativo
   * @param queue tag
   * @return a nova lista de strings
   */
  final public String[] getApplicationEventQueue(
    final ApplicationType application, final String[] queue) {
    if (queue == null || queue.length <= 0) {
      final String err = "Null or empty queue detected!";
      throw new IllegalArgumentException(err);
    }

    final int len = queue.length;
    final String[] newQueue = new String[len + 2];
    newQueue[0] = appQueueName;
    newQueue[1] = application.getId();
    for (int i = 0; i < len; i++) {
      newQueue[i + 2] = queue[i];
    }
    return newQueue;
  }

  /**
   * Consulta o identificador de instncia
   *
   * @param app aplicao.
   * @return id o identificador se a aplicao estiver rodando, ou um valor
   *         negativo caso contrrio.
   */
  final public int getInstanceIndex(final ApplicationType app) {
    final String id = app.getId();
    final Vector<ApplicationType> running = this.runningApplications.get(id);
    if (running == null) {
      return -1;
    }
    return running.indexOf(app) + 1;
  }

  /**
   * Obtem o locale das aplicaes.
   *
   * @return um locale.
   */
  public Locale getLocale() {
    return this.locale;
  }

  /**
   * Mtodo de acesso para a lista de instncias de aplicaes em execuo.
   *
   * @return Lista de instncias de aplicaes em execuo.
   */
  final public Hashtable<String, Vector<ApplicationType>> getRunningApplications() {
    return this.runningApplications;
  }

  /**
   * Determina se existem aplicaes em execuo.
   *
   * @return {@code true} caso existam aplicaes em execuo no momento ou
   *         {@code false}, caso contrrio.
   */
  final public boolean hasRunningApplications() {
    for (final Enumeration<String> appKeys = this.runningApplications
      .keys(); appKeys.hasMoreElements();) {
      final String key = appKeys.nextElement();
      if (key != null) {
        final Vector<ApplicationType> running = this.runningApplications.get(
          key);
        while (running.size() > 0) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Obtm um vetor com todas instncias de uma determinada aplicao que esto
   * executando.
   *
   * @param id identificador da aplicao
   * @return vetor com todas instncias em execuo de uma determinada aplicao
   *         que esto
   */
  public Vector<ApplicationType> getRunningApplications(final String id) {
    return this.runningApplications.get(id);
  }

  /**
   * Indica se existe pelo menos uma instncia de uma determinada aplicao em
   * execuo.
   *
   * @param id ID da aplicao
   * @return {@code true} se existe alguma instncia da aplicao em execuo
   * @see #isApplicationRunning(ApplicationType)
   */
  public boolean isAnyApplicationRunning(final String id) {
    Vector<ApplicationType> running = runningApplications.get(id);
    if (running == null || running.isEmpty()) {
      return false;
    }
    return true;
  }

  /**
   * Indica se uma instncia especfica de uma aplicao est em execuo.
   *
   * @param app instncia da aplicao
   * @return {@code true} se a instncia da aplicao est em execuo
   */
  public boolean isApplicationRunning(final ApplicationType app) {
    final Vector<ApplicationType> running = this.runningApplications.get(app
      .getId());
    if (running == null) {
      return false;
    }
    return running.contains(app);
  }

  /**
   * Mtodo para finalizar uma instncia de aplicao.
   *
   * @param app a instncia a ser finalizada
   * @return flag de sucesso da operao.
   */
  public boolean killApplication(final ApplicationType app) {
    try {
      setAsDead(app);
      app.killApplication();
      app.finishApplication();
      return true;
    }
    catch (final Exception e) {
      return false;
    }
  }

  /**
   * Mtodo para notificao de fim de aplicao (via usurio) ao gerente de
   * aplicaes.
   *
   * @param app a aplicao
   */
  public void notifyDeath(final ApplicationType app) {
    setAsDead(app);
    logExitApplication(app);
  }

  /**
   * Faz log de trmino de aplicao.
   *
   * @param app aplicao
   */
  private void logExitApplication(final ApplicationType app) {
    final String[] aQueue = getApplicationManagerStartStopQueue();
    final String endMsg = LNG.get("application.manager.finished.program");
    final String hashText = "HASHCODE:" + app.hashCode();
    final String[] initMsg = { endMsg, app.getName(), hashText, app.getId() };
    EventLogProxy.addClientInformation(aQueue, initMsg);
  }

  /**
   * Faz log de incio de aplicao.
   *
   * @param app aplicao
   */
  private void logStartApplication(ApplicationType app) {
    final String[] aQueue = getApplicationManagerStartStopQueue();
    final String startMsg = LNG.get("application.manager.started.program");
    final String hashText = "HASHCODE:" + app.hashCode();
    final String[] initMsg = { startMsg, app.getName(), hashText, app.getId() };
    EventLogProxy.addClientInformation(aQueue, initMsg);
  }

  /**
   * Executa uma aplicao.
   *
   * @param application A aplicao.
   * @throws ApplicationException Em caso de erro ao executar a aplicao.
   */
  public void runApplication(final ApplicationType application)
    throws ApplicationException {
    application.launchApplication();
    setAsRunning(application);
  }

  /**
   * Mtodo para criao de uma instncia de aplicao. Extraordinariamente,
   * este mtodo inibe <i>warning</i> de unchecked type, pois precisa fazer um
   * <i>cast</i> explcito para o tipo (classe) que efetivamente define a
   * aplicao.
   *
   * @param id o identificador da aplicao.
   * @param <A> a classe da aplicao chamada.
   * @return a aplicao instanciada.
   * @throws ApplicationException se houver falha no lanamento da aplicao.
   */
  @SuppressWarnings("unchecked")
  final public <A extends ApplicationType> A runApplication(final String id)
    throws ApplicationException {
    if (id == null) {
      final String err = "application null";
      throw new ApplicationException(err);
    }

    final ApplicationRegistry reg = getApplicationRegistry(id);
    try {
      ApplicationType app;
      try {
        app = createApplicationType(id);
      }
      catch (ApplicationCacheCorrupted e) {
        /*
         * Caso o cache local da aplicao tenha sido corrompido por algum
         * motivo, limpa o cache e tenta criar a aplicao novamente.
         */
        try {
          ApplicationCache.clearCacheForApplication(reg);
        }
        catch (Exception ex) {
          /*
           * Se o diretrio de cache no tiver sido apagado com sucesso, pode
           * ser que tenham ficado arquivos .nfs pendentes. Podemos seguir e
           * tentar criar a aplicao novamente.
           */
        }
        app = createApplicationType(id);
      }
      runApplication(app);
      app.postInitialization();
      logStartApplication(app);
      return (A) app;
    }
    catch (ApplicationUpdateException e) {
      return null;
    }
    catch (final Exception e) {
      final String msg = String.format(LNG.get(
        "application.manager.failed.program"), id);
      EventLogProxy.addClientInformation(appQueue, new String[] { msg });
  	  // TODO i18n
  	  StandardDialogs.showErrorDialog(null, LNG.get("application.manager.failed.program"), LNG.get("application.manager.failed.program.message"));
      throw new ApplicationException(msg, e);
    }
  }

  /**
   * Mtodo para criao de uma instncia de aplicao com base em sua classe.
   *
   * @param <A> classe da aplicao
   * @param clazz classe
   * @return a instncia executada.
   * @throws ApplicationException se houver falha no lanamento.
   */
  public <A extends ApplicationType> A runApplication(final Class<A> clazz)
    throws ApplicationException {
    final String id = getApplicationId(clazz);
    // O trecho "this.<A>"  um workaround sugerido para um bug do javac 1.6:
    // http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6302954
    return this.<A> runApplication(id);
  }

  /**
   * Indica a aplicao como terminada.
   *
   * @param app o tipo da aplicao.
   */
  private void setAsDead(final ApplicationType app) {
    final String id = app.getId();
    final Vector<ApplicationType> running = this.runningApplications.get(id);

    // Alguma aplicao pode ter morrido sem ter sido totalmente inicializada.
    if (running != null) {
      if (running.contains(app)) {
        running.remove(app);
      }
    }

    for (final ApplicationManagerListener lst : listeners) {
      lst.applicationEnded(id);
    }
  }

  /**
   * Indica a aplicao como em execuo.
   *
   * @param app o tipo da aplicao.
   */
  private void setAsRunning(final ApplicationType app) {
    final String id = app.getId();
    Vector<ApplicationType> running = this.runningApplications.get(id);
    if (running == null) {
      this.runningApplications.put(id, new Vector<>());
      running = this.runningApplications.get(id);
    }
    final ApplicationRegistry registry = app.getApplicationRegistry();
    if (!running.contains(app) || !registry.isSingleton()) {
      running.add(app);
    }

    for (final ApplicationManagerListener lst : listeners) {
      lst.applicationStarted(id);
    }
  }

  /**
   * Ajuste de flag de necessidade de projeto.
   *
   * @param hasProject flag indicativo.
   */
  public void setProjectFlag(final boolean hasProject) {
    if (registries == null) {
      return;
    }
    for (final ApplicationRegistry reg : registries.values()) {
      if (reg == null) {
        continue;
      }
      final String aid = reg.getId();
      final InternalStructure structure = getInternalStructure(aid);
      final ApplicationLink lnk = structure.getLink();
      final AbstractAction ac = structure.getAction();
      final boolean enable = canEnableApplication(hasProject, reg);
      if (lnk != null) {
        lnk.setEnabled(enable);
      }
      if (ac != null) {
        ac.setEnabled(enable);
      }
    }
  }

  /**
   * Obtm o link da aplicao.
   *
   * @param reg registry
   * @return o link.
   */
  final public ApplicationLink getApplicationLink(ApplicationRegistry reg) {
    return structures.get(reg.getId()).getLink();
  }

  /**
   * Obtm a representao de um caminho absoluto a partir raiz do sistema de
   * uma representao relativa  raiz da aplicao.
   *
   * @param reg o registro da aplicao.
   * @param relativePath o caminho relativo  raiz da aplicao (ex:
   *        resources/Notepad_pt_BR.properties).
   * @return o caminho relativo  raiz do sistema (ex:
   *         /csbase/client/applications/notepad/resources/
   *         Notepad_pt_BR.properties).
   */
  final public String getClassInternalPath(final ApplicationRegistry reg,
    final String relativePath) {
    final String fullPackageName = getApplicationFullPackageName(reg);
    final String classPath = fullPackageName.replace('.', '/');
    final String resPath = classPath + "/" + relativePath;
    return resPath;
  }

  /**
   * Utilitrio para saber o nome o pacote (nome do diretrio no fonte) aonde se
   * encontra a aplicao.
   *
   * @param reg regsitry
   * @return nome do pacote
   */
  final public String getApplicationFullPackageName(
    final ApplicationRegistry reg) {
    final String className = reg.getClassName();
    if (className == null) {
      final String err = "null class name for registry!";
      throw new IllegalStateException(err);
    }

    final int idx = className.lastIndexOf(".");
    if (idx == -1) {
      final String err = "Application class is not inside a module package!";
      throw new IllegalStateException(err);
    }
    final String fullPackageName = className.substring(0, idx);
    return fullPackageName;
  }

  /**
   * Utilitrio para saber o nome o pacote (nome do diretrio no fonte) aonde se
   * encontra a aplicao.
   *
   * @param reg regsitry
   * @return nome do pacote
   */
  final public String getApplicationPackageName(final ApplicationRegistry reg) {
    final String fullPackageName = getApplicationFullPackageName(reg);
    final int lastIndexOf = fullPackageName.lastIndexOf(".");
    if (lastIndexOf < 0) {
      return "";
    }
    final String packageName = fullPackageName.substring(lastIndexOf + 1);
    return packageName;
  }

  /**
   * Obtm uma ao de disparo da aplicao.
   *
   * @param reg registry
   * @return a ao.
   */
  final public AbstractAction getApplicationAction(ApplicationRegistry reg) {
    if (reg == null) {
      return null;
    }
    final String id = reg.getId();
    final InternalStructure structure = getInternalStructure(id);
    final AbstractAction action = structure.getAction();
    return action;
  }

  /**
   * Obtm um {@code ActionListener} de disparo da aplicao.
   *
   * @param reg registry
   * @return a ao.
   */
  final public ActionListener getApplicationActionListener(
    ApplicationRegistry reg) {
    if (reg == null) {
      return null;
    }
    final String id = reg.getId();
    final InternalStructure structure = getInternalStructure(id);
    final ActionListener actionListener = structure.getAction();
    return actionListener;
  }

  /**
   * Construtor interno.
   *
   * @param locale o Locale desejado.
   * @throws RemoteException em caso de falha.
   */
  private ApplicationManager(final Locale locale) throws RemoteException {
    this.locale = locale;
    this.registries = loadApplicationRegistries();
    this.categories = loadApplicationCategories();
    this.externalAppsManager = new ExternalApplicationManager();
    buildFileTypesAssociations();
    setProjectFlag(false);
  }

  /**
   * Faz a carga dos registros de aplicao do servidor.
   *
   * @return a hash.
   * @throws RemoteException em caso de falha.
   */
  private Hashtable<String, ApplicationRegistry> loadApplicationRegistries()
    throws RemoteException {
    final ApplicationServiceInterface appService =
      ClientRemoteLocator.applicationService;
    if (appService == null) {
      return null;
    }
    final Hashtable<String, ApplicationRegistry> regs = appService
      .getApplicationRegistries();
    for (ApplicationRegistry reg : regs.values()) {
      loadBundles(reg, getClass());
      final String id = reg.getId();
      final ApplicationStartAction action = new ApplicationStartAction(reg,
        locale);
      final ApplicationLink link = new ApplicationLink(action);
      final InternalStructure structure = new InternalStructure(action, link);
      structures.put(id, structure);
    }

    return regs;
  }

  /**
   * Faz a carga das categorias de aplicao.
   *
   * @return a hash.
   * @throws RemoteException em caso de falha.
   */
  private Hashtable<String, ApplicationCategory> loadApplicationCategories()
    throws RemoteException {
    final ApplicationServiceInterface appService =
      ClientRemoteLocator.applicationService;
    if (appService == null) {
      throw new IllegalStateException("ApplicationService not found or loaded");
    }

    final Hashtable<String, ApplicationCategory> cats = appService
      .getApplicationCategories();
    return cats;
  }

  /**
   * Consulta o nome de uma aplicao.
   *
   * @param applicationRegistry registry
   * @return nome
   */
  public String getApplicationName(ApplicationRegistry applicationRegistry) {
    return applicationRegistry.getApplicationName(locale);
  }

  /**
   * Consulta o nome de uma categoria.
   *
   * @param category category
   * @return nome
   */
  public String getCategoryName(ApplicationCategory category) {
    return category.getCategoryName(locale);
  }

  /**
   * Consulta a descrio de uma aplicao.
   *
   * @param applicationRegistry registry
   * @return descrio
   */
  public String getApplicationDescription(
    ApplicationRegistry applicationRegistry) {
    return applicationRegistry.getApplicationDescription(locale);
  }

  /**
   * Consulta o id de uma aplicao com base na classe que a implementa.
   *
   * @param clazz classe da aplicao
   * @return o id da aplicao (se registrada) ou {@code null}.
   */
  public String getApplicationId(final Class<? extends ApplicationType> clazz) {
    final String className = clazz.getName();
    final Set<Entry<String, ApplicationRegistry>> entries = registries
      .entrySet();
    for (Entry<String, ApplicationRegistry> entry : entries) {
      final ApplicationRegistry reg = entry.getValue();
      if (className.equals(reg.getClassName())) {
        return reg.getId();
      }
    }
    return null;
  }

  /**
   * Constroi um menu com applicaes de uma categoria.
   *
   * @param cat categoria
   * @return menu
   */
  final public JMenu buildCategoryMenu(ApplicationCategory cat) {
    final JMenu menu = new JMenu();
    final ArrayList<String> ids = cat.getApplicationIds();
    for (String id : ids) {
      final ApplicationRegistry reg = getApplicationRegistry(id);
      if (reg.isShownAtApplicationMenu()) {
        final AbstractAction action = getApplicationAction(reg);
        menu.add(action);
      }
    }
    menu.setText(cat.getCategoryName(locale));
    final byte[] ic = cat.getIcon();
    if (ic != null) {
      menu.setIcon(new ImageIcon(ic));
    }
    return menu;
  }

  /**
   * Retorna um stream de entrada correspondente a um arquivo que se encontra no
   * repositrio da aplicao.
   *
   * NOTA:  necessrio fechar o stream de entrada retornado por este mtodo.
   *
   * @param parent - janela pai usada na task que faz a consulta ao servidor.
   * @param appRegistry - registro da aplicao.
   * @param resourcePath - path do arquivo.
   * @return stream de entrada correspondente ao arquivo.
   */
  public InputStream getApplicationResource(Window parent,
    final ApplicationRegistry appRegistry, final String[] resourcePath) {

    if (appRegistry == null) {
      throw new IllegalArgumentException("appRegistry no pode ser nulo.");
    }
    if (resourcePath == null || resourcePath.length == 0) {
      throw new IllegalArgumentException(
        "resourcePath deve ter ao menos um elemento que define o arquivo "
          + "buscado.");
    }

    final ApplicationServiceInterface appService =
      ClientRemoteLocator.applicationService;

    Task<InputStream> task = new Task<InputStream>() {
      @Override
      protected void performTask() {
        RemoteFileChannelInfo info;
        try {
          info = appService.getApplicationResource(appRegistry.getId(),
            resourcePath);

          if (info != null) {
            InputStream in = new RemoteFileInputStream(info);
            setResult(in);
          }

        }
        catch (Exception e) {
          setResult(null);
        }
      }
    };

    task.execute(parent, getApplicationName(appRegistry), LNG.get(
      "application.manager.task.body"));

    return task.getResult();

  }

  /**
   * Consulta se aplicao (id)  favorita do usurio.
   *
   * @param id id
   * @return indicativo
   */
  final public boolean isFavoriteApplication(String id) {
    final List<String> ids = getFavoriteApplications();
    return ids.contains(id);
  }

  /**
   * Consulta o valor de preferncias para aplicaes preferenciais.
   *
   * @return valor de preferencias
   */
  final public List<String> getFavoriteApplications() {
    final PVList pv = getFavorites();
    final List<String> valuesIds = pv.getValue();
    final List<String> ids = new ArrayList<>();
    for (String valueId : valuesIds) {
      final ApplicationRegistry reg = getApplicationRegistry(valueId);
      if (reg != null) {
        ids.add(valueId);
      }
    }
    Collections.unmodifiableList(ids);
    return ids;
  }

  /**
   * Consulta o valor da preferncia de favoritos.
   *
   * @return valor
   */
  private PVList getFavorites() {
    final PreferenceManager pm = PreferenceManager.getInstance();
    final PreferenceCategory root = pm.loadPreferences();
    final PreferenceCategory pc = root.getCategory(DesktopPref.class);
    final DesktopPref fav = DesktopPref.FAVORITE_APPLICATIONS;
    final PVList pv = (PVList) pc.getPreference(fav);
    return pv;
  }

  /**
   * Adiciona uma aplicao como favorita
   *
   * @param id id da aplicao
   */
  final public void addAsFavoriteApplication(String id) {
    final ApplicationRegistry reg = getApplicationRegistry(id);
    if (reg == null) {
      return;
    }

    final List<String> ids = getFavoriteApplications();
    if (ids.contains(id)) {
      return;
    }
    ids.add(id);

    final PVList fav = getFavorites();
    fav.setValue(ids);
    PreferenceManager.getInstance().savePreferences();
  }

  /**
   * Remove uma aplicao como favorita
   *
   * @param id id da aplicao
   */
  final public void delAsFavoriteApplication(String id) {
    final ApplicationRegistry reg = getApplicationRegistry(id);
    if (reg == null) {
      return;
    }

    final List<String> ids = getFavoriteApplications();
    if (!ids.contains(id)) {
      return;
    }
    final PVList fav = getFavorites();
    ids.remove(id);
    fav.setValue(ids);
    PreferenceManager.getInstance().savePreferences();
  }

  /**
   * Obtm a instncia da aplicao a partir do seu identificador.
   *
   * @param id o identificador.
   * @return a instncia.
   */
  public ApplicationType getApplicationInstance(String id) {
    for (Vector<ApplicationType> instances : runningApplications.values()) {
      for (ApplicationType app : instances) {
        if (app.getInstanceId().equals(id)) {
          return app;
        }
      }
    }
    return null;
  }

  /**
   * Atualiza o registro da aplicao.
   *
   * @param newRegistry o novo registro.
   * @throws ApplicationUpdateException em caso de erro ao atualizar a
   *         aplicao.
   */
  private void updateRegistry(ApplicationRegistry newRegistry)
    throws ApplicationUpdateException {
    if (isAnyApplicationRunning(newRegistry.getId())) {
      throw new IllegalStateException(LNG.get(
        "application.manager.update.fail.program"));
    }
    else {
      registries.put(newRegistry.getId(), newRegistry);
    }
  }

  /**
   * Envia uma mensagem assncrona para a aplicao.
   *
   * @param appInstanceId id da instncia da aplicao.
   * @param type tipo da mensagem.
   * @param value valor da mensagem.
   * @param senderId id da aplicao de origem da mensagem.
   */
  public void sendAsyncMessage(String appInstanceId, String type, Object value,
    String senderId) {

    if (externalAppsManager.hasId(appInstanceId)) {
      externalAppsManager.sendAsyncMessage(appInstanceId, type, value,
        senderId);
      return;
    }

    ApplicationType appInstance = getApplicationInstance(appInstanceId);
    if (appInstance == null) {
      throw new IllegalArgumentException(LNG.get(
        "application.manager.instance.not.found"));
    }
    if (appInstance instanceof csbase.client.csdk.v2.application.CSDKApplication) {
      threadPool.submit(
        () -> ((csbase.client.csdk.v2.application.CSDKApplication) appInstance)
          .getApplication().onAsyncMessageReceived(new Message(type, value),
            new MessageSender("csbaseclient", senderId)));
    }
    else {
      threadPool.submit(() -> appInstance.sendMessage(type, value, senderId));
    }
  }

  /**
   * Envia uma mensagem assncrona para a todas as aplicaes.
   *
   * @param type tipo da mensagem.
   * @param value valor da mensagem.
   * @param senderId id da aplicao de origem da mensagem.
   */
  public void broadcastAsyncMessage(String type, Object value,
    String senderId) {
    ApplicationType sender = getApplicationInstance(senderId);
    if (sender == null) {
      throw new IllegalArgumentException(LNG.get(
        "application.manager.sender.instance.not.found"));
    }

    for (Vector<ApplicationType> instances : runningApplications.values()) {
      for (ApplicationType app : instances) {
        if (!app.getInstanceId().equals(senderId)) {
          final ApplicationType appInstance = app;
          if (appInstance instanceof csbase.client.csdk.v2.application.CSDKApplication) {
            threadPool.submit(
              () -> ((csbase.client.csdk.v2.application.CSDKApplication) appInstance)
                .getApplication().onAsyncMessageReceived(new Message(type,
                  value), new MessageSender("csbaseclient", senderId)));
          }
          else {
            threadPool.submit(() -> appInstance.sendMessage(type, value,
              senderId));
          }
        }
      }
    }

    externalAppsManager.broadcastMessage(type, value, senderId);
  }

  /**
   * Envia uma mensagem sncrona para a aplicao.
   *
   * @param appInstanceId id da instncia da aplicao.
   * @param type tipo da mensagem.
   * @param value valor da mensagem.
   * @param senderId id da aplicao de origem da mensagem.
   * @return A resposta dapa pela aplicao
   */
  public Object sendSyncMessage(String appInstanceId, String type, Object value,
    String senderId) {

    if (externalAppsManager.hasId(appInstanceId)) {
      return externalAppsManager.sendSyncMessage(appInstanceId, type, value,
        senderId);
    }

    ApplicationType appInstance = getApplicationInstance(appInstanceId);
    if (appInstance == null) {
      throw new IllegalArgumentException(LNG.get(
        "application.manager.instance.not.found"));
    }
    if (appInstance instanceof csbase.client.csdk.v2.application.CSDKApplication) {

      return ((csbase.client.csdk.v2.application.CSDKApplication) appInstance)
        .getApplication().onSyncMessageReceived(new Message(type, value),
          new MessageSender("csbaseclient", senderId));
    }

    throw new IllegalArgumentException(LNG.get(
      "application.manager.instance.not.csdk2"));
  }

  /**
   * Registra uma aplicao externa
   * 
   * @param url O caminho onde est a aplicao externa
   * @param type O tipo da aplicao
   * @return O id associado a aplicao
   */
  public String registerExternalApplication(String url, String type) {
    return externalAppsManager.registerApplication(url, type);
  }

  /**
   * Registra uma aplicao externa
   *
   * @param id O ID da aplicao
   * @return true caso a aplicao estava previamente registrada, false caso
   *         contrario
   */
  public boolean unregisterExternalApplication(String id) {
    return externalAppsManager.unregisterApplication(id);
  }
}
