/*
 * $Id$
 */
package csbase.client.applicationmanager;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyVetoException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Hashtable;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JDesktopPane;
import javax.swing.JInternalFrame;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.applications.ApplicationImages;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.util.CSBaseUI;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.applicationservice.ApplicationCategory;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Painel de aplicaes
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ApplicationPanel extends JDesktopPane {

  /**
   * Frames
   */
  final private Hashtable<String, ApplicationGroupInternalFrame> groupFrames =
    new Hashtable<String, ApplicationGroupInternalFrame>();

  /**
   * Nome da propriedade
   */
  private static final String APPS_INFO = "APPS_INFO";

  /**
   * Informe de projeto aberto
   * 
   * @param hasProject indicativo de projeto aberto
   */
  public void setProjectFlag(final boolean hasProject) {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    if (appManager == null) {
      return;
    }
    appManager.setProjectFlag(hasProject);
  }

  /**
   * Busca de ao que dispare uma aplicao.
   * 
   * @param aid id da aplicao
   * @return uma ao.
   */
  public AbstractAction getApplicationAction(final String aid) {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    if (appManager == null) {
      return null;
    }
    ApplicationRegistry reg = appManager.getApplicationRegistry(aid);
    if (reg == null) {
      return null;
    }
    return appManager.getApplicationAction(reg);
  }

  /**
   * Retorna a lista ordenada de frames.
   * 
   * @return a lista
   */
  private List<ApplicationGroupInternalFrame> getAllInternalFrames() {
    final List<ApplicationGroupInternalFrame> list =
      new ArrayList<ApplicationGroupInternalFrame>();
    final JInternalFrame[] frames = this.getAllFrames();
    for (JInternalFrame frm : frames) {
      list.add((ApplicationGroupInternalFrame) frm);
    }
    Collections.sort(list, new Comparator<JInternalFrame>() {
      @Override
      public int compare(JInternalFrame o1, JInternalFrame o2) {
        final String n1 = o1.getTitle();
        final String n2 = o2.getTitle();
        return n1.compareToIgnoreCase(n2);
      }
    });
    return list;
  }

  /**
   * Faz tile horizontal
   */
  public void tileGroupsHorizontal() {
    final List<ApplicationGroupInternalFrame> frames = getAllInternalFrames();
    int nframes = frames.size();
    if (nframes == 0) {
      return;
    }
    int height = this.getHeight();
    int width = (this.getWidth() / nframes);
    int i = 0;
    for (ApplicationGroupInternalFrame frame : frames) {
      // O setIcon pode levantar exceo se o look and feel
      // no prover suporte a iconificao
      try {
        frame.setIcon(false);
      }
      catch (PropertyVetoException e) {
      }

      // Restaura o tamanho da janela antes de mudar o seu
      // tamanho, evitando que a mesma fique travada.
      if (frame.isMaximum()) {
        try {
          frame.setMaximum(false);
        }
        catch (PropertyVetoException e) {
        }
      }
      frame.setSize(width, height);
      frame.setLocation(i * width, 0);
      frame.adjustIcons();
      i++;
    }
  }

  /**
   * Faz tile vertical
   */
  public void tileGroupsVertical() {
    final List<ApplicationGroupInternalFrame> frames = getAllInternalFrames();
    int nframes = frames.size();
    int height = (this.getHeight() / nframes);
    int width = this.getWidth();
    int i = 0;
    for (ApplicationGroupInternalFrame frame : frames) {
      // O setIcon pode levantar exceo se o look and feel
      // no prover suporte a iconificao
      try {
        frame.setIcon(false);
      }
      catch (PropertyVetoException e) {
      }

      // Restaura o tamanho da janela antes de mudar o seu
      // tamanho, evitando que a mesma fique travada.
      if (frame.isMaximum()) {
        try {
          frame.setMaximum(false);
        }
        catch (PropertyVetoException e) {
        }
      }
      frame.setSize(width, height);
      frame.setLocation(0, i * height);
      frame.adjustIcons();
      i++;
    }
  }

  /**
   * Iconificao de grupos
   * 
   * @param flag indicativo de iconificao.
   */
  public void iconifyGroups(boolean flag) {
    final List<ApplicationGroupInternalFrame> frames = getAllInternalFrames();
    for (ApplicationGroupInternalFrame frame : frames) {
      try {
        frame.setIcon(flag);
      }
      catch (Exception e) {
      }
    }
  }

  /**
   * Constri o menu para configurao de grupos de aplicaes. Este menu
   * permite arrumar/ladrilhar as janelas que contm grupos de aplicaes.
   * 
   * @return o menu de configurao dos grupos
   */
  final public JPopupMenu createGroupsConfigurationPopupMenu() {
    final JPopupMenu menu = new JPopupMenu();
    fillMenuItens(menu);
    return menu;
  }

  /**
   * Constri o menu para configurao de grupos de aplicaes. Este menu
   * permite arrumar/ladrilhar as janelas que contm grupos de aplicaes.
   * 
   * @return o menu de configurao dos grupos
   */
  final public JMenu createGroupsConfigurationMenu() {
    final JMenu menu = new JMenu(LNG.get("desktop.config.groups"));
    fillMenuItens(menu);
    return menu;
  }

  /**
   * @param cmp componente (menu ou popup menu).
   */
  private void fillMenuItens(final JComponent cmp) {
    if (!(cmp instanceof JPopupMenu) && !(cmp instanceof JMenu)) {
      throw new IllegalArgumentException("bad element set");
    }

    /* Iconificar janelas */
    JMenuItem iconItem = new JMenuItem(LNG.get("desktop.groups.iconify"));
    iconItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        iconifyGroups(true);
      }
    });

    /* Desiconificar janelas */
    JMenuItem deiconItem = new JMenuItem(LNG.get("desktop.groups.deiconify"));
    deiconItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        iconifyGroups(false);
      }
    });

    /* Cascatear janelas */
    JMenuItem cascItem = new JMenuItem(LNG.get("desktop.groups.cascade"));
    cascItem.setIcon(ApplicationImages.ICON_WINCASCADE_16);
    cascItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        cascadeGroups();
      }
    });

    /* Janelas na vertical */
    JMenuItem tileVertItem =
      new JMenuItem(LNG.get("desktop.groups.tile_vertical"));
    tileVertItem.setIcon(ApplicationImages.ICON_WINTILEVERTICAL_16);
    tileVertItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        tileGroupsVertical();
      }
    });

    /* Janelas na horizontal */
    JMenuItem tileHorItem =
      new JMenuItem(LNG.get("desktop.groups.tile_horizontal"));
    tileHorItem.setIcon(ApplicationImages.ICON_WINTILEHORIZONTAL_16);
    tileHorItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        tileGroupsHorizontal();
      }
    });
    cmp.add(iconItem);
    cmp.add(deiconItem);
    cmp.add(cascItem);
    cmp.add(tileVertItem);
    cmp.add(tileHorItem);
  }

  /**
   * Adiciona um listener para a rvore de projetos para garantir que a
   * configurao (layout) dos groups frames seja preservada na abertura e
   * fechamento dos projetos.
   * 
   * @param projectTree a rvore observada.
   */
  final public void addProjectTreeListernerForLayoutAutoPersistency(
    final ProjectTree projectTree) {
    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectChanged(CommonClientProject project) {
        if (project != null) {
          final User loggedUser = User.getLoggedUser();
          setGroupsFrameInfo(project.getGroupsFrameConfig(loggedUser.getId()));
        }
      }

      @Override
      public void projectClosed(final CommonClientProject project) {
        if (project != null) {
          final User loggedUser = User.getLoggedUser();
          project.setGroupsFrameInfo(loggedUser.getId(), getGroupsFrameInfo());
        }
      }
    });
  }

  /**
   * Adiciona um listener de mouse para ativao (com boto 1) de um menu popup
   * que permite ajustar (automaticamente) o layout dos frames de aplicao.
   */
  final public void addMouseListenerForLayoutMenu() {
    addMouseListener(new MouseAdapter() {
      @Override
      public void mousePressed(MouseEvent e) {
        final int x = e.getX();
        final int y = e.getY();
        final int button = e.getButton();
        if (button == MouseEvent.BUTTON1) {
          final JPopupMenu menu = createGroupsConfigurationPopupMenu();
          if (menu != null) {
            menu.show(ApplicationPanel.this, x, y);
          }
        }
      }
    });
  }

  /**
   * Adiciona um listener de mouse para ativao (com boto 3) de um menu popup
   * que permite lanar aplicaes com base no group manager.
   */
  final public void addMouseListenerForApplicationsMenu() {
    addMouseListener(new MouseAdapter() {
      @Override
      public void mousePressed(MouseEvent e) {
        final int x = e.getX();
        final int y = e.getY();
        final int button = e.getButton();
        if (button == MouseEvent.BUTTON3) {
          JPopupMenu menu = ApplicationManagerMenuUtilities.buildPopupMenu();
          if (menu != null) {
            menu.show(ApplicationPanel.this, x, y);
          }
        }
      }
    });
  }

  /**
   * Faz cascade de janelas de grupos.
   */
  public void cascadeGroups() {
    final List<ApplicationGroupInternalFrame> frames = getAllInternalFrames();
    int dx = 70;
    int dy = 50;
    int offsetX = 10;
    int offsetY = 20;
    int height = (int) (this.getHeight() * 0.6);
    int width = (int) (height * 1.6);
    int i = 0;
    for (ApplicationGroupInternalFrame frame : frames) {
      // O setIcon pode levantar exceo se o look and feel
      // no prover suporte a iconificao
      try {
        frame.setIcon(false);
      }
      catch (PropertyVetoException e) {
      }

      // Restaura o tamanho da janela antes de mudar o seu
      // tamanho, evitando que a mesma fique travada.
      if (frame.isMaximum()) {
        try {
          frame.setMaximum(false);
        }
        catch (PropertyVetoException e) {
        }
      }
      frame.setSize(width, height);
      frame.setLocation(offsetX + i * dx, offsetY + i * dy);
      frame.adjustIcons();
      frame.toFront();
      i++;
    }
  }

  /**
   * 
   */
  private void mountFrames() {
    final ApplicationManager mng = ApplicationManager.getInstance();
    final Hashtable<String, ApplicationCategory> cts =
      mng.getAllApplicationCategories();
    if (cts == null) {
      return;
    }

    final Collection<ApplicationCategory> categories = cts.values();
    for (ApplicationCategory cat : categories) {
      if (!cat.isShownAtApplicationPanel()) {
        continue;
      }
      /* Somente cria janela para o grupo se h aplicaes com link */
      final String name = mng.getCategoryName(cat);
      final Icon icon =
        cat.getIcon() == null ? null : (new ImageIcon(cat.getIcon()));
      final ApplicationGroupInternalFrame groupFrame =
        new ApplicationGroupInternalFrame(name, icon, getApplicationLinks(cat));
      add(groupFrame);
      groupFrames.put(cat.getId(), groupFrame);

    }
  }

  /**
   * Monta lista de links de uma categoria.
   * 
   * @param cat categoria
   * @return lista
   */
  private List<ApplicationLink> getApplicationLinks(ApplicationCategory cat) {
    final List<ApplicationLink> links = new ArrayList<ApplicationLink>();
    final ArrayList<String> ids = cat.getApplicationIds();

    for (String id : ids) {
      final ApplicationManager mng = ApplicationManager.getInstance();
      final ApplicationRegistry reg = mng.getApplicationRegistry(id);
      if (reg != null && reg.isShownAtApplicationPanel()) {
        final ApplicationLink link = mng.getApplicationLink(reg);
        if (link != null) {
          links.add(link);
        }
      }
    }
    return links;
  }

  /**
   * Metodo para configurar os frames dos grupos e os icones das aplicacoes. A
   * chave do grpsInfo eh o id do grupo.
   * 
   * @param grpsInfo info do grupo.
   */
  private void setGroupsFrameInfo(Hashtable grpsInfo) {
    if (grpsInfo == null) {
      return;
    }

    final ApplicationManager appManager = ApplicationManager.getInstance();
    final Hashtable<String, ApplicationCategory> hash =
      appManager.getAllApplicationCategories();
    final Collection<ApplicationCategory> cats = hash.values();
    for (ApplicationCategory cat : cats) {
      /* Obtem o frame do grupo */
      ApplicationGroupInternalFrame grpFrame = groupFrames.get(cat.getId());
      if (grpFrame == null) {
        continue; // grupo sem janela
      }
      grpFrame.getPanel().setLayout(null);
      Hashtable grpInfo = (Hashtable) grpsInfo.get(cat);
      if (grpInfo == null) {
        continue;
      }

      /* Atribui as configuracoes das posicoes das janelas dos grupos */
      grpFrame.setInfo(grpInfo);

      /* Obtem as configuracoes dos icones */
      Hashtable appsInfo = (Hashtable) grpInfo.get(APPS_INFO);
      if (appsInfo == null) {
        continue;
      }

      /* Obtem as aplicacoes deste grupo */
      final List<String> grpApps = cat.getApplicationIds();
      for (String grpApp : grpApps) {
        /* Configuracao da aplicacao */
        Hashtable appInfo = (Hashtable) appsInfo.get(grpApp);
        if (appInfo == null) {
          continue;
        }

        /* Obtem o link(icone) da aplicacao */

        final ApplicationRegistry reg =
          appManager.getApplicationRegistry(grpApp);
        ApplicationLink al = appManager.getApplicationLink(reg);
        if (al == null) {
          continue;
        }
        al.setInfo(appInfo);
      }
    }
  }

  /**
   * Obtem uma Hashtable de GroupFrameInfo que possui a disposio dos frames
   * das aplicacoes e dos cones correpondentes.
   * 
   * @return .
   */
  private Hashtable getGroupsFrameInfo() {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    final Hashtable<String, ApplicationCategory> hash =
      appManager.getAllApplicationCategories();
    final Collection<ApplicationCategory> cats = hash.values();

    Hashtable<String, Hashtable> grpsInfo = new Hashtable<String, Hashtable>();
    for (ApplicationCategory cat : cats) {
      ApplicationGroupInternalFrame grpFrame = groupFrames.get(cat.getId());
      if (grpFrame == null) {
        continue; // grupo sem janela...
      }

      Hashtable<String, Object> grpInfo = grpFrame.getInfo();

      /* Obtem as aplicacoes deste grupo */
      final List<String> grpApps = cat.getApplicationIds();

      Hashtable<String, Hashtable> appsInfo =
        new Hashtable<String, Hashtable>();
      for (String grpApp : grpApps) {
        final ApplicationRegistry reg =
          appManager.getApplicationRegistry(grpApp);
        if (reg == null) {
          continue;
        }
        ApplicationLink al = appManager.getApplicationLink(reg);
        if (al != null) {
          appsInfo.put(grpApp, al.getInfo());
        }
      }
      grpInfo.put(APPS_INFO, appsInfo);
      grpsInfo.put(cat.getId(), grpInfo);
    }
    return grpsInfo;
  }

  /**
   * Construtor
   */
  public ApplicationPanel() {
    this.setBackground(CSBaseUI.DESKTOP_BACKGROUND_COLOR);
    setProjectFlag(false);
    mountFrames();
    this.cascadeGroups();
  }
}
