/*
 * $Id:$
 */

package csbase.client.util.xmlpanel.xmlsearchpanel;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.UIManager;

import org.w3c.dom.Node;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.ApplicationImages;
import csbase.client.util.xmlpanel.XMLAbstractPanel;
import csbase.client.util.xmlpanel.XMLPanelNodeSelectionListener;

/**
 * Painel de busca de elementos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class XMLSearchPanel extends XMLAbstractPanel {

  /**
   * Listeners de seleo.
   */
  private final List<XMLPanelNodeSelectionListener> listeners =
    new ArrayList<XMLPanelNodeSelectionListener>();

  /**
   * O campo com o texto a ser procurado.
   */
  private final JTextField searchTextField = new JTextField();

  /**
   * A checkbox que indica se queremos dif. maisculas de minsculas
   */
  private final JCheckBox caseToggle = new JCheckBox();

  /**
   * Boto de procura: next
   */
  private final JButton nextButton = new JButton();

  /**
   * Boto de procura: previous
   */
  private final JButton prevButton = new JButton();

  /**
   * Label
   */
  private final JLabel findLabel = new JLabel();

  /**
   * Boto de procura: next
   */
  private final JButton closeButton = new JButton();

  /**
   * Direo de busca
   * 
   * @author Tecgraf/PUC-Rio
   */
  enum Direction {
    /**
     * Frente
     */
    FWD,
    /**
     * Trs
     */
    BCK
  }

  /**
   * Prxima busca.
   * 
   * @param direction direo.
   */
  protected void goToNodeFromCurrent(Direction direction) {
    Node searchNode = getNode();
    if (searchNode == null) {
      return;
    }

    Node nextNode = getNextNode(direction, searchNode);
    Node findNode = findNode(direction, nextNode);
    if (findNode == null) {
      return;
    }
    fireSearchSelection(findNode);
  }

  /**
   * Recurso para achar n.
   * 
   * @param direction direo
   * @param searchNode n de busca
   * @return n achado
   */
  private Node findNode(Direction direction, Node searchNode) {
    if (searchNode == null) {
      return null;
    }

    if (isSearchedNode(searchNode)) {
      return searchNode;
    }

    Node nextNode = getNextNode(direction, searchNode);
    while (nextNode != null) {
      if (isSearchedNode(nextNode)) {
        return nextNode;
      }
      nextNode = getNextNode(direction, nextNode);
    }
    return null;
  }

  /**
   * Indica se  o n procurado.
   * 
   * @param searchNode n
   * @return indicativo
   */
  private boolean isSearchedNode(Node searchNode) {
    String searchText = getSearchText();
    final String searchName = searchNode.getNodeName();

    final boolean found;
    if (caseToggle.isSelected()) {
      found = searchName.equals(searchText);
    }
    else {
      found = searchName.equalsIgnoreCase(searchText);
    }

    if (found) {
      return true;
    }
    return false;
  }

  /**
   * Busca prximo n a ser pesquisado.
   * 
   * @param direction indicativo de direo
   * @param searchNode n de busca
   * @return n (ou {@code null}).
   */
  private Node getNextNode(Direction direction, Node searchNode) {
    if (searchNode == null) {
      return null;
    }

    final Node parent = searchNode.getParentNode();
    switch (direction) {
      case FWD:
        Node fwdChild = searchNode.getFirstChild();
        if (fwdChild != null) {
          return fwdChild;
        }
        fwdChild = searchNode.getNextSibling();
        if (fwdChild != null) {
          return fwdChild;
        }
        if (parent == null) {
          return null;
        }
        fwdChild = parent.getNextSibling();
        return fwdChild;

      case BCK:
        Node bckChild = searchNode.getPreviousSibling();
        if (bckChild != null) {
          Node tryNode = bckChild.getLastChild();
          if (tryNode != null) {
            return tryNode;
          }
          return bckChild;
        }
        if (parent == null) {
          return null;
        }
        return parent;

      default:
        return null;
    }
  }

  /**
   * Prxima busca.
   */
  protected void findNext() {
    goToNodeFromCurrent(Direction.FWD);
  }

  /**
   * Prxima busca.
   */
  protected void findPrevious() {
    goToNodeFromCurrent(Direction.BCK);
  }

  /**
   * Consulta o texto de busca.
   * 
   * @return texto
   */
  private String getSearchText() {
    String findTagText = searchTextField.getText();
    if (findTagText == null) {
      return null;
    }
    findTagText = findTagText.trim();
    if (findTagText.isEmpty()) {
      return null;
    }
    return findTagText;
  }

  /**
   * Ajuste de n corrente
   * 
   * @param node n
   */
  private void fireSearchSelection(Node node) {
    for (XMLPanelNodeSelectionListener listener : listeners) {
      listener.nodeSelected(node);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setNode(Node node) {
    internalSetNode(node);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(boolean visible) {
    if (!visible) {
      setNode(null);
    }
    else {
      searchTextField.setText(null);
      searchTextField.requestFocus();
    }
    super.setVisible(visible);
  }

  /**
   * Construtor
   */
  public XMLSearchPanel() {
    initComponents();
    mountPanel();
    adjustActions();
  }

  /**
   * Ajuste da aes de teclado.
   */
  private void adjustActions() {
    // adicionar teclas no mapa de eventos
    final InputMap inputMap =
      getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
    final ActionMap aMap = getActionMap();

    // inserir ao para o ENTER: procurar novamente
    final KeyStroke enterKey = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
    final AbstractAction enterAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        findNext();
      }
    };
    inputMap.put(enterKey, enterKey.toString());
    aMap.put(enterKey.toString(), enterAction);

    // inserir ao para o ESC: fechar [esconder] painel de busca
    final KeyStroke escKey = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);
    final AbstractAction escAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        setVisible(false);
      }
    };
    inputMap.put(escKey, escKey.toString());
    aMap.put(escKey.toString(), escAction);
  }

  /**
   * Montagem do painel.
   */
  private void mountPanel() {
    int i = 0;
    setLayout(new GridBagLayout());
    add(closeButton, new GBC(i++, 0));
    add(new JLabel("    "), new GBC(i++, 0));
    add(findLabel, new GBC(i++, 0));
    add(searchTextField, new GBC(i++, 0).horizontal());
    add(prevButton, new GBC(i++, 0));
    add(nextButton, new GBC(i++, 0));
    add(new JPanel(), new GBC(i++, 0).horizontal());
    add(caseToggle, new GBC(i++, 0));
  }

  /**
   * Inicializao dos componentes internos.
   */
  private void initComponents() {
    findLabel.setText(getString("find.label"));
    caseToggle.setText(getString("case.label"));

    final Dimension minSize = new Dimension(50, 10);
    searchTextField.setMinimumSize(minSize);
    closeButton.setIcon(UIManager.getIcon("InternalFrame.closeIcon"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        setVisible(false);
      }
    });

    GUIUtils.trimImageButton(closeButton);
    searchTextField.setColumns(20);
    nextButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        findNext();
      }
    });
    nextButton.setIcon(ApplicationImages.ICON_FIND_NEXT_16);
    GUIUtils.trimImageButton(nextButton);

    prevButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        findPrevious();
      }
    });
    prevButton.setIcon(ApplicationImages.ICON_FIND_PREV_16);
    GUIUtils.trimImageButton(prevButton);
  }

  /**
   * Adiciona listener.
   * 
   * @param listener listener
   */
  public void addSelectionListener(XMLPanelNodeSelectionListener listener) {
    listeners.add(listener);
  }
}
