/*
 * Detalhes da ltima alterao:
 * 
 * $Author: fpina $ $Date: 2016-09-22 18:12:51 -0300 (Thu, 22 Sep 2016) $
 * $Revision: 176168 $
 */
package csbase.client;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.swing.JComponent;
import javax.swing.JFrame;

import csbase.client.algorithms.parameters.DefaultParameterViewFactory;
import csbase.client.algorithms.parameters.ParameterViewFactory;
import csbase.client.desktop.LocalTask;
import csbase.client.desktop.RemoteTask;
import csbase.client.externalresources.ExternalResources;
import csbase.client.kernel.ClientException;
import csbase.client.login.AbstractLoginUI;
import csbase.client.login.InitialContext;
import csbase.client.login.LoginInterface;
import csbase.client.login.LoginUI;
import csbase.client.login.PreLogin;
import csbase.client.login.UserPasswordLogin;
import csbase.client.openbus.OpenBusAccessPoint;
import csbase.client.openbus.OpenBusLogin;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseException;
import csbase.exception.ConfigurationException;
import csbase.logic.ManifestVersion;
import csbase.logic.Permission;
import csbase.logic.ServerManager;
import csbase.logic.ServerURI;
import csbase.logic.User;
import csbase.logic.url.URLParameters;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ServerEntryPoint;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.configurationmanager.MissingPropertyException;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNG.TranslationListener;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * Representa um cliente.
 *
 * @author Tecgraf/PUC-Rio
 */
public abstract class Client {

  /**
   * O endereo padro para a mquina onde o cliente est rodando.
   */
  private static final String DEFAULT_HOST_ADDRESS = "127.0.0.1";

  /**
   * O nome da propriedade que indica qual a classe do localizador de servios
   * remotos.
   */
  private static final String REMOTE_LOCATOR_CLASS_PROPERTY =
    "remote.locator.class";

  /**
   * O nome da propriedade que indica qual a classe do monitorador do servidor.
   */
  private static final String REMOTE_MONITOR_CLASS_PROPERTY =
    "remote.monitor.class";

  /**
   * O nome da propriedade que define os arquivos de idiomas que pertencem ao
   * CSBase.
   */
  private static final String BASIC_LANGUAGE_FILE_PROPERTY =
    "basic.language.file";

  /**
   * O nome da propriedade que define os arquivos de idiomas adicionais (das
   * instncias).
   */
  private static final String ADDITIONAL_LANGUAGE_FILE_PROPERTY =
    "additional.language.file";

  /**
   * O nome da propriedade que define os arquivos de idiomas externos (das
   * instncias).
   */
  private static final String EXTERNAL_LANGUAGE_FILE_BASENAME_PROPERTY =
    "external.language.file.basename";

  /**
   * O nome da propriedade que definm os arquivos de idiomas para a traduo de
   * componentes do Swing.
   */
  private static final String SWING_LANGUAGE_FILE_PROPERTY =
    "swing.language.file";

  /**
   * O nome da propriedade que define a locale default na falta de uma ser
   * informada.
   */
  protected static final String DEFAULT_LOCALE_PROPERTY = "default.locale";

  /**
   * O nome da propriedade que defina uma permisso para execuo do sistema.
   */
  private static final String EXECUTION_PERMISSION_PROPERTY =
    "execution.permission";

  /**
   * O nome da propriedade que indica qual a classe do dilogo de login.
   */
  private static final String LOGIN_CLASS_PROPERTY = "login.class";

  /**
   * O nome da propriedade que indica qual a classe da interface de usurio.
   */
  private static final String UI_CLASS_PROPERTY = "ui.class";

  /**
   * O nome da propriedade que indica qual a classe do contexto inicial de
   * execuo do cliente.
   */
  private static final String INITIAL_CONTEXT_CLASS_PROPERTY =
    "initial.context.class";

  /**
   * O nome da propriedade que indica o nome do servidor deve ser exibido nas
   * janelas primrias.
   */
  private static final String SHOW_SERVER_PROPERTY = "show.server.name";

  /**
   * A configurao de um cliente.
   */
  private Configuration configuration;

  /**
   * A instncia nica do cliente.
   */
  private static Client instance;

  /**
   * Objeto de login utilizado.
   */
  private LoginInterface loginObject;

  /**
   * A fbrica responsvel por criar a viso dos parmetros dos algoritmos.
   */
  private ParameterViewFactory parameterViewFactory;

  /**
   * Fbrica responsvel por criar os contextos para aplicaes CSDK.
   */
  private final csbase.client.csdk.v1_0.CSDKContextFactory csdkContextFactory;

  /**
   * O nome do sistema.
   */
  private String systemName;

  /**
   * O nome do servidor, sem a porta, ao qual o cliente est conectado.
   */
  private String serverName;

  /**
   * O ip do servidor ao qual o cliente est conectado.
   */
  private String ipStr;

  /**
   * Um mapa com os parmetros de inicializao do cliente.
   */
  private Map<String, String> parameterMap;

  /**
   * Um mapa com os textos encontrados sem traduo em nenhum idioma carregado.
   */
  private Map<String, String> keyNotFoundInAnyIdiom;

  /**
   * Um mapa com os textos encontrados sem traduo para o idioma selecionado no
   * login do usurio.
   */
  private Map<String, String> keyNotFoundInSelectedIdiom;

  /**
   * Indica se as janelas primrias devem ter o nome do servidor
   */
  private boolean showServerName;

  /**
   * A interface de usurio.
   */
  private ClientUI clientUI;

  /**
   * Contador para gerao de identificadores de clientes-filhos.
   */
  private int sonCounter = 1;

  /**
   * Charset default para o sistema (buscado do servidor).
   */
  private Charset systemDefaultCharset;

  /**
   * Charset default da mquina cliente.
   */
  private Charset clientHostCharset;

  /**
   * Verso do cliente.
   */
  private ManifestVersion clientVersion;

  /**
   * Retorna o charset default do sistema, conforme configurado pelo servidor.
   *
   * @return o valor
   */
  public final Charset getSystemDefaultCharset() {
    return systemDefaultCharset;
  }

  /**
   * Retorna o charset default da mquina do cliente.
   *
   * @return o valor
   */
  public final Charset getClientHostCharset() {
    return clientHostCharset;
  }

  /**
   * Retorna o nome charset default do sistema, conforme definio de
   * {@link #getSystemDefaultCharset()}.
   *
   * @return o nome
   */
  public final String getSystemDefaultCharsetName() {
    final Charset ch = getSystemDefaultCharset();
    return ch.name();
  }

  /**
   * Retorna o nome charset default da mquina do cliente, conforme
   * {@link #getClientHostCharset()}.
   *
   * @return o nome
   */
  public final String getClientHostCharsetName() {
    final Charset ch = getClientHostCharset();
    if (ch == null) {
      return null;
    }
    return ch.name();
  }

  /**
   * Cria um cliente, definindo a sua instncia nica.
   *
   * @param systemName .
   * @param parameterMap A URL para conexo com o servidor.
   *
   * @throws IllegalStateException Caso j exista um cliente criado.
   * @throws ConfigurationException Caso no seja possvel obter a configurao
   *         da classe.
   */
  protected Client(String systemName, Map<String, String> parameterMap) {
    if (instance != null) {
      throw new IllegalStateException("J existe um cliente criado.");
    }
    SwingThreadDispatcher.init();
    this.systemName = systemName;
    this.parameterMap = parameterMap;
    this.keyNotFoundInAnyIdiom = new HashMap<String, String>();
    this.keyNotFoundInSelectedIdiom = new HashMap<String, String>();
    this.clientVersion = new ManifestVersion(getClass());
    String server = this.parameterMap.get(URLParameters.SERVER_URL_PARAMETER);
    ServerURI uri = ServerURI.parse(server);
    this.serverName = uri.getHost();
    try {
      this.configuration = ConfigurationManager.getInstance().getConfiguration(
        this.getClass());
    }
    catch (ConfigurationManagerException e) {
      throw new ConfigurationException(e);
    }
    this.showServerName = configuration.getOptionalBooleanProperty(
      SHOW_SERVER_PROPERTY, false);
    this.parameterViewFactory = new DefaultParameterViewFactory();
    this.csdkContextFactory = createCSDKContextFactory();
    instance = this;
  }

  /**
   * Cria a fbrica de contextos do CSDK.
   *
   * @return a fbrica de contextos.
   */
  protected csbase.client.csdk.v1_0.CSDKContextFactory createCSDKContextFactory() {
    return new csbase.client.csdk.v1_0.DefaultCSDKContextFactory();
  }

  /**
   * Retorna o nome da instncia do Client.
   *
   * @return Retorna o nome da instncia do Client.
   */
  public String getSystemName() {
    return this.systemName;
  }

  /**
   * Indica se o nome do servidor deve ser exibido no ttulo das janelas
   * primrias.
   *
   * @return true se o nome do servidor deve ser exibido no ttulo das janelas
   *         primrias
   */
  public boolean showServerName() {
    return showServerName;
  }

  /**
   * Executa o cliente.
   *
   * @return true, caso o cliente tenha sido executado com sucesso, ou false,
   *         caso contrrio.
   */
  public boolean execute() {
    /*
     * Define o locale nativo como sendo pt_BR. Isso garante que temos arquivos
     * de bundle necessrios para todos os textos.
     */
    LNG.setNativeLocale(new Locale("pt", "BR"));
    /* Busca locale inicial */
    final Locale initLocale = loadInitialLocale();
    System.out.println("Locale inicial: " + initLocale.toString());

    LNG.setTranslationListener(new TranslationListener() {
      @Override
      public String keyNotFound(String key, String text) {
        keyNotFoundInAnyIdiom.put(key, text);
        return text;
      }

      @Override
      public String keyNotFoundInDefaultLanguage(String key, String text) {
        keyNotFoundInSelectedIdiom.put(key, text);
        return text;
      }
    });

    /* Imprime a porta utilizada pelo cliente para exportar objetos. */
    System.out.println("Porta RMI do cliente: " + getRMIExportPort());

    /* Cria remote monitor */
    createClientRemoteMonitor();

    /* Cria objeto login */
    loginObject = loadLoginObject(initLocale);
    if (loginObject == null) {
      final String errTag = "csbase.client.Client.without.login.object";
      StandardDialogs.showErrorDialog(null, systemName, LNG.get(errTag));
      return false;
    }

    /* Efetua o login */
    final InitialContext initialContext = loginObject.login(initLocale,
      systemName);
    if (initialContext == null) {
      return false;
    }

    loadParameters(initialContext);

    /*
     * A ordem de carga dos bundles determina como ser montada a hierarquia de
     * bundles atravs das chamadas aos mtodos LNG.load() e LNG.loadURL()
     */
    Locale selectedLocale = initialContext.getLocale();
    loadLanguageFiles(selectedLocale);
    loadExternalLanguageFiles(selectedLocale);

    /*
     * Busca do charsets do servidor e da mquina. Se retornar o null representa
     * falha ou no ajuste.
     */
    clientHostCharset = loadHostCharset();
    systemDefaultCharset = loadDefaultCharsetFromServer();

    /* O charset vindo do servidor  obrigatrio */
    if (systemDefaultCharset == null) {
      final String errTag = "csbase.client.Client.charset.failed.message";
      final String err = LNG.get(errTag);
      StandardDialogs.showErrorDialog(null, systemName, err);
      return false;
    }

    /* O charset vindo do servidor deve ser suportado pelo cliente. */
    final String sysChName = getSystemDefaultCharsetName();
    if (!Charset.isSupported(sysChName)) {
      final String errTag = "csbase.client.Client.charset.unsupported.message";
      final String err = LNG.get(errTag) + "[" + sysChName + "]";
      StandardDialogs.showErrorDialog(null, systemName, err);
      return false;
    }
    System.out.println(" - Charset do servidor/sistema: " + sysChName);
    printOnNotRegisteredCharset(systemDefaultCharset);

    final String localChName = getClientHostCharsetName();
    System.out.println(" - Charset local (cliente): " + localChName);
    printOnNotRegisteredCharset(clientHostCharset);

    /*
     * Verificao de permisso do usurio.
     */
    if (!this.doesLoggedUserHavePermission()) {
      final String errTag =
        "csbase.client.Client.user.without.permission.message";
      StandardDialogs.showErrorDialog(null, systemName, LNG.get(errTag));
      return false;
    }

    postInitialization(initialContext);

    /* incio da carga do cliente */
    final boolean wasClientExecuted = executeClientUI(initialContext);
    return wasClientExecuted;
  }

  /**
   * Carrega novos parmetros definidos na tela de login
   * 
   * @param initialContext Contexto com o mapa de parmetros
   */
  private void loadParameters(InitialContext initialContext) {
    if (initialContext.getParameterMap() == null || initialContext
      .getParameterMap().size() <= 0) {
      return;
    }

    Map<String, String> newParamers = initialContext.getParameterMap();

    for (String key : newParamers.keySet()) {

      // log alteraes. Para os parmetros que foram logados previamente h um tratamento especial para usar o mesmo texto. 
      if (key.equals(URLParameters.RMI_PORT_PARAMETER)) {
        String value = newParamers.get(key);

        if (newParamers.get(key) != null) {
          int newPort = Integer.valueOf(value);
          if (newPort != getRMIExportPort()) {
            System.out.println("Parmetro alterado: Porta RMI do Cliente: "
              + newPort);
          }
        }
      }
      else if (this.parameterMap.get(key) != null) {
        System.out.println("Parmetro alterado: " + key + " = " + newParamers
          .get(key));
      }
      else {
        System.out.println("Parmetro includo: " + key + " = " + newParamers
          .get(key));
      }

      parameterMap.put(key, newParamers.get(key));
    }
  }

  /**
   * Carrega objeto de login.
   *
   * @param initLocale locale inicial
   * @return objeto {@link LoginInterface}.
   */
  protected LoginInterface loadLoginObject(Locale initLocale) {
    final LoginInterface lObject;
    final String tokenTag = URLParameters.PRE_LOGIN_TOKEN_PARAMETER;
    final String preLoginTokenText = parameterMap.get(tokenTag);
    if (preLoginTokenText != null) {
      final InitialContext initContext = createInitialContext(initLocale);
      lObject = new PreLogin(preLoginTokenText, initContext);
    }
    else {
      final String openBusTokenTag = URLParameters.OPENBUS_TOKEN_USER_PARAMETER;
      final String openBusTokenText = parameterMap.get(openBusTokenTag);
      if (openBusTokenText != null) {
        lObject = new OpenBusLogin(createInitialContext(initLocale));
      }
      else {
        lObject = new UserPasswordLogin(createLoginDialog());
      }
    }
    return lObject;
  }

  /**
   * Carrega locale inicial do cliente.
   *
   * @return locale
   */
  private Locale loadInitialLocale() {
    final String paramLocaleText = parameterMap.get(
      URLParameters.LOCALE_PARAMETER);
    final Locale paramLocale = FormatUtils.parseLocale(paramLocaleText);
    final Locale defaultLocale = Locale.getDefault();
    final Locale initLocale = (paramLocale == null ? defaultLocale
      : paramLocale);
    return initLocale;
  }

  /**
   * Carrega a propriedade de chaset.
   *
   * @return o charset
   */
  private Charset loadHostCharset() {
    final Charset charset = Charset.defaultCharset();
    return charset;
  }

  /**
   * Faz log se charset no for IANA.
   *
   * @param charset charset
   */
  private void printOnNotRegisteredCharset(final Charset charset) {
    if (charset == null) {
      return;
    }
    final boolean registered = charset.isRegistered();
    if (!registered) {
      final String fmt = "Charset [%s] no  registrado IANA!";
      final String chName = charset.name();
      final String msg = String.format(fmt, chName);
      System.out.println(msg);
    }
  }

  /**
   * Busca o charset (no servidor) para uso no cliente.
   *
   * @return o charset ou {@code null} (em caso de falha)
   */
  private Charset loadDefaultCharsetFromServer() {
    RemoteTask<String> chTask = new RemoteTask<String>() {
      @Override
      protected void performTask() throws RemoteException {
        final ClientRemoteMonitor monitor = ClientRemoteMonitor.getInstance();
        final ServerEntryPoint server = monitor.getServer();
        final String chName = server.getSystemDefaultCharsetName();
        setResult(chName);
      }
    };
    final String msgTag = "csbase.client.Client.charset.searching.message";
    final String msg = LNG.get(msgTag);
    boolean chTaskExecuted = chTask.execute(null, systemName, msg);
    if (!chTaskExecuted || chTask.wasCancelled()) {
      return null;
    }
    final String chName = chTask.getResult();
    if (chName == null || !Charset.isSupported(chName)) {
      return null;
    }

    final Charset charset = Charset.forName(chName);
    return charset;
  }

  /**
   * Retorna o identificador da instncia do cliente com base no login feito.
   *
   * @return o id.
   */
  final public String getClientInstanceId() {
    if (loginObject == null) {
      return null;
    }
    return loginObject.getClientInstanceId();
  }

  /**
   * Retorna o identificador da instncia do cliente que originou a execuo
   * deste cliente; a informao  obtida com base no login feito.
   *
   * @return o id.
   */
  final public String getFatherClientInstanceId() {
    if (loginObject == null) {
      return null;
    }
    return loginObject.getFatherClientInstanceId();
  }

  /**
   * Executa o cliente solicitando login e senha ao usurio.
   *
   * @param defaultLocale O locale default configurado na instalao do
   *        servidor.
   * @return true, caso a execuo do cliente ocorra sem problemas, ou false,
   *         caso contrrio.
   */
  protected InitialContext executeDefaultLogin(Locale defaultLocale) {
    final AbstractLoginUI loginDialog = this.createLoginDialog();
    return loginDialog.execute(defaultLocale);
  }

  /**
   * Obtem a classe de Interface de usurio definida pela propriedade respectiva
   *
   * @param uiClassProperty .
   *
   * @return A classe de interface de usurio.
   *
   * @throws ConfigurationException Caso a propriedade informando a interface de
   *         usurio no seja encontrada ou a classe informada no exista.
   */
  protected Class<ClientUI> getUIClass(String uiClassProperty) {
    Class<ClientUI> uiClass;
    try {
      uiClass = this.configuration.getMandatoryClassProperty(uiClassProperty);
      return uiClass;
    }
    catch (MissingPropertyException e) {
      throw new ConfigurationException(e);
    }
    catch (ClassNotFoundException e) {
      throw new ConfigurationException(e);
    }
  }

  /**
   * Carrega o cliente aps o login do usurio no sistema.
   *
   * @param initialContext Informaes de login.
   *
   * @return true, caso a carga do cliente ocorra sem problemas, ou false, caso
   *         contrrio.
   */
  protected boolean executeClientUI(final InitialContext initialContext) {
    try {
      this.clientUI = this.createUI(getUIClass(UI_CLASS_PROPERTY));
    }
    catch (ClientException e) {
      e.printStackTrace();
      final String tag = "csbase.client.Client.create.client.ui.error.message";
      final String err = LNG.get(tag);
      StandardDialogs.showErrorDialog(null, systemName, err);
      loginObject.notifyException(e);
      return false;
    }

    loginObject.notifyPreInitialization();

    RemoteTask<Void> remoteTask = new RemoteTask<Void>() {
      @Override
      protected void handleError(Exception e) {
        if (e instanceof RemoteException) {
          super.handleError(e);
        }
        else {
          e.printStackTrace();
          final String msg;
          if (e.getMessage() == null) {
            msg = LNG.get("csbase.client.Client.start.client.ui.error.message");
          }
          else {
            msg = e.getMessage();
          }
          StandardDialogs.showErrorDialog(null, systemName, msg);
        }
        loginObject.notifyException(e);
      }

      @Override
      protected void performTask() throws RemoteException, ClientException {
        clientUI.preInitialization(initialContext);
        clientUI.showUI();
      }
    };

    // task exibida somente no pre-login
    if (loginObject instanceof UserPasswordLogin) {
      remoteTask.setProgressDialogEnabled(false);
    }
    else {
      remoteTask.setProgressDialogDelay(0);
    }

    final String msgTag = "csbase.client.Client.starting.client.ui.message";
    final String msg = LNG.get(msgTag);
    final String message = MessageFormat.format(msg, systemName);
    boolean wasTaskExecuted = remoteTask.execute(null, systemName, message, 0,
      LocalTask.CLOSE_BUTTON);
    if (!wasTaskExecuted) {
      return false;
    }

    try {
      clientUI.postInitialization();
    }
    catch (Exception e) {
      final String tag =
        "csbase.client.Client.create.client.post.error.message";
      final String err = LNG.get(tag);
      StandardDialogs.showErrorDialog(null, systemName, err);
      loginObject.notifyException(e);
      return false;
    }

    loginObject.notifyInitialized();
    return true;
  }

  /**
   * Verifica se o usurio logado tem permisso para executar o cliente.
   *
   * @return true, se o usurio pode executar o cliente, ou false, caso
   *         contrrio.
   *
   * @throws ConfigurationException Caso a classe da permisso no seja
   *         encontrada.
   */
  protected boolean doesLoggedUserHavePermission() {
    Class<? extends Permission> executionPermissionClass;
    try {
      executionPermissionClass = this.configuration.getOptionalClassProperty(
        EXECUTION_PERMISSION_PROPERTY);
    }
    catch (ClassNotFoundException e) {
      throw new ConfigurationException(e);
    }
    User loggedUser = User.getLoggedUser();
    if (!loggedUser.isAdmin()) {
      if (executionPermissionClass != null) {
        if (loggedUser.getPermission(executionPermissionClass) == null) {
          return false;
        }
      }
    }
    return true;
  }

  /**
   * Carrega os arquivos de idioma.
   *
   * @param locale O locale do usurio logado.
   *
   * @throws ConfigurationException Caso no tenham sido definidos arquivos de
   *         idiomas ou no tenha sido definida alguma propriedade obrigatria.
   */
  public void loadLanguageFiles(Locale locale) {
    List<String> basicLanguageFileList;
    try {
      basicLanguageFileList = this.configuration.getMandatoryListProperty(
        BASIC_LANGUAGE_FILE_PROPERTY);
    }
    catch (MissingPropertyException e) {
      throw new ConfigurationException(e);
    }
    List<String> additionalLanguageFileList = this.configuration
      .getOptionalListProperty(ADDITIONAL_LANGUAGE_FILE_PROPERTY);
    List<String> allLanguageFileList = new LinkedList<String>();
    allLanguageFileList.addAll(basicLanguageFileList);
    allLanguageFileList.addAll(additionalLanguageFileList);
    Iterator<String> allLanguageFileIterator = allLanguageFileList.iterator();
    while (allLanguageFileIterator.hasNext()) {
      String idiomFileBaseName = allLanguageFileIterator.next();
      LNG.load(idiomFileBaseName, locale);
    }
    JComponent.setDefaultLocale(locale);
    if (!locale.getLanguage().equals(new Locale("en", "", "").getLanguage())) {
      try {
        GUIUtils.translateSwingComponents(locale, this.configuration
          .getMandatoryListProperty(SWING_LANGUAGE_FILE_PROPERTY));
      }
      catch (MissingPropertyException e) {
        throw new ConfigurationException(e);
      }
    }
  }

  /**
   * Carrega os arquivos de idioma que so externos. Os arquivos externos
   * definidos sero inseridos no fim da hierarquia de bundles.
   *
   * @param locale O locale do usurio logado.
   *
   * @return true, se todos os bundles externos foram carregados com sucesso
   */
  protected boolean loadExternalLanguageFiles(Locale locale) {

    // prefixo que indica a URL-Base para obteno dos bundles
    String urlPrefix = getCodeBaseStr();

    // carrega todos os arquivos de idiomas externos definidos
    List<String> externalLanguageFileList = getConfiguration()
      .getOptionalListProperty(EXTERNAL_LANGUAGE_FILE_BASENAME_PROPERTY);

    boolean success = true;
    for (String idiomFileBaseName : externalLanguageFileList) {
      try {
        LNG.loadURL(urlPrefix, idiomFileBaseName, locale);
      }
      catch (IOException e) {
        String name = urlPrefix + idiomFileBaseName + "_" + locale
          + ".properties";
        StandardErrorDialogs.showErrorDialog(null, getSystemName(),
          "Falha na carga de bundle de idiomas externo: " + name, e);
        success = false;
      }
    }
    return success;
  }

  /**
   * Retorna a URL que representa o codebase da aplicao ou, se no for
   * possvel obter os recursos externos, retorna o path absoluto corrente.
   *
   * @return prefixo de uma url, incluindo protocolo (http://, file:// etc)
   *
   */
  public static String getCodeBaseStr() {
    if (ExternalResources.getInstance().isEnabled()) {
      try {
        return ExternalResources.getInstance().getCodeBase().toExternalForm();
      }
      catch (CSBaseException e) {
        return FileUtils.prepareURL("");
      }
    }
    return FileUtils.prepareURL("");
  }

  /**
   * Cria o monitorador do servidor.
   */
  protected void createClientRemoteMonitor() {
    Class<? extends ServerManager> monitorClass = getClientRemoteMonitorClass();
    Constructor<? extends ServerManager> constructor;

    // Tipos dos argumentos do construtor do Monitor
    Class<?>[] argTypes = { String.class, String.class, Class.class };
    try {
      constructor = monitorClass.getConstructor(argTypes);
    }
    catch (NoSuchMethodException e) {
      throw new ConfigurationException(e,
        "No foi possvel obter o construtor da classe de monitorao do "
          + "servidor.");
    }

    // Argumentos para o construtor do Monitor
    Object[] args = { this.systemName, this.parameterMap.get(
      URLParameters.SERVER_URL_PARAMETER), this.getClientRemoteLocatorClass() };

    try {
      constructor.newInstance(args);
    }
    catch (InstantiationException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar o monitorador do servidor.");
    }
    catch (IllegalAccessException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar o monitorador do servidor.");
    }
    catch (InvocationTargetException e) {
      Throwable cause = e.getCause();
      if (cause instanceof RuntimeException) {
        throw (RuntimeException) cause;
      }
      throw new ConfigurationException(e,
        "No foi possvel criar o monitorador do servidor.");
    }
    ClientRemoteMonitor.getInstance().setSystemName(this.systemName);
  }

  /**
   * A classe do localizador de servios remotos.
   *
   * @return O localizador de servios remotos.
   *
   * @throws ConfigurationException Caso a classe informada no exista.
   */
  protected Class<? extends ClientRemoteLocator> getClientRemoteLocatorClass() {
    try {
      return this.configuration.getOptionalClassProperty(
        REMOTE_LOCATOR_CLASS_PROPERTY, ClientRemoteLocator.class);
    }
    catch (ClassNotFoundException e) {
      throw new ConfigurationException(e);
    }
  }

  /**
   * A classe de monitorao do servidor.
   *
   * @return O monitorador do servidor.
   *
   * @throws ConfigurationException Caso a classe informada no exista.
   */
  protected Class<? extends ServerManager> getClientRemoteMonitorClass() {
    try {
      return this.configuration.getOptionalClassProperty(
        REMOTE_MONITOR_CLASS_PROPERTY, ClientRemoteMonitor.class);
    }
    catch (ClassNotFoundException e) {
      throw new ConfigurationException(e);
    }
  }

  /**
   * <p>
   * Esse mtodo  chamado aps o usurio ter feito o login.
   * </p>
   * <p>
   * OBS: Nesse momento, as configuraes de idioma e os campos do locator para
   * acesso aos servios j esto definidos.
   * </p>
   *
   * @param initialContext O contexto inicial de execuo do cliente.
   */
  protected void postInitialization(InitialContext initialContext) {

    /*
     * Este mtodo no deve ser abstrato, pois nem todas as classes que herdam
     * de Client tem ps-inicializao.
     */
  }

  /**
   * Cria a interface de usurio.
   *
   * @param uiClass Classe da interface de usurio
   *
   * @return A interface de usurio.
   *
   * @throws ClientException Caso ocorra algum problema ao iniciar a interface
   *         de usurio.
   * @throws ConfigurationException Caso ocorra algum problema durante a
   *         construo da interface de usurio
   */
  protected ClientUI createUI(Class<ClientUI> uiClass) throws ClientException {
    Constructor<ClientUI> constructor;
    try {
      constructor = uiClass.getConstructor();
    }
    catch (NoSuchMethodException e) {
      throw new ConfigurationException(e,
        "No foi possvel obter o construtor da classe de interface do usurio.");
    }
    try {
      return constructor.newInstance();
    }
    catch (InstantiationException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar a interface do usurio.");
    }
    catch (IllegalAccessException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar a interface do usurio.");
    }
    catch (InvocationTargetException e) {
      Throwable cause = e.getCause();
      if (cause instanceof ClientException) {
        throw (ClientException) cause;
      }
      if (cause instanceof RuntimeException) {
        throw (RuntimeException) cause;
      }
      throw new ConfigurationException(e,
        "No foi possvel criar a interface do usurio.");
    }
  }

  /**
   * Cria o dilogo de login.
   *
   * @return O dilogo de login.
   *
   * @throws ConfigurationException Caso ocorra algum erro na criao do dilogo
   *         de login ou a classe informada como classe de login no exista.
   */
  protected AbstractLoginUI createLoginDialog() {
    Class<? extends AbstractLoginUI> loginClass;
    try {
      loginClass = this.configuration.getOptionalClassProperty(
        LOGIN_CLASS_PROPERTY, LoginUI.class);
    }
    catch (ClassNotFoundException e) {
      throw new ConfigurationException(e);
    }
    Constructor<? extends AbstractLoginUI> constructor;
    try {
      constructor = loginClass.getConstructor();
    }
    catch (NoSuchMethodException e) {
      throw new ConfigurationException(e,
        "No foi possvel obter o construtor da classe de login.");
    }
    try {
      return constructor.newInstance();
    }
    catch (InstantiationException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar o dilogo de login.");
    }
    catch (IllegalAccessException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar o dilogo de login.");
    }
    catch (InvocationTargetException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar o dilogo de login.");
    }
  }

  /**
   * Obtm a instncia nica do cliente.
   *
   * @return A instncia nica do cliente.
   *
   * @throws IllegalArgumentException Caso a instncia ainda no tenha sido
   *         criada.
   */
  public static Client getInstance() {
    if (instance == null) {
      final String err = "A instncia do cliente ainda no foi criada.";
      throw new IllegalArgumentException(err);
    }
    return instance;
  }

  /**
   * Obtm um parmetro do cliente.
   *
   * @param parameterName O nome do parmetro que se deseja.
   *
   * @return O parmetro, ou null, caso no exista.
   */
  public final String getParameter(String parameterName) {
    final String value = this.parameterMap.get(parameterName);
    if (value == null) {
      return null;
    }
    return value.trim();
  }

  /**
   * Obtm o nome do servidor, sem a porta, ao qual o cliente est conectado. O
   * endereo do servidor com a porta tambm  um parmetro do cliente que pode
   * ser obtido atravs do mtodo
   * <code>getParameter(Client.SERVER_URL_PARAMETER)</code>.
   *
   * @return O nome do servidor
   */
  public final String getServerName() {
    return serverName;
  }

  /**
   * Retorna o nome da mquina cliente
   *
   * @return nome
   */
  public final String getClientAddress() {
    try {
      final InetAddress myInetAddr = InetAddress.getLocalHost();
      final byte[] myAddr = myInetAddr.getAddress();
      ipStr = "";
      for (int i = 0; i < myAddr.length; i++) {
        if (i > 0) {
          ipStr += ".";
        }
        ipStr += myAddr[i] & 0xFF;
      }
      return ipStr;
    }
    catch (UnknownHostException e) {
      return "???.???.???.???";
    }
  }

  /**
   * Retorna o endereo IP do cliente.
   *
   * @return endereo
   */
  public final String getClientName() {
    try {
      final InetAddress myInetAddr = InetAddress.getLocalHost();
      return myInetAddr.getHostName();
    }
    catch (UnknownHostException e) {
      return "???";
    }
  }

  /**
   * Obtm o ip do servidor.
   *
   * @return o ip do servidor.
   *
   * @throws ConfigurationException se no foi possvel obter o ip, se um ou se
   *         um scope_id foi especificado para um endereo IPv6 global ou se
   *         existe um {@link SecurityManager gerente de segurana} e seu mtodo
   *         checkConnect no permite a operao.
   */
  public synchronized String getServerAddress() throws ConfigurationException {
    if (null == ipStr) {
      try {
        InetAddress address = InetAddress.getByName(getServerName());
        byte[] ip = address.getAddress();
        ipStr = "";
        for (int i = 0; i < ip.length; i++) {
          if (i > 0) {
            ipStr += ".";
          }
          ipStr += ip[i] & 0xFF;
        }
      }
      catch (UnknownHostException e) {
        throw new ConfigurationException(e,
          "No foi possvel obter o ip do servidor.");
      }
      catch (SecurityException e) {
        throw new ConfigurationException(e,
          "No foi possvel obter o ip do servidor.");
      }
    }
    return ipStr;
  }

  /**
   * Cria o contexto inicial de execuo do cliente.
   *
   * @param locale O idioma selecionado pelo usurio.
   *
   * @return O contexto inicial de execuo do cliente.
   *
   * @throws ConfigurationException Caso ocorra algum erro na criao do
   *         contexto inicial.
   */
  protected InitialContext createInitialContext(Locale locale) {
    Class<InitialContext> initialContextClass;
    try {
      initialContextClass = this.configuration.getOptionalClassProperty(
        INITIAL_CONTEXT_CLASS_PROPERTY, InitialContext.class);
    }
    catch (ClassNotFoundException e) {
      throw new ConfigurationException(e);
    }
    Constructor<InitialContext> constructor;
    try {
      constructor = initialContextClass.getConstructor(Locale.class);
    }
    catch (NoSuchMethodException e) {
      throw new ConfigurationException(e,
        "No foi possvel obter o construtor da classe contexto inicial.");
    }
    try {
      return constructor.newInstance(locale);
    }
    catch (InstantiationException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar o contexto inicial.");
    }
    catch (IllegalAccessException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar o contexto inicial.");
    }
    catch (InvocationTargetException e) {
      throw new ConfigurationException(e,
        "No foi possvel criar o contexto inicial.");
    }
  }

  /**
   * Obtm o endereo da mquina onde o cliente est rodando.
   *
   * @return O endereo da mquina onde o cliente est rodando.
   */
  public final String getHostAddress() {
    final String paramName = URLParameters.CLIENT_ADDRESS_PARAMETER;
    final String hostAddress = this.getParameter(paramName);
    if (hostAddress != null) {
      return hostAddress;
    }
    try {
      return InetAddress.getLocalHost().getHostAddress();
    }
    catch (UnknownHostException e) {
      return DEFAULT_HOST_ADDRESS;
    }
  }

  /**
   * Obtm o endereo web da aplicao no servidor.
   *
   * @return o endereo web da aplicao no servidor.
   */
  public final String getAppAddress() {
    return this.getParameter(URLParameters.URL_APP_PARAMETER);
  }

  /**
   * Retorna o locale default configurado como parmetro no arquivo JNLP do
   * cliente.
   *
   * @return O locale configurado no JNLP do cliente ou null se o parmetro no
   *         foi encontrado.
   */
  public final Locale getDefaultLocale() {
    final String paramName = URLParameters.LOCALE_PARAMETER;
    final String localeString = this.getParameter(paramName);
    if (localeString != null) {
      return FormatUtils.parseLocale(localeString);
    }
    return null;
  }

  /**
   * Retorna indicativo se dekstop deve iniciar visvel.
   *
   * @return indicativo.
   */
  public final boolean shouldDesktopStartVisible() {
    final String paramName = URLParameters.DESKTOP_VISIBLE_PARAMETER;
    final String paramValue = this.getParameter(paramName);
    if (paramValue != null) {
      final String trueValue = String.valueOf(true);
      return trueValue.equals(paramValue);
    }
    return true;
  }

  /**
   * Retorna indicativo se dekstop deve iniciar visvel.
   *
   * @return indicativo.
   */
  public final String getStartApplicationId() {
    final String paramName = URLParameters.START_APPLICATION_PARAMETER;
    final String paramValue = this.getParameter(paramName);
    if (paramValue == null) {
      return null;
    }
    return paramValue.trim();
  }

  /**
   * Obtm a configurao da classe.
   *
   * @return A configurao do usurio
   */
  protected Configuration getConfiguration() {
    return configuration;
  }

  /**
   * Obtm o componente grfico concreto que representa a interface de usurio.
   *
   * @return O componente grfico concreto.
   */
  public final JFrame getView() {
    if (this.clientUI == null) {
      return null;
    }
    return this.clientUI.getView();
  }

  /**
   * Obtm a fbrica de viso de parmetros dos algoritmos.
   *
   * @return A fbrica.
   */
  public ParameterViewFactory getParameterViewFactory() {
    return parameterViewFactory;
  }

  /**
   * Atribui a fbrica de viso de parmetros dos algoritmos.
   *
   * @param parameterViewFactory A fbrica.
   */
  protected void setParameterViewFactory(
    ParameterViewFactory parameterViewFactory) {
    this.parameterViewFactory = parameterViewFactory;
  }

  /**
   * Verifica se o acesso ao OpenBus est habilitado.
   *
   * @return {@code true}, caso o acesso esteja habilitado, ou {@code false},
   *         caso contrrio.
   */
  public boolean isOpenBusNeeded() {
    return (loginObject instanceof OpenBusLogin);
  }

  /**
   * Ajuste o objeto de login utilizada pelo cliente.
   *
   * @param loginObject o objeto.
   */
  final protected void setLoginObject(final LoginInterface loginObject) {
    if (loginObject == null) {
      throw new IllegalStateException("Bad login object set!");
    }
    this.loginObject = loginObject;
  }

  /**
   * Consulta o objeto de login utilizada pelo cliente.
   *
   * @return o objeto.
   */
  final public LoginInterface getLoginObject() {
    if (loginObject == null) {
      throw new IllegalStateException("Bad login object consult!");
    }
    return loginObject;
  }

  /**
   * Obtm as chaves de idioma no encontradas em todos os bundles carregados.
   *
   * @return um texto com as chaves no encontradas
   */
  public String getTextNotFoundInAnyIdiom() {
    StringBuffer textNotFound = new StringBuffer("\n");
    for (String key : keyNotFoundInAnyIdiom.keySet()) {
      textNotFound.append(key);
      textNotFound.append(": ");
      textNotFound.append(keyNotFoundInSelectedIdiom.get(key));
      textNotFound.append("\n");
    }
    return textNotFound.toString();
  }

  /**
   * Obtm as chaves de idioma no encontradas no idioma selecionado pelo
   * usurio. As chaves vem acompanhadas com o texto usado no idioma nativo.
   *
   * @return um texto com as chaves no traduzidas
   */
  public String getTextNotFoundInSelectedIdiom() {
    StringBuffer textWithoutTranslation = new StringBuffer("\n");
    for (String key : keyNotFoundInSelectedIdiom.keySet()) {
      textWithoutTranslation.append(key);
      textWithoutTranslation.append(": ");
      textWithoutTranslation.append(keyNotFoundInSelectedIdiom.get(key));
      textWithoutTranslation.append("\n");
    }
    return textWithoutTranslation.toString();
  }

  /**
   * Finalizao do cliente.
   */
  public void shutdown() {
    if (OpenBusAccessPoint.getInstance() != null) {
      if (OpenBusAccessPoint.getInstance().isInited()) {
        OpenBusAccessPoint.destroy();
      }
    }
    try {
      System.exit(0);
    }
    catch (Throwable t) {
      // Applet
    }
  }

  /**
   * Gera um novo id para um cliente-filho.
   *
   * @return o id
   */
  final public synchronized String generateNewSonId() {
    final String sonId = getClientInstanceId() + ":" + sonCounter;
    sonCounter++;
    return sonId;
  }

  /**
   * Configura o nome do servidor ao qual o usurio acabou de se conectar. O ip
   * do servidor  setado para null, forando que o mesmo seja recalculado sob
   * demanda.
   *
   * @param serverName o nome.
   */
  public final synchronized void setServerName(String serverName) {
    this.serverName = serverName;
    this.ipStr = null;
  }

  /**
   * <p>
   * Obtm a porta na qual os objetos RMI devem ser exportados.
   * </p>
   * <p>
   * Um valor menor ou igual a zero indica que objetos no devem ser exportados.
   * </p>
   *
   * @return porta na qual os objetos RMI devem ser exportados.
   */
  public int getRMIExportPort() {
    String port = getParameter(URLParameters.RMI_PORT_PARAMETER);
    if (port == null) {
      return 0;
    }
    return Integer.valueOf(port);
  }

  /**
   * Obtm a fbrica de contextos do CSDK.
   *
   * @return a fbrica.
   */
  public csbase.client.csdk.v1_0.CSDKContextFactory getCSDKContextFactory() {
    return csdkContextFactory;
  }

  /**
   * Obtm a verso do cliente.
   *
   * @return a verso.
   */
  public abstract String getVersion();

  /**
   * Indica se este cliente pode ou no utilizar algum modo de otimizao de
   * acesso  rea de projetos. O padro  usar sempre a otimizao.
   *
   * @return true, caso possa usar; false, caso contrrio.
   */
  public boolean useClientOptimizationMode() {
    final String paramName = URLParameters.OPT_MODE_PARAMETER;
    final String paramValue = this.getParameter(paramName);
    if (paramValue != null) {
      final String trueValue = String.valueOf(true);
      return trueValue.equals(paramValue);
    }
    return true;
  }

  /**
   * Indica se este cliente deve tentar abrir abrir um projeto.
   * 
   * @return o projeto indicado ou {@code null}.
   */
  public Object startProjectId() {
    final String paramName = URLParameters.START_PROJECT_PARAMETER;
    final String paramValue = this.getParameter(paramName);
    if (paramValue == null) {
      return null;
    }
    return paramValue;
  }
}
