package csbase.client.algorithms;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.InputEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.ScrollPaneConstants;

import csbase.client.Client;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.DesktopWindowInterface;
import csbase.client.desktop.Task;
import csbase.client.util.ClientUtilities;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;

/**
 * Dilogo para a visualizao de arquivo texto.
 *
 * @author Tecgraf/PUC-Rio
 */
public class FileViewerDialog extends JDialog implements
  DesktopWindowInterface {

  /**
   * rea de texto
   */
  private JTextArea textArea;

  /**
   * Painel com a rea de texto e com o painel de procura de textos
   */
  private JPanel textAreaPanel;

  /**
   * ScrollPane para a rea de texto
   */
  private JScrollPane textScrollPane;

  /**
   * Tamanho da fonte em porcentagem do tamanho original
   */
  private float fontSizePercent = 100;

  /**
   * Tamanho da fonte
   */
  private static final int ORIGINAL_FONT_SIZE = 12;

  /**
   * Fonte fixed
   */
  private static final Font FIXED_SIZE_FONT = new Font("Monospaced", Font.PLAIN,
    FileViewerDialog.ORIGINAL_FONT_SIZE);

  /**
   * InputStream de onde ser lido o arquivo texto
   */
  private InputStream fileInputStream;

  /**
   * Contrutor.
   * 
   * @param owner Janela me deste dilogo.
   * @param title Ttulo deste dilogo.
   * @param fileInputStream InputStream de onde ser lido o arquivo texto.
   */
  public FileViewerDialog(Window owner, String title,
    InputStream fileInputStream) {
    super(owner, title);
    this.fileInputStream = fileInputStream;
    setSize(new Dimension(800, 600));
    setLocationRelativeTo(owner);
    getContentPane().add(buildPanel());
    loadFile();
  }

  /**
   * Montagem do painel principal.
   * 
   * @return o painel.
   */
  private JPanel buildPanel() {
    final JPanel panel = new JPanel();
    panel.setLayout(new BorderLayout());
    panel.add(buildTextArea(), BorderLayout.CENTER);
    panel.add(buildButtonsPanel(), BorderLayout.SOUTH);
    return panel;
  }

  /**
   * Montagem do painel com a rea de texto.
   * 
   * @return o painel.
   */
  private JPanel buildTextArea() {

    textAreaPanel = new JPanel(new GridBagLayout());

    textArea = new JTextArea();
    textArea.setEditable(true);
    textArea.setFont(FileViewerDialog.FIXED_SIZE_FONT);
    textArea.setTabSize(4);
    ClientUtilities.applyUndoRedoActions(textArea);

    // listener para aumentar/diminuir fonte:
    textArea.addMouseWheelListener(new MouseWheelListener() {
      @Override
      public void mouseWheelMoved(final MouseWheelEvent e) {
        if ((e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0) {
          if (e.getScrollType() == MouseWheelEvent.WHEEL_UNIT_SCROLL) {
            final int totalScrollAmount = e.getUnitsToScroll();
            modifyFontSize(totalScrollAmount);
          }
        }
        else {
          textScrollPane.getMouseWheelListeners()[0].mouseWheelMoved(e);
        }
      }
    });

    textArea.setEditable(false);

    textScrollPane = new JScrollPane(textArea);
    textScrollPane.setVerticalScrollBarPolicy(
      ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
    final Dimension scrDim = new Dimension(600, 400);
    textScrollPane.setSize(scrDim);
    textScrollPane.setPreferredSize(scrDim);
    textScrollPane.setMinimumSize(scrDim);
    textAreaPanel.add(textScrollPane, new GBC(0, 0).insets(0, 1, 0, 0).both());

    return textAreaPanel;
  }

  /**
   * Montagem do painel de botes.
   * 
   * @return o painel.
   */
  private JPanel buildButtonsPanel() {
    final String cancelText = LNG.get("FileViwerDialog.button.close");
    JButton cancelButton = new JButton(cancelText);
    cancelButton.addActionListener(e -> this.close());

    JPanel buttonsPanel = new JPanel(new GridBagLayout());
    buttonsPanel.add(cancelButton, new GBC(0, 0).east().insets(0, 5, 0, 0));
    buttonsPanel.setBorder(BorderFactory.createEmptyBorder(10, 0, 10, 0));

    return buttonsPanel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() {
    DesktopFrame desktopFrame = DesktopFrame.getInstance();
    if (desktopFrame != null) {
      desktopFrame.removeWindow(this);
    }

    dispose();
  }

  /**
   * Aumento do tamanho do fonte
   * 
   * @param addSize incremento
   */
  protected void modifyFontSize(final int addSize) {
    if (fontSizePercent + addSize > 200) {
      fontSizePercent = 200;
    }
    else if (fontSizePercent + addSize < 75) {
      fontSizePercent = 75;
    }
    else {
      fontSizePercent += addSize;
    }
    final Font font = textArea.getFont();
    final float newFontSize = FileViewerDialog.ORIGINAL_FONT_SIZE
      * (fontSizePercent / 100f);
    textArea.setFont(font.deriveFont(newFontSize));
  }

  /**
   * Carrega o contedo do arquivo na rea de texto.
   */
  private void loadFile() {
    try {
      final String content = readFile(fileInputStream);
      textArea.setText(content);
      textArea.setCaretPosition(0);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * L um arquivo texto a partir de um InputStream, dentro de uma task.
   * 
   * @param stream InputStream.
   * @return Contedo do arquivo texto.
   * @throws Exception Erro na leitura do arquivo.
   */
  private String readFile(InputStream stream) throws Exception {
    Task<String> task = new Task<String>() {

      @Override
      protected void performTask() throws Exception {
        final Client client = Client.getInstance();
        Charset currentCharset = client.getSystemDefaultCharset();

        try {
          final String result = readFromInputStream(stream, currentCharset);
          setResult(result);
        }
        catch (IOException e) {
          e.printStackTrace();
        }
      }

    };

    String msg = LNG.get("FileViwerDialog.task.description");
    if (!task.execute(this, "", msg, true, false)) {
      final Exception error = task.getError();
      if (error != null) {
        throw error;
      }
      return "";
    }

    final String str = task.getResult();
    return str;
  }

  /**
   * L um arquivo texto a partir de um input stream.
   * 
   * @param stream input stream.
   * @param charset codificao do arquivo.
   * @return contedo do arquivo texto.
   * 
   * @throws IOException Erro de leitura do arquivo.
   */
  private String readFromInputStream(InputStream stream, Charset charset)
    throws IOException {
    final Charset readCharset;
    if (charset == null) {
      final Client client = Client.getInstance();
      readCharset = client.getSystemDefaultCharset();
    }
    else {
      readCharset = charset;
    }

    BufferedReader reader = null;
    try {

      final StringBuilder stringBuilder = new StringBuilder();
      final InputStreamReader rd = new InputStreamReader(stream, readCharset);
      reader = new BufferedReader(rd);
      final char[] buffer = new char[1024 * 10];
      int nbytes = reader.read(buffer);
      while (nbytes > 0) {
        stringBuilder.append(buffer, 0, nbytes);
        nbytes = reader.read(buffer);
      }
      reader.close();
      return stringBuilder.toString();
    }
    finally {
      if (reader != null) {
        reader.close();
      }
    }
  }

}
