/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.client.algorithms.parameters;

import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.border.Border;
import javax.swing.border.TitledBorder;

import csbase.client.Client;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.algorithms.validation.ViewValidator;
import csbase.client.algorithms.view.simple.SimpleAlgorithmConfiguratorPanel;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.ParameterLoader;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.logic.algorithms.validation.Validation;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.panel.ExpandablePanel;
import tecgraf.javautils.gui.panel.ExpandablePanel.Position;

/**
 * A Viso para {@link ParameterGroup grupo de parmetros}.
 */
public final class ParameterGroupView extends ParameterView<ParameterGroup> {

  /**
   * Lista de observadores de {@link ParameterGroupView}
   */
  private final List<ParameterGroupViewListener> listeners;

  /**
   * Painel que armazena os componentes da viso dos parmetros do grupo.
   */
  private JPanel componentsPanel;

  /**
   * Lista das vises dos parmetros pertencentes ao grupo.
   */
  private List<ParameterView<?>> parameterViews;

  /**
   * Viso do carregador da parmetros do grupo (se houver).
   */
  private ParameterLoaderView parameterLoaderView;

  /**
   * Painel com a viso do configurador do algoritmo que contm este grupo.
   */
  private final SimpleAlgorithmConfiguratorPanel configurationPanel;

  /**
   * Adiciona um observador de {@link ParameterGroupView}.
   *
   * @param listener O observador.
   */
  public void addParameterGroupViewListener(
    final ParameterGroupViewListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * Cria o painel que armazena os componentes da viso dos parmetros do grupo.
   */
  private void createComponentsPanel() {
    final String title = getParameter().getLabel();
    if (getGroup().isCollapsible()) {
      ExpandablePanel expandablePanel =
        new ExpandablePanel(title, Position.CONTROL_LEFT);
      final Border titledBorder = BorderFactory.createTitledBorder("");
      expandablePanel.setBorder(titledBorder);
      this.componentsPanel = expandablePanel;
    }
    else {
      this.componentsPanel = new JPanel(new GridBagLayout());
      if (!title.trim().isEmpty()) {
        final TitledBorder titledBorder =
          BorderFactory.createTitledBorder(title);
        titledBorder.setTitleJustification(TitledBorder.LEFT);
        this.componentsPanel.setBorder(titledBorder);
      }
    }
    populatePanel();
  }

  /**
   * Cria uma lista de observadores dos parmetros pertencentes a este grupo.
   * Esta viso precisa ser notificada das mudanas de visibilidade dos
   * parmetros, para poder reconstruir seu, quando necessrio.
   */
  private void createParameterListeners() {
    for (final ParameterView<?> view : getParameterViews()) {
      view.addParameterViewListener(new ParameterViewListener() {
        @Override
        public void visibilityWasChanged(final ParameterView<?> paramView) {
          updateVisibilyView();
          fireChildParameterVisibilityWasChanged();
        }
      });
    }
  }

  /**
   * Cria a viso do {@link ParameterLoader} do grupo, caso exista.
   */
  private void createParameterLoaderView() {
    final ParameterLoader parameterLoader = getGroup().getParameterLoader();
    if (parameterLoader != null && getMode().equals(Mode.CONFIGURATION)) {
      this.parameterLoaderView = new ParameterLoaderView(this, parameterLoader);
    }
  }

  /**
   * Cria a viso dos parmetros pertencentes a este grupo.
   */
  private void createParameterViews() {
    this.parameterViews = new LinkedList<ParameterView<?>>();
    for (final Parameter<?> parameter : getGroup().getParameters()) {
      final ParameterViewFactory parameterViewFactory =
        Client.getInstance().getParameterViewFactory();
      final ParameterView<?> parameterView =
        parameterViewFactory.create(configurationPanel, parameter, getMode());
      if (parameterView != null) {
        this.parameterViews.add(parameterView);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return resizesVertically(parameterViews);
  }

  /**
   * Avisa aos observadores do grupo sobre mudanas na visibilidade dos seus
   * parmetros.
   */
  private void fireChildParameterVisibilityWasChanged() {
    for (final ParameterGroupViewListener listener : this.listeners) {
      listener.childParameterVisibilityChanged(this);
    }
  }

  /**
   * Retorna o configurador do algoritmo que contm esse grupo de parmetros.
   *
   * @return O configurador do algoritmo.
   */
  public AlgorithmConfigurator getConfigurator() {
    return configurationPanel.getConfigurator();
  }

  /**
   * Retorna o parmetro do tipo grupo representado por esta viso.
   *
   * @return O grupo.
   */
  public ParameterGroup getGroup() {
    return getParameter();
  }

  /**
   * Retorna o painel onde essa viso foi adicionada.
   *
   * @return O painel.
   */
  public JPanel getPanel() {
    for (final ParameterView<?> parameterView : getParameterViews()) {
      if (parameterView.isVisible()) {
        return componentsPanel;
      }
    }

    componentsPanel.setVisible(false);
    return componentsPanel;
  }

  /**
   * Retorna a viso do parmetro com o nome especificado, caso esteja contida
   * nesse grupo ou nulo, caso contrrio.
   *
   * @param parameterName O nome de parmetro.
   * @return A viso do parmetro ou nulo, caso o parmetro no pertena a esse
   * grupo.
   */
  public SimpleParameterView<?> getParameterView(final String parameterName) {
    if (parameterName == null) {
      throw new IllegalArgumentException("O parmetro parameterName est nulo.");
    }
    for (final SimpleParameterView<?> parameterView : getParameterViews()) {
      if (parameterView.getParameter().getName().equals(parameterName)) {
        return parameterView;
      }
    }
    return null;
  }

  /**
   * Retorna a lista das vises dos parmetros que pertencem a esse grupo.
   *
   * @return A lista de vises dos parmetros.
   */
  public List<SimpleParameterView<?>> getParameterViews() {
    final List<SimpleParameterView<?>> paramViews =
      new LinkedList<SimpleParameterView<?>>();
    SimpleParameterView<?> simpleParameterView = null;
    for (final ParameterView<?> parameterView : this.parameterViews) {
      if (parameterView instanceof ParameterGroupView) {
        final ParameterGroupView groupView = (ParameterGroupView) parameterView;
        paramViews.addAll(groupView.getParameterViews());
      }
      else if (parameterView instanceof SimpleParameterView<?>) {
        simpleParameterView = (SimpleParameterView<?>) parameterView;
        paramViews.add(simpleParameterView);
      }
    }
    return Collections.unmodifiableList(paramViews);
  }

  /**
   * Retorna a janela que contm essa viso.
   *
   * @return A janela.
   */
  public Window getWindow() {
    return configurationPanel.getWindow();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean isVisible() {
    return this.componentsPanel.isVisible();
  }

  /**
   * Monta um grid com as vises dos parmetros listadas no painel especificado.
   *
   * @param panel O painel.
   * @param views A vises dos parmetros.
   */
  private final void mountBasicGridPanel(final JPanel panel,
    final List<SimpleParameterView<?>> views) {
    panel.setLayout(new GridBagLayout());
    if (views == null) {
      return;
    }

    final int T = 12;
    final int TI = 6;
    final int L = 11;
    final int B = 12;
    final int R = 11;

    final int numRows = views.size();
    for (int rowNumber = 0; rowNumber < numRows; rowNumber++) {
      final SimpleParameterView<?> row = views.get(rowNumber);
      if (row == null) {
        continue;
      }
      final JComponent[] components = row.getComponents();
      final int numCols = components.length;
      for (int colNumber = 0; colNumber < numCols; colNumber++) {
        final JComponent cell = components[colNumber];
        if (cell == null) {
          continue;
        }

        GBC gbc = new GBC(colNumber, rowNumber);
        int gw = 1;
        if(numCols == 1) {
          gw = GridBagConstraints.REMAINDER;
        }
        gbc = gbc.gridheight(1).gridwidth(gw);
        gbc = gbc.west().weights(1.0, 0.0);

        if (numRows == 1) {
          gbc = gbc.insets(T, L, B, R);
        }
        else {
          if (rowNumber == 0) {
            gbc = gbc.insets(T, L, 0, R);
          }
          else if (rowNumber == numRows - 1) {
            gbc = gbc.insets(TI, L, B, R);
          }
          else {
            gbc = gbc.insets(TI, L, 0, R);
          }
        }

        if (colNumber == 1 || numCols == 1) {
          if (row.fillVerticalSpace()) {
            gbc = gbc.both();
          }
          else {
            gbc = gbc.horizontal();
          }
          gbc = gbc.weightx(100.0);
        }
        else {
          gbc = gbc.none().west();
        }

        panel.add(cell, gbc);
      }
    }
  }

  /**
   * Monta o painel com as vises dos parmetros.
   */
  public void populatePanel() {
    this.componentsPanel.removeAll();
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.NORTHWEST;
    constraints.gridheight = 1;
    constraints.gridwidth = 1;
    constraints.gridx = 1;
    constraints.gridy = 1;
    constraints.insets = new Insets(0, 0, 0, 0);
    constraints.weightx = 1.0;

    final List<SimpleParameterView<?>> views =
      new LinkedList<SimpleParameterView<?>>();

    final Iterator<ParameterView<?>> parameterViewIterator =
      this.parameterViews.iterator();
    while (parameterViewIterator.hasNext()) {
      final ParameterView<?> parameterView = parameterViewIterator.next();
      if (parameterView instanceof SimpleParameterView<?>) {
        final SimpleParameterView<?> simpleParameterView =
          (SimpleParameterView<?>) parameterView;
        views.add(simpleParameterView);
      }
      else if (parameterView instanceof ParameterGroupView) {
        if (!views.isEmpty()) {
          final JPanel componentPanel = new JPanel();
          mountBasicGridPanel(componentPanel, views);
          if (resizesVertically(views)) {
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weighty = 1.0;
          }
          else {
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.weighty = 0;
          }
          views.clear();
          this.componentsPanel.add(componentPanel, constraints);
          constraints.gridy++;
        }
        final ParameterGroupView parameterGroupView =
          (ParameterGroupView) parameterView;
        parameterGroupView.populatePanel();
        final JPanel groupPanel = parameterGroupView.getPanel();
        if (parameterGroupView.fillVerticalSpace()) {
          constraints.fill = GridBagConstraints.BOTH;
          constraints.weighty = 1.0;
        }
        else {
          constraints.fill = GridBagConstraints.HORIZONTAL;
          constraints.weighty = 0;
        }
        this.componentsPanel.add(groupPanel, constraints);
        constraints.gridy++;
      }
    }
    if (!views.isEmpty()) {
      final JPanel componentPanel = new JPanel();
      mountBasicGridPanel(componentPanel, views);
      if (resizesVertically(views)) {
        constraints.fill = GridBagConstraints.BOTH;
        constraints.weighty = 1.0;
      }
      else {
        constraints.fill = GridBagConstraints.HORIZONTAL;
        constraints.weighty = 0;
      }
      this.componentsPanel.add(componentPanel, constraints);
      constraints.gridy++;
    }
    if (parameterLoaderView != null) {
      /* Adiciona o boto no final do grupo */
      final JPanel loaderPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
      loaderPanel.add(parameterLoaderView.getComponent());
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.weighty = 0;
      this.componentsPanel.add(loaderPanel, constraints);
    }
    this.componentsPanel.revalidate();
  }

  /**
   * Indica se a viso do grupo deve ser redimensionada verticalmente, de acordo
   * com a viso dos parmetros nele contidos.
   *
   * @param paramViews A lista de vises dos parmetros.
   * @return Verdadeiro se esse grupo deve ser redimensionado verticalmente ou
   * falso, caso contrrio.
   */
  private boolean resizesVertically(
    final List<? extends ParameterView<?>> paramViews) {
    for (final ParameterView<?> paramView : paramViews) {
      if (paramView.fillVerticalSpace() && paramView.isVisible()) {
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setVisible(final boolean isVisible) {
    this.componentsPanel.setVisible(isVisible);
    this.componentsPanel.revalidate();

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ViewValidationResult validate(ValidationMode mode)
    throws RemoteException {

    final Iterator<ParameterView<?>> parameterViewIterator = this.parameterViews
      .iterator();

    ViewValidationResult result = new ViewValidationResult(this);

    while (parameterViewIterator.hasNext()) {
      final ParameterView<?> view = parameterViewIterator.next();
      final ViewValidationResult validation = view.validate(mode);

      result.addChild(validation);
    }

    if (parameterLoaderView != null) {
      result.addChild(parameterLoaderView.validate(mode));
    }

    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean highlightValidationResult(ViewValidationResult result) {
    ViewValidator element = result.getElement();
    boolean highlighted = false;

    if (element.equals(this)) {
      for (Validation childResult : result.getChildren()) {
        if (childResult instanceof ViewValidationResult) {
          ViewValidationResult viewChildResult =
            (ViewValidationResult) childResult;
          highlighted |= viewChildResult.getElement().highlightValidationResult(
            viewChildResult);
        }
      }
    }

    if (highlighted && !result.isWellSucceeded()) {

      if (this.componentsPanel instanceof ExpandablePanel) {
        ExpandablePanel expandablePanel =
          (ExpandablePanel) this.componentsPanel;
        expandablePanel.setExpanded(true);
      }
      return true;
    }

    return false;
  }

  /**
   * Construtor
   *
   * @param configurationPanel o painel do configurador
   * @param group o grupo
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   * valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   * {@link ParameterView.Mode#REPORT}.
   */
  public ParameterGroupView(
    final SimpleAlgorithmConfiguratorPanel configurationPanel,
    final ParameterGroup group, final Mode mode) {
    super(group, mode);
    this.configurationPanel = configurationPanel;
    this.listeners = new LinkedList<ParameterGroupViewListener>();
    createParameterViews();
    createParameterLoaderView();
    createComponentsPanel();

    createParameterListeners();
  }
}
