package csbase.client.applications.fileexchanger.logic;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.rmi.RemoteException;

import tecgraf.javautils.core.io.FileUtils;
import csbase.client.applications.fileexchanger.FileExchangerUI;
import csbase.client.applications.fileexchanger.exceptions.ChildInvalidNameException;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.client.util.ClientUtilities;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileChannelLoadListener;
import csbase.logic.ProjectFileType;

/**
 * @author Tecgraf/PUC-Rio
 */
public class ExchangeImportThread extends ExchangeThread {

  /**
   * {@inheritDoc}
   */
  @Override
  public void run() {
    // TODO - Falta ajustar os tamahos para a progress bar
    Exchange exchange = null;
    File localFile = null;
    ClientProjectFile remoteFile = null;
    try {
      exchange = getExchange();
      localFile = exchange.getLocalFile();
      ClientProjectFile remoteDir = exchange.getRemoteDir();
      remoteFile = exchange.getRemoteFile();
      if (remoteFile == null) {
        remoteFile = createRemoteFile(localFile, remoteDir, remoteFile);
      }
      exchange.setRemoteFile(remoteFile);
      importFile(localFile, remoteFile);
    }
    // Tratamento de excees: do casos mais especficos para casos mais gerais.
    catch (FileNotFoundException e) {
      IOException io = null;
      try {
        final String[] args = new String[] { localFile.getAbsolutePath() };
        final String msg =
          FileExchangerUI.getString(
            "ExchangeImportThread.FileNotFoundException.msg", args);
        io = new IOException(msg, e);
        if (remoteFile != null && remoteFile.isOpen()) {
          remoteFile.close(true);
        }
      }
      catch (Exception e1) {
        e1.printStackTrace();
      }
      exchange.signalEnded(io);
      return;
    }
    // Tratamento geral para todo tipo de erro RemoteException.    
    catch (RemoteException e) {
      RemoteException re = null;
      try {
        final String tag = "FileExchanger.RemoteException.msg";
        final String msg = FileExchangerUI.getString(tag);
        re = new RemoteException(msg, e);
        if (remoteFile != null && remoteFile.isOpen()) {
          remoteFile.close(true);
        }
      }
      catch (Exception e1) {
        e1.printStackTrace();
      }
      exchange.signalEnded(re);
      return;
    }
    // Tratamento geral para todo tipo de erro IOException.
    catch (IOException e) {
      IOException io = null;
      try {
        final String[] args = new String[] { localFile.getAbsolutePath() };
        final String msg =
          FileExchangerUI.getString("ExchangeImportThread.IOException.msg",
            args);
        io = new IOException(msg, e);
        if (remoteFile != null && remoteFile.isOpen()) {
          remoteFile.close(true);
        }
      }
      catch (Exception e1) {
        e1.printStackTrace();
      }
      exchange.signalEnded(io);
      return;
    }
    // Tratamento para os outros casos.
    catch (Exception e) {
      try {
        if (remoteFile != null && remoteFile.isOpen()) {
          remoteFile.close(true);
        }
      }
      catch (Exception e1) {
        e1.printStackTrace();
      }
      exchange.signalEnded(e);
      return;
    }
    exchange.signalEnded(null);
  }

  /**
   * Envia os arquivos para o servidor.
   * 
   * @param localFile Arquivo local
   * @param remoteFile Arquivo remoto
   * @throws Exception Exceo lanada
   */

  private void importFile(File localFile, ClientProjectFile remoteFile)
    throws Exception {

    if (localFile.isDirectory()) {
      boolean childInvalidName = false;
      for (File child : localFile.listFiles()) {
        String childName = child.getName();
        // No importa arquivos de controle do csbase.
        if (childName.matches("^\u002E.*csbase$")
          || childName.matches("^\u002E.*csbase_description$")) {
          continue;
        }
        // Verifica se o nome do arquivo  vlido. 
        if (ClientUtilities.isValidFileName(childName)) {
          // O usurio j confirmou a sobreescrita do diretrio.
          ClientProjectFile remoteChild =
            GetChildFromNameTask.runTask(remoteFile, childName);
          if (remoteChild == null) {
            remoteChild = createRemoteFile(child, remoteFile, null);
          }
          try {
            importFile(child, remoteChild);
          }
          catch (ChildInvalidNameException e) {
            childInvalidName = true;
          }
        }
        else {
          childInvalidName = true;
        }
      }
      if (childInvalidName) {
        throw new ChildInvalidNameException(localFile.getName());
      }
    }
    else {
      final Exchange exchange = getExchange();
      final ExchangeMode transferMode = exchange.getTransferMode();
      switch (transferMode) {
        case STREAM:
          importAsStream(localFile, remoteFile);
          break;
        case OPERATION:
          importAsUpload(localFile, remoteFile);
          break;
      }
    }
  }

  /**
   * Cria o arquivo remoto no servidor se ele ainda no existir.
   * 
   * @param localFile arquivo local
   * @param remoteDir diretrio onde ser colocado o arquivo remoto
   * @param remoteFile arquivo remoto.
   * 
   * @return o arquivo remoto
   * @throws RemoteException
   */
  private ClientProjectFile createRemoteFile(File localFile,
    ClientProjectFile remoteDir, ClientProjectFile remoteFile)
    throws RemoteException {
    if (remoteFile != null) {
      return remoteFile;
    }

    final CommonClientProject project = getExchange().getProject();
    final String fileName = localFile.getName();
    ClientProjectFile newFile = null;
    if (ClientUtilities.isValidFileName(fileName)) {
      final String fileExtension = FileUtils.getFileExtension(fileName);
      final ProjectFileType pft =
        ProjectFileType.getProjectFileTypeFromExtension(fileExtension,
          localFile.isDirectory());
      final String fileType = (pft == null ? "" : pft.getCode());
      newFile = project.createFile(remoteDir, fileName, fileType);
    }
    return newFile;
  }

  /**
   * Envia os dados por stream.
   * 
   * @param localFile Arquivo local a ser importado.
   * @param remoteFile Arquivo remoto.
   * @throws IOException Exceo lanada.
   */
  private void importAsStream(File localFile, ClientProjectFile remoteFile)
    throws IOException {
    final Exchange exchange = getExchange();
    final int bufferSize = exchange.getBlockSize().getSize();
    final OutputStream prjStream = remoteFile.getOutputStream();
    final BufferedOutputStream out = new BufferedOutputStream(prjStream);
    final FileInputStream fileStream = new FileInputStream(localFile);
    BufferedInputStream in = new BufferedInputStream(fileStream);
    byte[] buffer = new byte[bufferSize];
    int len = 0;
    while ((len = in.read(buffer)) > 0) {
      exchange.addCurrentTransferSize(len);
      out.write(buffer, 0, len);
    }
    out.flush();
    out.close();
    in.close();
  }

  /**
   * Envia os dados por stream.
   * 
   * @param localFile Arquivo local a ser importado.
   * @param remoteFile Arquivo remoto.
   * @throws IOException Exceo lanada.
   */
  private void importAsUpload(File localFile, ClientProjectFile remoteFile)
    throws IOException {
    final Exchange exchange = getExchange();
    final int chunkSize = exchange.getBlockSize().getSize();
    InputStream in = new BufferedInputStream(new FileInputStream(localFile));
    remoteFile.upload(in, chunkSize, new ProjectFileChannelLoadListener() {
      @Override
      public void transferedBytes(long currentSize, long numBytes) {
        exchange.addCurrentTransferSize(numBytes);
      }
    });
    in.close();
  }

  /**
   * Construtor
   * 
   * @param exchange operao
   */
  public ExchangeImportThread(final Exchange exchange) {
    super(exchange);
  }
}
