package csbase.client.applications.flowapplication.graph;

import java.awt.geom.Rectangle2D.Double;

import com.kitfox.svg.app.beans.SVGIcon;

import csbase.client.applications.flowapplication.SVGVO;
import csbase.client.applications.flowapplication.SVGVO.HorizontalAlignment;
import csbase.client.applications.flowapplication.SVGVO.VerticalAlignment;
import tecgraf.javautils.core.lng.LNG;

/**
 * Representa uma decorao do tipo imagem de um n do grafo.
 */
public class GraphNodeImageDecoration extends
  AbstractGraphNodeDecoration<SVGVO> {

  /**
   * Tipos pr-definidos de decorao que podem ser usados pelo n.
   */
  public enum DecorationType {
    /**
     * Ocorreu uma falha no sistema que impossibilitou recuperar os dados do
     * comando.
     */
    SYSTEM_FAILURE,
    /** Aguardando sua vez no escalonador. */
    SCHEDULED,
    /**
     * Decorao que representa um algoritmo em execuo
     */
    EXECUTION,
    /** Decorao que representa um erro de execuo do algortimo **/
    EXECUTION_ERROR,
    /** Decorao que representa o sucesso de execuo do algortimo **/
    EXECUTION_SUCCESS,
    /**
     * Decorao que representa a incerteza sobre como terminou a execuo do
     * algoritmo
     **/
    EXECUTION_UNKNOWN,
    /**
     * Decorao utilizada somente em n de fluxos, que representa a interrupo
     * da execuo de um algoritmo por erro em outro algoritmo do fluxo.
     **/
    INTERRUPTED,
    /**
     * Decorao utilizada somente em n de fluxos, que representa um erro na
     * execuo do algoritmo, mas que no foi o resposvel pela interrupo do
     * fluxo.
     **/
    NON_FATAL_ERROR,
    /**
     * Decorao utilizada somente em n de fluxos, que representa um erro na
     * execuo de um algoritmo que foi o resposvel pela interrupo do fluxo.
     **/
    FATAL_ERROR,
    /**
     * Decorao utilizada somente em n de fluxos, que representa a presena de
     * alertas na execuo do algoritmo.
     **/
    EXECUTION_WARNING,
    /** Representa a opo de no ter nenhuma decorao **/
    BLANK;
  }

  /**
   * O tipo de decorao.
   */
  protected DecorationType type;

  /**
   * Construtor
   */
  public GraphNodeImageDecoration() {
    super(null, "");
    this.type = DecorationType.BLANK;
  }

  /**
   * Construtor
   * 
   * @param type Tipo da decorao
   * @param image cone da decorao
   * @param description Descrio da decorao
   */
  public GraphNodeImageDecoration(final DecorationType type, final SVGVO image,
    final String description) {
    super(image, description);
    if (image != null) {
      image.setVerticalAlignment(VerticalAlignment.CENTER);
      image.setHorizontalAlignment(HorizontalAlignment.RIGHT);
    }
    this.type = type;
  }

  /**
   * Retorna o tipo da decorao.
   * 
   * @return O tipo.
   */
  public DecorationType getType() {
    return type;
  }

  /**
   * Atribui o tipo da decorao
   * 
   * @param type O tipo
   */
  public void setType(final DecorationType type) {
    this.type = type;
  }

  /**
   * Cria a decorao de um determinado tipo pr-definido
   * 
   * @param type O tipo pr-definido
   * @return Uma decorao do tipo especificado
   */
  public static GraphNodeDecoration<SVGVO> createDecoration(
    final DecorationType type) {
    switch (type) {
      case SCHEDULED:
        return createScheduledDecoration();
      case SYSTEM_FAILURE:
        return createSystemFailureDecoration();
      case EXECUTION:
        return createExecutingDecoration();
      case EXECUTION_ERROR:
        return createExecutionErrorDecoration();
      case EXECUTION_SUCCESS:
        return createExecutionSuccessDecoration();
      case EXECUTION_UNKNOWN:
        return createExecutionUnknownDecoration();
      case EXECUTION_WARNING:
        return createExecutionWarningDecoration();
      case INTERRUPTED:
        return createInterruptedDecoration();
      case NON_FATAL_ERROR:
        return createNonFatalErrorDecoration();
      case FATAL_ERROR:
        return createFatalErrorDecoration();
      default:
        return createNullDecoration();
    }
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#SCHEDULED}
   * 
   * @return Uma decorao do tipo {@link DecorationType#SCHEDULED}
   */
  private static GraphNodeDecoration<SVGVO> createScheduledDecoration() {
    return createDecoration(DecorationType.SCHEDULED,
      GraphImages.ICON_SCHEDULED, "algorithms.tooltip.scheduled");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#INTERRUPTED}
   * 
   * @return Uma decorao do tipo {@link DecorationType#INTERRUPTED}
   */
  private static GraphNodeDecoration<SVGVO> createInterruptedDecoration() {
    return createDecoration(DecorationType.INTERRUPTED,
      GraphImages.ICON_INTERRUPTED, "algorithms.tooltip.interrupted");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#NON_FATAL_ERROR}
   * 
   * @return Uma decorao do tipo {@link DecorationType#NON_FATAL_ERROR}
   */
  private static GraphNodeDecoration<SVGVO> createNonFatalErrorDecoration() {
    return createDecoration(DecorationType.NON_FATAL_ERROR,
      GraphImages.ICON_NON_FATAL_ERROR, "algorithms.tooltip.non_fatal_error");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#FATAL_ERROR}
   * 
   * @return Uma decorao do tipo {@link DecorationType#FATAL_ERROR}
   */
  private static GraphNodeDecoration<SVGVO> createFatalErrorDecoration() {
    return createDecoration(DecorationType.FATAL_ERROR,
      GraphImages.ICON_FATAL_ERROR, "algorithms.tooltip.fatal_error");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#SYSTEM_FAILURE}
   * 
   * @return Uma decorao do tipo {@link DecorationType#SYSTEM_FAILURE}
   */
  private static GraphNodeDecoration<SVGVO> createSystemFailureDecoration() {
    return createDecoration(DecorationType.SYSTEM_FAILURE,
      GraphImages.ICON_SYSTEM_FAILURE, "algorithms.tooltip.system_failure");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#EXECUTION}
   * 
   * @return Uma decorao do tipo {@link DecorationType#EXECUTION}
   */
  private static GraphNodeDecoration<SVGVO> createExecutingDecoration() {
    return createDecoration(DecorationType.EXECUTION,
      GraphImages.ICON_EXECUTING, "algorithms.tooltip.executing");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#EXECUTION_SUCCESS}
   * 
   * @return Uma decorao do tipo {@link DecorationType#EXECUTION_SUCCESS}
   */
  private static GraphNodeDecoration<SVGVO> createExecutionSuccessDecoration() {
    return createDecoration(DecorationType.EXECUTION_SUCCESS,
      GraphImages.ICON_SUCCESS, "algorithms.tooltip.execution_end.success");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#EXECUTION_ERROR}
   * 
   * @return Uma decorao do tipo {@link DecorationType#EXECUTION_ERROR}
   */
  private static GraphNodeDecoration<SVGVO> createExecutionErrorDecoration() {
    return createDecoration(DecorationType.EXECUTION_ERROR,
      GraphImages.ICON_ERROR, "algorithms.tooltip.execution_end.error");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#EXECUTION_UNKNOWN}
   * 
   * @return Uma decorao do tipo {@link DecorationType#EXECUTION_UNKNOWN}
   */
  private static GraphNodeDecoration<SVGVO> createExecutionUnknownDecoration() {
    return createDecoration(DecorationType.EXECUTION_UNKNOWN,
      GraphImages.ICON_NO_CODE, "algorithms.tooltip.execution_end.no_code");
  }

  /**
   * Cria uma decorao do tipo {@link DecorationType#EXECUTION_WARNING}
   *
   * @return Uma decorao do tipo {@link DecorationType#EXECUTION_WARNING}
   */
 private static GraphNodeDecoration<SVGVO> createExecutionWarningDecoration() {
    return createDecoration(DecorationType.EXECUTION_WARNING,
      GraphImages.ICON_WARNING, "algorithms.tooltip.execution_end.warning");
  }

  /**
   * Cria uma decorao nula (sem imagem ou descrio)
   * 
   * @return Uma decorao nula
   */
  private static GraphNodeDecoration<SVGVO> createNullDecoration() {
    return new GraphNodeImageDecoration();
  }

  /**
   * Cria uma decorao a partir tipo, da imagem a ser usada e da chave para
   * localizao da string de descrio.
   * 
   * @param type O tipo da decorao
   * @param image A imagem que representa a decorao
   * @param key A chave para localizao da string de descrio da decorao.
   * @return Uma decorao
   */
  private static GraphNodeDecoration<SVGVO> createDecoration(
    final DecorationType type, final SVGIcon image, final String key) {
    return new GraphNodeImageDecoration(type, new SVGVO(image), LNG.get(key));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setRectangle(Double rect) {
    if (this.vo != null) {
      this.vo.setRectangle(rect);
    }
  }

}
