/*
 * $Id: DesktopTask.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.desktop;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DecimalFormat;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.Timer;
import javax.swing.border.BevelBorder;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.exception.PermissionException;

/**
 * A classe <code>DesktopTask</code> modela uma tarefa possivelmente demorada
 * executada no cliente (tipicamente uma chamada remota ao servidor). Uma thread
 *  criada para a execuo da tarefa, para que eventos de interface (como
 * redesenho) possam ser atendidos. As interaes de teclado e mouse, contudo,
 * so bloqueadas. Caso a durao da tarefa ultrapasse um valor limite (1 seg),
 * um dilogo (modal) contendo uma barra de progresso ser exibido. Este dilogo
 * ser fechado automaticamente ao final da tarefa.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class DesktopTask implements Runnable {
  /**
   * Formatador de inteiro para (hh:mm:ss) default
   */
  private static DecimalFormat integerTimeFormatter = new DecimalFormat("00");

  /**
   * Tempo de espera antes da abertura do dilogo de progresso.
   */
  private static final int PROGRESS_DIALOG_DELAY = 2000;

  /**
   * Tempo de espera entre as verificaes de trmino da tarefa.
   */
  private static final int PROGRESS_CHECK_DELAY = 1000;

  /**
   * Tempo de espera antes de oferecer boto de cancelamento.
   */
  private static final int PROGRESS_CANCEL_DELAY = 30000;

  /**
   * Dilogo de espera, indicando atividade.
   */
  private JDialog progressDialog;

  /**
   * Resultado gerado pela tarefa
   */
  protected Object result = null;

  /**
   * Exceo gerada pela tarefa
   */
  protected Exception exception = null;

  /**
   * Label de exibio de tempo
   */
  private final JTextField elapsedText = new JTextField();

  /**
   * Texto de exibio de passo corrente
   */
  private final JTextField stepText = new JTextField();

  /**
   * Texto de exibio de tempo total
   */
  private final JTextField totalText = new JTextField();

  /**
   * Label de exibio de tempo
   */
  private final JLabel elapsedLabel = new JLabel();

  /**
   * Label de exibio de passo corrente
   */
  private final JLabel stepLabel = new JLabel();

  /**
   * Label de exibio de tempo total
   */
  private final JLabel totalLabel = new JLabel();

  /**
   * Tempo (s) decorrido
   */
  private long elapsedSeconds = 0L;

  /** Ttulo da tarefa */
  protected String taskTitle = null;

  /**
   * Janela
   */
  private Window window;

  /**
   * Barra de progresso
   */
  final JProgressBar progress = new JProgressBar();

  /**
   * Cria um dilogo modal contendo uma mensagem e uma barra de progresso de
   * durao "indeterminada".
   * 
   * @param wnd janela "base" do dilogo criado
   * @param title ttulo do dilogo
   * @param msg mensagem a ser exibida
   */
  private void createProgressDialog(Window wnd, String title, String msg) {
    progressDialog =
      (wnd instanceof Dialog) ? new JDialog((Dialog) wnd, title, true)
        : new JDialog((Frame) wnd, title, true);
    progress.setIndeterminate(true);
    progress.setStringPainted(false);
    Container cp = progressDialog.getContentPane();
    cp.setLayout(new BorderLayout());
    JPanel mainPanel = new JPanel();
    mainPanel.setLayout(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.CENTER;
    c.gridy = 0;
    c.gridx = 0;
    c.insets = new Insets(20, 10, 10, 15);
    mainPanel.add(new JLabel(msg), c);
    elapsedText.setFont(new Font("Monospaced", Font.PLAIN, 12));
    elapsedText.setEditable(false);
    elapsedText.setHorizontalAlignment(JTextField.RIGHT);
    elapsedLabel.setText(LNG.get("UTIL_ELAPSED_TIME"));
    totalText.setFont(new Font("Monospaced", Font.PLAIN, 12));
    totalText.setEditable(false);
    totalText.setHorizontalAlignment(JTextField.RIGHT);
    totalLabel.setText(LNG.get("UTIL_TOTAL_TIME"));
    stepText.setFont(new Font("Monospaced", Font.PLAIN, 12));
    stepText.setEditable(false);
    stepText.setText("");
    stepLabel.setText(LNG.get("UTIL_STEP_TEXT"));
    JComponent[][] components =
      { { elapsedLabel, elapsedText, null }, { totalLabel, totalText, null },
          { stepLabel, stepText, null }, };
    JPanel timePanel = GUIUtils.createBasicGridPanel(components);
    timePanel.setBorder(BorderFactory.createBevelBorder(BevelBorder.LOWERED));
    c.gridy = 1;
    c.anchor = GridBagConstraints.WEST;
    c.insets = new Insets(5, 10, 5, 5);
    mainPanel.add(timePanel, c);
    c.gridy = 2;
    c.insets = new Insets(10, 10, 5, 5);
    c.anchor = GridBagConstraints.CENTER;
    mainPanel.add(progress, c);
    cp.add(mainPanel, BorderLayout.CENTER);
    progressDialog.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
    progressDialog.setResizable(false);
    progressDialog.pack();
    Dimension wSize = progressDialog.getSize();
    Dimension sSize = Toolkit.getDefaultToolkit().getScreenSize();
    int newX = (sSize.width - wSize.width) / 3;
    int newY = (sSize.height - wSize.height) / 2;
    setStepText(null);
    setUnknownStatus();
    progressDialog.setLocation(newX, newY);
  }

  /**
   * Tarefa a ser executada.
   * 
   * @throws Exception exceo.
   */
  protected abstract void performTask() throws Exception;

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public void run() {
    try {
      performTask();
    }
    catch (PermissionException pe) {
      /*
       * Soluo temporria para que PermissionException seja tratada de forma
       * diferenciada e apenas um dilogo (este) seja mostrado
       */
      exception = null;
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          String msg = LNG.get("desktop.msg.permissionfailure");
          StandardDialogs.showInfoDialog(DesktopTask.this.window, taskTitle,
            msg);
        }
      });
    }
    catch (java.rmi.ServerException se) {
      ClientRemoteMonitor.getInstance().invalidate();
      exception = se;
    }
    catch (Exception e) {
      ClientRemoteMonitor.getInstance().invalidate();
      exception = e;
    }
    catch (Throwable t) {
      exception = new Exception(t);
    }
  }

  /**
   * Obtm a exceo gerada pela tarefa.
   * 
   * @return a exceo
   */
  public Exception getException() {
    return exception;
  }

  /**
   * Coloca o valor de resultado da tarefa executada.
   * 
   * @param result o valor de resultado da tarefa
   */
  protected void setResult(Object result) {
    this.result = result;
  }

  /**
   * Obtm o valor de resultado da tarefa executada. Este mtodo ser
   * tipicamente redefinido nas especializaes de <code>DesktopTask</code>.
   * 
   * @return o valor de resultado da tarefa
   */
  public Object getResult() {
    return result;
  }

  /**
   * Verifica se a tarefa foi concluida com sucesso; este mtodo ser
   * tipicamente redefinido nas especializaes de <code>DesktopTask</code>.
   * 
   * @return <code>true</code> em caso de sucesso, <code>false</code> caso
   *         contrrio.
   */
  public boolean succeeded() {
    return (exception == null);
  }

  /**
   * Ajuste do texto de passo.
   * 
   * @param text o texto
   */
  public void setStepText(final String text) {
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        if (text == null) {
          stepLabel.setVisible(false);
          stepText.setVisible(false);
          stepText.setText("");
        }
        else {
          stepLabel.setVisible(true);
          stepText.setVisible(true);
          stepText.setText(text);
        }
        if (progressDialog != null) {
          progressDialog.pack();
        }
      }
    });
  }

  /**
   * Ajuste do percentual
   * 
   * @param perc o percentual
   */
  public void setProgressStatus(final int perc) {
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        totalLabel.setVisible(true);
        totalText.setVisible(true);
        progress.setMinimum(0);
        progress.setMaximum(100);
        progress.setIndeterminate(false);
        progress.setValue(perc);
        progress.setStringPainted(true);
        int c = 200 - (int) (200.0 * perc / 100.0);
        int b = 255 - (int) (150.0 * perc / 100.0);
        progress.setForeground(new Color(c, c, b));
        if (progressDialog != null) {
          progressDialog.pack();
        }
      }
    });
  }

  /**
   * 
   */
  public void setUnknownStatus() {
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        totalLabel.setVisible(false);
        totalText.setVisible(false);
        progress.setIndeterminate(true);
        progress.setStringPainted(false);
        if (progressDialog != null) {
          progressDialog.pack();
        }
      }
    });
  }

  /**
   * Converso de inteiro para string com dois dgitos.
   * 
   * @param v a quantidade.
   * 
   * @return uma string formatada.
   */
  private String timeFormat(final long v) {
    return integerTimeFormatter.format(v);
  }

  /**
   * Conversao de segundos para uma string de tempo.
   * 
   * @param seconds a quantidade de segundos.
   * 
   * @return uma string formatada.
   */
  private String secToString(long seconds) {
    if (seconds < 0) {
      return "-";
    }
    long sec = seconds;
    long min = 0;
    long hou = 0;
    if (sec >= 60) {
      min = (int) sec / 60;
      sec = sec - (min * 60);
    }
    if (min >= 60) {
      hou = (int) min / 60;
      min = min - (hou * 60);
    }
    return timeFormat(hou) + "h " + timeFormat(min) + "m " + timeFormat(sec)
      + "s";
  }

  /**
   * Inicia a execuo da tarefa.
   * 
   * @param parentWindow janela base a partir da qual a da tarefa foi comandada
   * @param title ttulo do dilogo de progresso (caso seja exibido)
   * @param taskMessage mensagem no dilogo de progresso (caso seja exibido)
   */
  public void start(Window parentWindow, String title, String taskMessage) {
    elapsedSeconds = 0;
    this.taskTitle = title;
    if (parentWindow == null) {
      this.window = DesktopFrame.getInstance().getDesktopFrame();
    }
    else {
      this.window = parentWindow;
    }
    try {
      this.window.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      final Thread worker = new Thread(this);
      worker.start();
      try {
        worker.join(PROGRESS_DIALOG_DELAY);
      }
      catch (InterruptedException ie) {
      }
      if (!worker.isAlive()) {
        return;
      }
      createProgressDialog(this.window, title, taskMessage);
      final Thread elapsedTimer = new Thread() {
        @Override
        public void run() {
          while (worker.isAlive()) {
            try {
              SwingThreadDispatcher.invokeLater(new Runnable() {
                @Override
                public void run() {
                  elapsedText.setText(secToString(elapsedSeconds));
                  if (progress.isIndeterminate()) {
                    totalText.setText("???");
                  }
                  else {
                    final int perc = progress.getValue();
                    final long es = elapsedSeconds;
                    long ts = -999;
                    if (perc != 0) {
                      ts = (int) (es * 100.0 / perc);
                    }
                    totalText.setText(secToString(ts));
                  }
                }
              });
              Thread.sleep(1000);
              elapsedSeconds++;
            }
            catch (Throwable ex) {
            }
          }
        }
      };
      elapsedTimer.start();
      final Timer closeDialogTimer = new Timer(PROGRESS_CHECK_DELAY, null);
      closeDialogTimer.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (!worker.isAlive()) {
            if (progressDialog != null) {
              progressDialog.setVisible(false);
              progressDialog.dispose();
            }
            closeDialogTimer.stop();
          }
        }
      });
      closeDialogTimer.start();
      final Timer showCancelTimer = new Timer(PROGRESS_CANCEL_DELAY, null);
      showCancelTimer.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (worker.isAlive()) {
            Container cp = progressDialog.getContentPane();
            JButton cancelButton = new JButton(LNG.get("UTIL_CANCEL"));
            cancelButton.addActionListener(new ActionListener() {
              @Override
              public void actionPerformed(ActionEvent ev) {
                try {
                  worker.interrupt();
                }
                catch (Exception ex) {
                } // Applet
                progressDialog.setVisible(false);
                progressDialog.dispose();
                exception = new Exception(LNG.get("UTIL_USER_INTERRUPT"));
              }
            });

            JPanel buttonPanel = new JPanel();
            buttonPanel.add(cancelButton);
            cp.add(buttonPanel, BorderLayout.SOUTH);
            cp.invalidate();
            progressDialog.pack();
          }
        }
      });
      showCancelTimer.setRepeats(false);
      showCancelTimer.start();
      progressDialog.setVisible(true);
    }
    finally {
      this.window.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    }
  }
}
