/*
 * $Id: DirectoryContentsPanel.java 176168 2016-09-22 21:12:51Z fpina $
 */
package csbase.client.desktop.dircontents;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SortOrder;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectFileCellRenderer;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.project.ProjectTreeSelectionEvent;
import csbase.client.project.ProjectTreeSelectionListener;
import csbase.client.project.tasks.FileTypeIconCellRenderer;
import csbase.client.project.tasks.GetChildrenTask;
import csbase.client.util.DateTableCellRenderer;
import csbase.client.util.SizeTableCellRenderer;
import csbase.logic.ClientProjectFile;
import csbase.logic.ClientProjectFileComparator;
import csbase.logic.CommonClientProject;
import csbase.logic.FileTypeComparator;
import csbase.logic.NoHiddenFileFilter;
import csbase.logic.ProjectFileFilter;
import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * Cria um painel que exibe o contedo de um diretrio, toda vez que o usurio
 * selecionar algum na rvore de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DirectoryContentsPanel extends JPanel {

  /**
   * Formatador para as colunas da tabela.
   */
  private static class MyObjectTableProvider implements ObjectTableProvider {
    /**
     * Classe para cada coluna.
     */
    private static final Class<?>[] COL_CLASSES = { ImageIcon.class,
        ClientProjectFile.class, Long.class, Date.class };
    /**
     * Nomes para cada coluna.
     */
    private static final String[] COL_NAMES = new String[] { "", LNG.get(
      "DIR_CONTENTS_NAME"), LNG.get("DIR_CONTENTS_SIZE"), LNG.get(
        "DIR_CONTENTS_DATE") };

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getCellValue(final Object row, final int col) {
      final ClientProjectFile file = (ClientProjectFile) row;
      switch (col) {
        case 0:
          //          ImageIcon icon = ClientProjectFileProperties.getImageIcon(file);
          //          icon.setDescription(getType(file));
          return file;

        case 1:
          return file;

        case 2:
          return getSize(file);

        case 3:
          return getDate(file);
      }
      return null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getColumnClasses() {
      return MyObjectTableProvider.COL_CLASSES;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getColumnNames() {
      return MyObjectTableProvider.COL_NAMES;
    }

    /**
     * Obtm a data da ltima modificao do arquivo.
     * 
     * @param file arquivo.
     * 
     * @return data da ltima modificao do arquivo.
     */
    private Date getDate(final ClientProjectFile file) {
      return new Date(file.getModificationDate());
    }

    /**
     * Obtm o tamanho do arquivo.
     * 
     * @param file arquivo.
     * 
     * @return tipo do arquivo.
     */
    private Long getSize(final ClientProjectFile file) {
      if (file.isDirectory()) {
        return (long) -1;
      }
      return file.size();
    }
  }

  /**
   * ndice da coluna de cones que representam o tipo de aqruivo.
   */
  static final int TYPE_COL_INDEX = 0;

  /**
   * ndice da coluna que contm nome de aqruivo.
   */
  static final int NAME_COL_INDEX = 1;

  /**
   * ndice da coluna que contm nome de aqruivo.
   */
  static final int SIZE_COL_INDEX = 2;

  /**
   * ndice da coluna que contm nome de aqruivo.
   */
  static final int DATE_COL_INDEX = 3;

  /**
   * Tabela que exibe lista de arquivos de um diretrio
   */
  private SortableTable dirContentsTable;

  /**
   * Modelo para a tabela de contedo de um diretrio
   */
  private ObjectTableModel<ClientProjectFile> dirModel;

  /**
   * Barra de status do diretrio
   */
  private JLabel statusBar;

  /**
   * ltimo texto exibido na barra de status
   */
  private String lastStatusBarText = " ";

  /**
   * Seleo anterior de arquivo ou diretrio.  usado para controlar a
   * atualizao da SortableTable.
   */
  private ClientProjectFile previousSelectedFile;

  /** */
  private ProjectTree projectTree;

  /** */
  private JScrollPane scrollPane;

  /**
   * Filtro para os arquivos exibidos na tabela do painel.
   */
  private ProjectFileFilter filter;

  /**
   * Limpa a tabela e a barra de status
   */
  private void clear() {
    clearModel();
    lastStatusBarText = " ";
    statusBar.setText(lastStatusBarText);
  }

  /**
   * Limpa o modelo, na EDT.
   */
  private void clearModel() {
    SwingThreadDispatcher.invokeLater(new Runnable() {

      @Override
      public void run() {
        dirModel.clear();
      }
    });
  }

  /**
   * Exibe o contedo do diretrio/arquivo selecionado pelo usurio no painel de
   * detalhes. S atualiza a tabela se o arquivo selecionado mudar.
   * 
   * @param file diretrio selecionado pelo usurio
   */
  private void display(final ClientProjectFile file) {
    if (file.isDirectory()) {
      displayDir(file);
    }
    else {
      displayFile(file);
    }
    statusBar.setText(lastStatusBarText);
  }

  /**
   * Exibe na status dados dos arquivso selecionados.
   * 
   * @param files lista de arquivos
   */
  private void display(final ClientProjectFile[] files) {
    final List<ClientProjectFile> fileList = new ArrayList<ClientProjectFile>();
    for (final ClientProjectFile file : files) {
      fileList.add(file);
    }
    updateModel(fileList);
    lastStatusBarText = getStatusBarTextForFiles(files);
    statusBar.setText(lastStatusBarText);
  }

  /**
   * Exibe o diretrio com o ".." e os arquivos do diretrio dir dado.
   * 
   * @param dir diretrio
   */
  private void displayDir(final ClientProjectFile dir) {
    final ArrayList<ClientProjectFile> dirList =
      new ArrayList<ClientProjectFile>();
    final ClientProjectFile[] children = GetChildrenTask.runTask(dir);
    final NoHiddenFileFilter noDotFileFilter = NoHiddenFileFilter.getInstance();
    final boolean showHiddenFiles = DesktopFrame.getInstance()
      .shouldShowHiddenFiles();
    long size = 0L;
    for (final ClientProjectFile element : children) {
      final ClientProjectFile child = element;
      if (!showHiddenFiles && !noDotFileFilter.accept(child)) {
        continue;
      }
      if (filter == null || filter.accept(child)) {
        dirList.add(child);
        if (!child.isDirectory()) {
          size += child.size();
        }
      }
    }
    updateModel(dirList);
    lastStatusBarText = getStatusBarTextForFiles(dirList.size(), size);
  }

  /**
   * Exibe dado (na status) do arquivo
   * 
   * @param file o arquivo
   */
  private void displayFile(final ClientProjectFile file) {
    final List<ClientProjectFile> fileList = new ArrayList<ClientProjectFile>();
    fileList.add(file);
    updateModel(fileList);
    lastStatusBarText = getStatusBarTextForFile(file);
  }

  /**
   * Consulta string de seleo de arquivo
   * 
   * @param file o arquivo
   * @return o texto
   */
  private String getStatusBarTextForFile(final ClientProjectFile file) {
    final long fileSize = file.size();
    final String formattedFileSize = FormatUtils.formatSize(fileSize, 2);
    return MessageFormat.format(LNG.get("DIR_CONTENTS_STATUS_BAR"), 1,
      formattedFileSize);
  }

  /**
   * Consulta string de seleo de mltiplos arquivos
   * 
   * @param files a lista de arquivos
   * @return o texto
   */
  private String getStatusBarTextForFiles(final ClientProjectFile[] files) {
    final int fileCount = files.length;

    // Calcula o volume total do diretrio para exibio na barra de status
    try {
      long dirSize = 0;
      for (int i = 0; i < fileCount; i++) {
        final ClientProjectFile file = files[i];
        if (!file.isDirectory()) {
          dirSize += file.size();
        }
      }
      return getStatusBarTextForFiles(fileCount, dirSize);
    }
    catch (final Exception e) {
      e.printStackTrace();
      return LNG.get("DIR_CONTENTS_STATUS_BAR_ERROR");
    }
  }

  /**
   * Obtm o texto para a status bar para um determinado nmero de arquivos
   * totalizando um determinado tamanho.
   * 
   * @param fileCount nmero de arquivos
   * @param totalSize tamanho acumulado dos arquivos
   * @return texto para a status bar
   */
  private String getStatusBarTextForFiles(final int fileCount,
    final long totalSize) {
    final String formattedDirSize = FormatUtils.formatSize(totalSize, 2);
    return MessageFormat.format(LNG.get("DIR_CONTENTS_STATUS_BAR"), fileCount,
      formattedDirSize);
  }

  /**
   * <p>
   * Atribui o filtro para a tabela de arquivos deste painel.
   * </p>
   * ATENO:  importante que este mtodo seja chamado antes da rvore ser
   * filtrada, pois ela ir imediatamente notificar esta classe para se
   * redesenhar, e no momento do redesenho o filtro j dever estar atribudo.
   * 
   * @param filter novo filtro.
   */
  public void setFilter(final ProjectFileFilter filter) {
    this.filter = filter;
  }

  /**
   * Atualiza os arquivos selecionados.
   * 
   * @see #update(ClientProjectFile[], boolean)
   */
  private void update() {
    this.update(DirectoryContentsPanel.this.projectTree.getSelectedFiles(),
      true);
  }

  /**
   * Atualiza o contedo da tabela se a seleo mudou ou se recebeu um evento da
   * rvore sobre alterao do modelo.
   * 
   * @param selectedFiles a seleo
   * @param refresh indicativo de atualizao
   */
  private void update(final ClientProjectFile[] selectedFiles,
    final boolean refresh) {
    if (selectedFiles == null || selectedFiles.length == 0) {
      this.clear();
      previousSelectedFile = null;
    }
    else {
      if (selectedFiles.length == 1) {
        final ClientProjectFile selFile = selectedFiles[0];
        if (refresh || !selFile.equals(previousSelectedFile)) {
          display(selFile);
          previousSelectedFile = selFile;
        }
      }
      else {
        display(selectedFiles);
        previousSelectedFile = null;
      }
    }
  }

  /**
   * Atualiza o modelo da tabela, na EDT.
   * 
   * @param fileList lista de linhas
   */
  private void updateModel(final List<ClientProjectFile> fileList) {
    SwingThreadDispatcher.invokeLater(new Runnable() {

      @Override
      public void run() {
        dirModel.setRows(fileList);
      }
    });
  }

  /**
   * Construtor.
   * 
   * @param projectTree tree
   */
  public DirectoryContentsPanel(final ProjectTree projectTree) {
    this.filter = null;
    this.projectTree = projectTree;
    this.projectTree.addTreeModelListener(new TreeModelListener() {
      @Override
      public void treeNodesChanged(final TreeModelEvent e) {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void treeNodesInserted(final TreeModelEvent e) {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void treeNodesRemoved(final TreeModelEvent e) {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void treeStructureChanged(final TreeModelEvent e) {
        DirectoryContentsPanel.this.update();
      }
    });
    this.projectTree.addTreeSelectionListener(
      new ProjectTreeSelectionListener() {
        @Override
        public void update(final ProjectTreeSelectionEvent event) {
          DirectoryContentsPanel.this.update(event.getSelectedFiles(), false);
        }
      });

    // Cria o formato da tabela
    final ObjectTableProvider provider = new MyObjectTableProvider();

    // Cria o modelo da tabela
    dirModel = new ObjectTableModel<ClientProjectFile>(
      new ArrayList<ClientProjectFile>(), provider);
    dirContentsTable = new SortableTable(dirModel);
    dirContentsTable.setNoSortStateEnabled(true);

    // Seta o comparator para cone que representa o tipo do arquivo
    final FileTypeComparator iconComparator = new FileTypeComparator();
    dirContentsTable.setComparator(DirectoryContentsPanel.TYPE_COL_INDEX,
      iconComparator);

    // Seta o comparator para ClientProjectFile
    final ClientProjectFileComparator comparator =
      new ClientProjectFileComparator();
    dirContentsTable.setComparator(DirectoryContentsPanel.NAME_COL_INDEX,
      comparator);

    // Ordena por nome 
    dirContentsTable.sort(DirectoryContentsPanel.NAME_COL_INDEX,
      SortOrder.ASCENDING);

    // Renderizador da coluna "data"
    dirContentsTable.setDefaultRenderer(Date.class,
      new DateTableCellRenderer());

    // Renderizador da coluna "tamanho"
    dirContentsTable.setDefaultRenderer(Long.class,
      new SizeTableCellRenderer());

    // Renderizador da coluna "nome"
    final TableColumnModel columnModel = dirContentsTable.getColumnModel();
    final TableColumn nameColumn = columnModel.getColumn(
      DirectoryContentsPanel.NAME_COL_INDEX);
    nameColumn.setCellRenderer(new ProjectFileCellRenderer());

    // Renderizador da coluna de cones que representam o tipo do arquivo
    final TableColumn typeColumn = columnModel.getColumn(
      DirectoryContentsPanel.TYPE_COL_INDEX);
    typeColumn.setCellRenderer(new FileTypeIconCellRenderer());

    // Definio de tamanho fixo para a coluna de cones que representam o 
    // tipo do arquivo
    typeColumn.setMaxWidth(25);
    typeColumn.setMinWidth(25);
    typeColumn.setPreferredWidth(25);

    // Definio de tamanho fixo para a coluna de tamanho
    final TableColumn sizeColumn = columnModel.getColumn(
      DirectoryContentsPanel.SIZE_COL_INDEX);
    sizeColumn.setPreferredWidth(80);
    sizeColumn.setMaxWidth(80);

    // Definio de tamanho fixo para a coluna de data
    final TableColumn dateColumn = columnModel.getColumn(
      DirectoryContentsPanel.DATE_COL_INDEX);
    dateColumn.setPreferredWidth(150);
    dateColumn.setMaxWidth(150);

    // Remove as linhas da tabela
    dirContentsTable.setShowGrid(false);
    dirContentsTable.setRowMargin(0);

    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectChanged(final CommonClientProject project) {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void projectClosed(final CommonClientProject project) {
        /*
         * se o projeto fechado for o que estamos visualizando, no precisamos
         * atualizar o painel, basta o clear()
         */
        DirectoryContentsPanel.this.clear();
      }

      @Override
      public void projectInfoModified() {
        DirectoryContentsPanel.this.update();
      }

      @Override
      public void projectRemoved() {
        DirectoryContentsPanel.this.clear();
      }
    });
    scrollPane = new JScrollPane(dirContentsTable);
    scrollPane.getViewport().setBackground(Color.WHITE);
    statusBar = new JLabel();
    statusBar.setText(lastStatusBarText);
    setLayout(new BorderLayout());
    add(scrollPane, BorderLayout.CENTER);
    add(statusBar, BorderLayout.SOUTH);
    setPreferredSize(new Dimension(220, 100));
  }
}
