package csbase.client.util;

import java.awt.Window;
import java.rmi.RemoteException;
import java.text.MessageFormat;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JTextField;

import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.client.project.ProjectFileChooser;
import csbase.client.project.ProjectFileChooserOpen;
import csbase.client.project.ProjectFileChooserSave;
import csbase.client.project.ProjectTreePath;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.Utilities;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe que representa um componente para busca e validao de arquivos e/ou
 * diretrios no projeto. Oferece um JTextField (obrigatrio), um JLabel
 * (opcional) e um JButton (opcional) com a imagem de um binculo, que podem ser
 * posicionados de acordo com as preferncias de layout. O componentes pode
 * apresentar 5 comportamentos diferentes, definidos pela propriedade mode,
 * definida no construtor ou atravs do mtodo setMode(). De acordo com o modo
 * de operao escolhido, o componente ir validar as entradas do usurio no
 * campo texto, assim como definir filtros para quais tipos de
 * arquivos/diretrios podero ser escolhidas em uma caixa de dilogo, exibida
 * quando o usurio clica no boto. Para isso,  necessrio que seja passada uma
 * referncia para o projeto corrente atravs do mtodo
 * <code>setProject()</code>. Para ativar o dilogo e recuperar a(s) opo(es)
 * selecionada(s) pelo usurio, deve-se usar o mtodo
 * <code>getButtonPath()</code> ou <code>getButtonPaths()</code>. Para validar o
 * texto entrado no campo e recuperar o(s) caminho(s) lgico(s) escolhido(s)
 * pelo usurio, deve-se usar os mtodos <code>getFieldPath()</code> ou
 * <code>getFieldPaths()</code>.
 */
public class FileTextField {
  /** Rtulo a ser exibido ao lado do campo */
  private JLabel label;

  /** Representa o campo texto */
  private JTextField field;

  /** Boto a ser exibido junto ao campo texto */
  private JButton button;

  /** Indica se o componente deve ou no exibir mensagens em caso de erro. */
  private boolean complain = true;

  /**
   * Projeto no qual o componente pesquisar para validar as entradas do
   * usurio.
   */
  private CommonClientProject project;

  /**
   * Extenso opcional para o arquivo, usada para um filtro preferencial no
   * ProjectFileChooser (o que no impede o usurio de escolher outra extenso).
   * Caso o objeto esteja operando em modo ProjectFileChooser.SAVE_FILE, e o
   * usurio no tenha especificado nenhuma extenso ao fornecer o nome do
   * arquivo de sada, esta extenso preferencial ser acrescentada ao arquivo.
   */
  private String extension;
  private int mode;
  private Window window;
  private boolean isSaveMode;

  /**
   * Obtm o campo texto.
   * 
   * @return campo texto.
   */
  public JTextField getField() {
    return this.field;
  }

  /**
   * Obtm o texto do campo.
   * 
   * @return o texto do campo.
   */
  public String getText() {
    return this.field.getText();
  }

  /**
   * Obtm o rtulo associado ao campo texto.
   * 
   * @return o rtulo.
   */
  public JLabel getLabel() {
    return this.label;
  }

  /**
   * Obtm o boto de navegao na rvore de diretrios.
   * 
   * @return o boto de navegao
   */
  public JButton getButton() {
    return this.button;
  }

  /**
   * Obtm o caminho para um arquivo/diretrio a partir de uma seleo do
   * usurio em um dilogo ProjectFileChooser.
   * 
   * @return caminho para o arquivo/diretrio selecionado.
   * 
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto.
   */
  public ProjectTreePath getButtonPath() throws ClientException {
    ProjectFileChooser chooser;
    if (this.isSaveMode) {
      chooser = new ProjectFileChooserSave(this.window, this.project, this.mode,
        this.extension);
    }
    else {
      chooser = new ProjectFileChooserOpen(this.window, this.project, false,
        this.mode, this.extension);
    }
    return chooser.getSelectedPath();
  }

  /**
   * Obtm os caminhos para todos os arquivos/diretrios que o usurio
   * selecionar em um dilogo ProjectFileChooser.
   * 
   * @return array de caminhos para todos os arquivos/diretrios selecionados.
   * 
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto.
   */
  public ProjectTreePath[] getButtonPaths() throws ClientException {
    ProjectFileChooser chooser;
    if (this.isSaveMode) {
      chooser = new ProjectFileChooserSave(this.window, this.project, this.mode,
        this.extension);
    }
    else {
      chooser = new ProjectFileChooserOpen(this.window, this.project, false,
        this.mode, this.extension);
    }
    return chooser.getSelectedPaths();
  }

  /**
   * Recupera o arquivo/diretrio solicitado pelo usurio a partir do texto de
   * entrada, validando a existncia deste na rvore de diretrios (projeto)
   * passada.
   * 
   * @param textPath texto fornecido pelo usurio, representando um caminho no
   *        projeto.
   * @param dir rvore de diretrios (projeto) a ser pesquisada.
   * 
   * @return caminho "simples" para o arquivo/diretrio.
   * 
   * @throws ClientException
   */
  private ClientProjectFile retrieveFile(final String textPath,
    final ClientProjectFile dir) throws ClientException {
    final String[] simplePath = Utilities.splitProjectPath(textPath);
    if (simplePath == null) {
      throw new ClientException("UTIL_INVALID_PATH");
    }

    /*
     * Verifica, um a um, cada componente do path (menos o ltimo) para garantir
     * que todos sejam diretrios.
     */
    final RemoteTask<ClientProjectFile> task =
      new RemoteTask<ClientProjectFile>() {
        @Override
        public void performTask() throws Exception {
          ClientProjectFile d = dir;
          for (int i = 0; i < (simplePath.length - 1); i++) {
            final String pth = simplePath[i];
            d = d.getChild(pth);
            if ((d == null) || !d.isDirectory()) {
              throw new ClientException("UTIL_INVALID_PATH");
            }
          }

          /* Verifica a existncia do ltimo componente */
          final String lastPath = simplePath[simplePath.length - 1];
          setResult(d.getChild(lastPath));
        }
      };

    task.execute(null, "", "");
    final Exception e = task.getError();
    if (e instanceof ClientException) {
      throw (ClientException) e;
    }

    return task.getResult();
  }

  /**
   * Constri um caminho "completo" (relativo  raiz de todos os projetos) para
   * um arquivo a ser criado.
   * 
   * @param textPath texto fornecido pelo usurio, representando um caminho no
   *        projeto.
   * @param dir rvore de diretrios (projeto) a ser pesquisada.
   * 
   * @return caminho "completo" para o novo arquivo.
   * 
   * @throws RemoteException eventualmente nos acessos aos filhos de dir com o
   *         mtodo {@link ClientProjectFile#getChild(String)}
   */
  private String[] makeFilePath(String textPath, ClientProjectFile dir)
    throws RemoteException {
    String[] filePath;
    String[] simplePath = Utilities.splitProjectPath(textPath);
    String[] rootPath = dir.getPath();

    /*
     * Constri uma estrutura que representa o caminho completo para o arquivo,
     * necessria para sua criao (ProjectFileChooser.SAVE_FILE).
     */
    filePath = new String[rootPath.length + simplePath.length];
    System.arraycopy(rootPath, 0, filePath, 0, rootPath.length);
    for (int i = 0; i < (simplePath.length - 1); i++) {
      dir = dir.getChild(simplePath[i]);
      filePath[rootPath.length + i] = simplePath[i];
    }
    filePath[filePath.length - 1] = simplePath[simplePath.length - 1];
    String fileName = filePath[filePath.length - 1];
    if ((fileName.indexOf(".") == -1) && (extension != null)) {
      filePath[filePath.length - 1] = fileName + "." + extension.toLowerCase();
    }
    return filePath;
  }

  /**
   * Obtm o caminho para um diretrio.
   * 
   * @param textPath representando o caminho para o diretrio.
   * 
   * @return caminho para o diretrio.
   */
  private ProjectTreePath getDirectoryPath(String textPath) {
    ClientProjectFile dir = project.getRoot();
    if (textPath.equals("/")) {
      return new ProjectTreePath(dir);
    }
    ClientProjectFile file = null;
    try {
      file = retrieveFile(textPath, dir);
    }
    catch (ClientException e) {
      signalError(e.getMessage());
      return null;
    }
    if (file != null) { /* ltimo componente j existente */
      if (!file.isDirectory()) {
        signalError("UTIL_NOT_A_DIRECTORY", file.getName());
        return null;
      }
      else {
        return new ProjectTreePath(file);
      }
    }
    else { /* ltimo componente do path no existe */
      signalError("UTIL_INVALID_DIR_PATH");
      return null;
    }
  }

  /**
   * Obtm o caminho para um arquivo de entrada.
   * 
   * @param textPath representando o caminho para o arquivo.
   * 
   * @return o caminho para o arquivo.
   */
  private ProjectTreePath openFilePath(String textPath) {
    ClientProjectFile dir = project.getRoot();
    if (textPath.equals("/")) {
      signalError("UTIL_UNSPECIFIED_FILE");
      return null;
    }
    ClientProjectFile file = null;
    try {
      file = retrieveFile(textPath, dir);
    }
    catch (ClientException e) {
      signalError(e.getMessage());
      return null;
    }
    if (file != null) { /* ltimo componente j existente */
      if (file.isDirectory()) {
        signalError("UTIL_NOT_A_FILE", file.getName());
        return null;
      }
      else { /*  arquivo */
        return new ProjectTreePath(file);
      }
    }
    else { /* ltimo componente do path no existe */
      signalError("UTIL_INVALID_PATH");
      return null;
    }
  }

  /**
   * Retorna um erro para o usurio (caso a flag complain esteja ativada).
   * 
   * @param errorKey chave para a mensagem a ser exibida, a ser recuperada do
   *        arquivo de propriedades.
   */
  private void signalError(String errorKey) {
    signalError(errorKey, null);
  }

  /**
   * Retorna um erro para o usurio (caso a flag complain esteja ativada), com
   * um parmetro configurvel (usado para mensagens dinmicas).
   * 
   * @param errorKey chave para a mensagem a ser exibida, a ser recuperada do
   *        arquivo de propriedades.
   * @param param parmetro a ser includo no texto da mensagem dinmica.
   */
  private void signalError(String errorKey, String param) {
    if (complain) {
      String error = LNG.get(errorKey);
      if (param != null) {
        Object[] params = { param };
        error = MessageFormat.format(error, params);
      }
      ClientUtilities.showErrorByTextField(null, error, getLabel(), getField());
    }
  }

  /**
   * Obtm (e valida) o caminho para o arquivo/diretrio digitado pelo usurio
   * no campo texto.
   * 
   * @return caminho para o arquivo/diretrio.
   */
  public ProjectTreePath getFieldPath() {
    ProjectTreePath path = null;
    String trimmedText = getField().getText().trim();
    if (isEmpty(trimmedText)) {
      if (mode == ProjectFileChooser.DIRECTORY_ONLY) {
        signalError("UTIL_UNSPECIFIED_DIR");
      }
      else {
        signalError("UTIL_UNSPECIFIED_FILE");
      }
      return null;
    }
    if (!ClientUtilities.validatePathChars(trimmedText)) {
      signalError("UTIL_CHARACTER_ERROR");
      return null;
    }
    switch (mode) {
      case ProjectFileChooser.DIRECTORY_ONLY:
        path = getDirectoryPath(trimmedText);
        break;
      case ProjectFileChooser.FILE_ONLY:
        if (this.isSaveMode) {
          path = saveFilePath(trimmedText);
        }
        else {
          path = openFilePath(trimmedText);
        }
        break;
      default:
        return null;
    }
    return path;
  }

  /**
   * DOCUMENT ME!
   * 
   * @param textPath
   * 
   * @return o path
   */
  private ProjectTreePath saveFilePath(final String textPath) {
    if (textPath.equals("/")) {
      signalError("UTIL_UNSPECIFIED_FILE");
      return null;
    }

    final ClientProjectFile dir = project.getRoot();
    ClientProjectFile file = null;
    try {
      file = retrieveFile(textPath, dir);
    }
    catch (ClientException e) {
      signalError(e.getMessage());
      return null;
    }
    if (file != null) { /* ltimo componente j existente */
      if (file.isDirectory()) {
        signalError("UTIL_NOT_A_FILE", file.getName());
        return null;
      }
      return new ProjectTreePath(file);
    }

    final RemoteTask<String[]> task = new RemoteTask<String[]>() {
      @Override
      public void performTask() throws RemoteException {
        String[] filePath = makeFilePath(textPath, dir);
        setResult(filePath);
      }
    };
    task.execute(window, null, null);

    if (!task.getStatus()) {
      return null;
    }

    final String[] filePath = task.getResult();
    return new ProjectTreePath(filePath, dir);
  }

  /**
   * Verifica se o texto fornecido est vazio.
   * 
   * @param text texto a ser verificado.
   * 
   * @return true se o texto estiver vazio.
   */
  private static boolean isEmpty(String text) {
    return ((text == null) || text.equals(""));
  }

  /**
   * Construtor inicializando o rtulo, o campo texto e definindo o modo de
   * operao do objeto.
   * 
   * @param window .
   * @param labelText texto a ser exibido no rtulo.
   * @param fieldLength tamanho do campo texto.
   * @param project .
   * @param mode modo de operao (ver constantes em ProjectFileChooser).
   * @param isSaveMode .
   */
  public FileTextField(Window window, String labelText, int fieldLength,
    CommonClientProject project, int mode, boolean isSaveMode) {
    this.project = project;
    this.window = window;
    this.mode = mode;
    this.isSaveMode = isSaveMode;
    this.label = new JLabel();
    if (labelText != null) {
      this.label.setText(labelText);
    }
    this.field = new JTextField(fieldLength);
    this.button = ClientUtilities.createImageButton(
      ApplicationImages.ICON_BROWSEFILE_24);
    this.field = new JTextField(fieldLength);
  }

  public void setSaveMode(boolean isSaveMode) {
    this.isSaveMode = isSaveMode;
  }

  /**
   * Construtor inicializando o rtulo, o campo texto, definindo o modo de
   * operao do objeto e a extenso preferencial do arquivo.
   * 
   * @param window .
   * @param labelText texto a ser exibido no rtulo.
   * @param fieldLength tamanho do campo texto.
   * @param project .
   * @param mode modo de operao (ver constantes em ProjectFileChooser).
   * @param extension extenso preferencial para o arquivo.
   * @param isSaveMode .
   */
  public FileTextField(Window window, String labelText, int fieldLength,
    CommonClientProject project, int mode, String extension,
    boolean isSaveMode) {
    this(window, labelText, fieldLength, project, mode, isSaveMode);
    if (extension != null) {
      this.extension = extension.toUpperCase();
    }
  }
}
