package csbase.client.algorithms.commands.newview;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.WindowConstants;

import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StatusBar;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.cache.events.AbstractCommandUpdatedEventListener;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.kernel.ClientException;
import csbase.client.util.gui.log.tab.AbstractTab.TabType;
import csbase.logic.CommandInfo;

/**
 * Janela principal do visualizador de comandos.
 */
class CommandViewFrame extends DesktopComponentFrame {

  /**
   * Altura mnima da janela.
   */
  private static final long MINIMUN_HEIGHT = 600;

  /**
   * Largura mnina da janela
   */
  private static final long MINIMUN_WIDTH = 800;

  /**
   * Viso do comando.
   */
  private CommandView commandView;

  /**
   * Observador do comando.
   */
  private AbstractCommandUpdatedEventListener commandListener;

  /**
   * Construtor.
   * 
   * @param commandView viso do comando (No aceita {@code null}).
   * @param index identificador da view. Usado para garantir a unicidade.
   * @param owner Componente dono da viso (No aceita {@code null}).
   * @param command O comando (No aceita {@code null}).
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  public CommandViewFrame(CommandView commandView, final Object index,
    final DesktopComponentFrame owner, CommandInfo command)
    throws ClientException {
    super(index, owner, "");
    if (command == null) {
      throw new IllegalArgumentException("O parmetro command est nulo.");
    }
    if (commandView == null) {
      throw new IllegalArgumentException("O parmetro commandView est nulo.");
    }

    this.commandView = commandView;
    initializeGui();
    createListeners(command);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void pack() {
    super.pack();
    updateSize();
    if (owner != null) {
      setLocationRelativeTo(owner);
    }
  }

  /**
   * Cria os componentes da janela.
   * 
   * @throws ClientException Erro ao criar visualizao do comando.
   */
  private void initializeGui() throws ClientException {
    this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

    this.setLayout(new BorderLayout());
    add(this.commandView.getMainComponent(this), BorderLayout.CENTER);

    this.setTitle(this.commandView.getTitle());

    pack();
  }

  /**
   * Ajusta o tamanho do dilogo para que ele no passe dos limites do monitor.
   */
  private void updateSize() {
    /*
     * Existe o percentual mximo do tamanho de uma screen utilizvel pelo
     * dilogo de configurao. Isto para no permitir que toda a tela seja
     * usada. Tambm deixamos um offset para o tamanho final para alguma folga e
     * que evita o aparecimento da scrollbar horizontal.
     */
    final double factor = 0.85;
    final int offset = 35;

    Dimension screenSize = GUIUtils.getScreenDimension();
    Dimension currentSize = getSize();

    double currentWidth = currentSize.getWidth() * factor;
    double newWidth = Math.max(currentWidth, MINIMUN_WIDTH);
    double usuableWidth = screenSize.getWidth();
    newWidth = Math.min(newWidth, usuableWidth);

    double currentHeight = currentSize.getHeight() * factor;
    double newHeight = Math.max(currentHeight, MINIMUN_HEIGHT);
    double usuableHeight = screenSize.getHeight();
    newHeight = Math.min(newHeight, usuableHeight);

    Dimension newSize = new Dimension();
    newSize.setSize(newWidth + offset, newHeight + offset);
    setSize(newSize);
  }

  /**
   * Cria os listeneres para o comando a ser visualizado, para que a viso possa
   * ser atualizada.
   * 
   * @param command o comando sendo visualizado
   */
  private void createListeners(CommandInfo command) {
    commandListener =
      new AbstractCommandUpdatedEventListener(command.getProjectId(), command
        .getId()) {
        @Override
        public void eventFired(CommandUpdatedEvent.Type type,
          CommandInfo newCmdInfo) {
          // Atualiza a instncia com o valor recebido pelo evento.
          commandView.commandUpdated(type, newCmdInfo);
        }

        @Override
        protected void eventInterrupted(Exception exception, String description) {
          StatusBar statusBar = commandView.getStatusBar();
          statusBar.setError(description);
        }
      };
    CommandsCache.getInstance().addEventListener(commandListener);

    // Remove o listener do comando se a janela for fechada.
    this.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosed(WindowEvent e) {
        CommandsCache.getInstance().removeEventListener(commandListener);
        commandView.close();
        removeDesktopComponentFrame(getIndex());
      }
    });
  }

  /**
   * Determina qual aba da viso deve ser mostrada, baseado no tipo da aba. Mais
   * de uma aba pode ter o mesmo tipo; nesse caso, a primeira aba do tipo 
   * mostrada.
   * 
   * @param tabType o tipo ({@link TabType}) de aba a ser selecionada.
   * 
   * @throws ClientException em caso de erro na criao da viso da aba.
   */
  public void showTab(TabType tabType) throws ClientException {
    this.commandView.selectTab(tabType);
    setVisible(true);
  }

  /**
   * Determina qual aba da viso deve ser mostrada, baseado no ttulo da aba.
   * No pode haver mais de uma aba com o mesmo ttulo.
   * 
   * @param title o ttulo da aba a ser selecionada.
   * @throws ClientException em caso de erro na criao da viso da aba.
   */
  public void showTab(String title) throws ClientException {
    this.commandView.selectTab(title);
    setVisible(true);
  }
}
