package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import csbase.logic.algorithms.Category;

/**
 * Essa classe representa um painel para criao de uma nova categoria.
 * 
 * A criao  enviada para o servidor.
 * 
 */
public class CategoryCreateInfoPanel extends CommonInfoEditPanel {

  /** Campo texto (read-only) com o nome da categoria pai */
  private JTextField tf_categoryParentName;

  /** Campo texto com o nome da categoria */
  private JTextField tf_categoryName;

  /** Indica que est sendo feita uma edio */
  private boolean isEditing;

  /**
   * Constri o painel com as informaes bsicas de criao da categoria.
   * 
   * @param categoryPane painel com os dados de criao de uma categoria
   */

  public CategoryCreateInfoPanel(CommonEditTabbedPanel categoryPane) {
    super(categoryPane);
    setCancelActionState(true);
  }

  /**
   * Obtm o painel com o nome da categoria.
   * 
   * @return o painel criado
   */
  private JPanel getCategoryNamePanel() {
    JPanel namePanel = new JPanel(new GridBagLayout());

    JLabel categoryNameLabel =
      new JLabel(getString("CategoryCreateInfoPanel.label.name"));
    JLabel parentCategoryNameLabel =
      new JLabel(getString("CategoryCreateInfoPanel.label.parent.name"));
    tf_categoryParentName = new JTextField();
    tf_categoryParentName.setEditable(false);
    tf_categoryName = buildNameCategoryField();
    namePanel.add(parentCategoryNameLabel, new GBC(0, 0).none().west().insets(
      5, 5, 5, 5));
    namePanel.add(tf_categoryParentName, new GBC(1, 0).horizontal().west()
      .insets(5, 5, 5, 5));
    namePanel.add(categoryNameLabel, new GBC(0, 1).none().west().insets(5, 5,
      5, 5));
    namePanel.add(tf_categoryName, new GBC(1, 1).horizontal().west().insets(5,
      5, 5, 5));
    return namePanel;
  }

  /**
   * Constri o campo texto para entrar com o nome da categoria.
   * 
   * @return o campo texto criado
   */
  private JTextField buildNameCategoryField() {
    tf_categoryName = new JTextField(20);
    tf_categoryName.addKeyListener(new KeyAdapter() {
      @Override
      public void keyReleased(KeyEvent e) {
        if (e.getKeyCode() == KeyEvent.VK_ENTER) {
          apply();
        }
      }
    });
    tf_categoryName.addCaretListener(e -> {
      JTextField tf = (JTextField) e.getSource();
      boolean hasName = !(tf.getText().trim().isEmpty());
      isEditing = hasName;
      setApplyActionState(hasName);
    });
    return tf_categoryName;
  }

  /**
   * A partir dos nomes das categorias adicionadas pelo usurio, as categorias
   * sero efetivamente criadas no servidor.
   * 
   * @return retorna true se a categoria foi criada com sucesso, caso contrrio,
   *         retorna false
   */
  protected boolean createCategory() {
    String categoryName = getCategoryName();
    Category parentCategory = getParentCategory();
    boolean categoryExists = false;
    if (parentCategory != null) {
      categoryExists = parentCategory.containsCategory(categoryName);
    }
    else {
      categoryExists = getApplication().containsCategory(categoryName);
    }
    if (categoryExists) {
      getApplication().showError(
        getString("CategoryCreateInfoPanel.msg.error.create.category_exists"));
      return false;
    }

    Category categoryCreated =
      getCategoryCreateDataPanel().createCategory(categoryName);
    if (categoryCreated == null) {
      getApplication().showError(
        getString("CategoryCreateInfoPanel.msg.error.create.no_sucess"));
      return false;
    }
    return true;
  }

  /**
   * Obtm a categoria pai onde vai ser criada a nova categoria.
   * 
   * @return a categoria pai da nova categoria
   */
  private Category getParentCategory() {
    return getCategoryCreateDataPanel().getParentCategory();
  }

  /**
   * Obtm o nome da categoria especificado pelo usurio.
   * 
   * @return o nome da categoria
   */
  private String getCategoryName() {
    return tf_categoryName.getText().trim();
  }

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.applications.algorithmsmanager.dialogs.CommonInfoEditPanel#initializeData()
   */
  @Override
  public void initializeData() {
    tf_categoryParentName.setText(getCategoryCreateDataPanel()
      .getParentCategoryName());
    tf_categoryName.requestFocus();
  }

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.applications.algorithmsmanager.dialogs.CommonInfoEditPanel#apply()
   */
  @Override
  protected void apply() {
    isEditing = false;
    if (!createCategory()) {
      tf_categoryName.requestFocus();
    }
  }

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.applications.algorithmsmanager.dialogs.CommonInfoEditPanel#cancel()
   */
  @Override
  protected void cancel() {
    boolean confirmCancelling =
      !wasModified() || (wasModified() && confirmCancelling());
    if (confirmCancelling) {
      getCategoryCreateDataPanel().cancel();
      getSelectionPanel().selectCurrentCategory();
    }
  }

  /**
   * Obtm o painel de seleo de algoritmos.
   * 
   * @return o painel de seleo de algoritmos
   */
  private CategorySelectionPanel getSelectionPanel() {
    return getCategoryCreateDataPanel().getSelectionPanel();
  }

  /**
   * Confirma com o usurio se a operao de cancelamento deve ser efetivada, j
   * que perder todas as alteraes feitas sobre o dado.
   * 
   * @return retorna true, se a alterao do dado deve ser cancelada, caso
   *         contrrio, retorna false
   */
  private boolean confirmCancelling() {
    int confirm =
      getApplication().showOptionDialog(
        getString("CategoryCreateInfoPanel.msg.cancel.confirm"),
        new String[] {
            getString("CategoryCreateInfoPanel.msg.cancel.confirm.yes"),
            getString("CategoryCreateInfoPanel.msg.cancel.confirm.no") });
    if (confirm == 0) {
      isEditing = false;
      return true;
    }
    return false;
  }

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.applications.algorithmsmanager.dialogs.CommonInfoEditPanel#buildMainInfoPanel()
   */
  @Override
  protected JPanel buildMainInfoPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    JPanel namePanel = getCategoryNamePanel();

    panel.add(namePanel, new GBC(0, 0).horizontal().west().insets(0, 0, 0, 0));
    panel
      .add(buildEmptyPanel(), new GBC(0, 1).both().west().insets(0, 0, 0, 0));
    return panel;
  }

  @Override
  public void setDataChanged() {
    isEditing = false;
    changeOperationsState(isEditing);
  }

  /**
   * Obtm o painel global de criao de dados de categoria.
   * 
   * @return o painel global de criao de dados de categoria
   */
  private CategoryCreateDataPanel getCategoryCreateDataPanel() {
    return (CategoryCreateDataPanel) getEditPanel();
  }

  @Override
  protected String getTitle() {
    return getApplication().getString("CommonEditTabbedPanel.tab.info.basic");
  }

  @Override
  public boolean wasModified() {
    return isEditing;
  }
}
