/*
 * $Id: FileExchanger.java 175310 2016-08-03 14:38:03Z fpina $
 */
package csbase.client.applications.fileexchanger;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JSplitPane;
import javax.swing.JToolBar;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.fileexchanger.actions.BlockSizeAction;
import csbase.client.applications.fileexchanger.actions.ClearEndedAction;
import csbase.client.applications.fileexchanger.actions.DeleteAction;
import csbase.client.applications.fileexchanger.actions.ExportAction;
import csbase.client.applications.fileexchanger.actions.ImportAction;
import csbase.client.applications.fileexchanger.actions.InterruptAction;
import csbase.client.applications.fileexchanger.actions.ShowErrorAction;
import csbase.client.applications.fileexchanger.actions.StartAction;
import csbase.client.applications.fileexchanger.actions.ToggleAdvancedColumnsVisibilityAction;
import csbase.client.applications.fileexchanger.actions.ToggleDetailVisibilityAction;
import csbase.client.applications.fileexchanger.actions.TransferModeAction;
import csbase.client.applications.fileexchanger.exceptions.InvalidFilesToExportException;
import csbase.client.applications.fileexchanger.exceptions.InvalidNameException;
import csbase.client.applications.fileexchanger.exceptions.InvalidReservedNameException;
import csbase.client.applications.fileexchanger.logic.BlockSize;
import csbase.client.applications.fileexchanger.logic.Exchange;
import csbase.client.applications.fileexchanger.logic.ExchangeExport;
import csbase.client.applications.fileexchanger.logic.ExchangeImport;
import csbase.client.applications.fileexchanger.logic.ExchangeMode;
import csbase.client.applications.fileexchanger.logic.ExchangeState;
import csbase.client.applications.fileexchanger.logic.FileExchangerConfiguration;
import csbase.client.applications.fileexchanger.panels.detailpanel.DetailPanel;
import csbase.client.applications.fileexchanger.panels.tablepanel.TableListPanel;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectDirectoryChooser;
import csbase.client.project.ProjectFileChooser;
import csbase.client.project.ProjectFileChooserOpen;
import csbase.client.project.ProjectTreePath;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.client.project.tasks.GetChildrenTask;
import csbase.client.util.ClientUtilities;
import csbase.logic.ClientProjectFile;
import csbase.logic.NoHiddenFileFilter;
import csbase.logic.ProjectFileFilter;
import csbase.logic.ProjectFileType;

/**
 * Classe que representa a aplicao transferidor de arquivos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class FileExchanger extends ApplicationProject {

  /** Opo Sobrescrever arquivo */
  final private static int OPTION_YES = 0;
  /** Opo Sobrescrever todos os arquivos */
  final private static int OPTION_YES_ALL = 1;
  /** Opo No sobrescrever arquivo */
  final private static int OPTION_NO = 2;
  /** Opo No sobrescrever nenhum arquivo */
  final private static int OPTION_NO_ALL = 3;

  /**
   * Thread de atualizao da interface grfica.
   */
  final private Thread interfaceUpdateThread;

  /**
   * Thread que busca operaes na fila e as inicia.
   */
  final private Thread checkPendingExchangesThread;

  /**
   * Tempo de sleep atual da thread checkPendingExchangesThread.
   */
  private long currenIntervalToCheckPendingTransfers =
    FileExchangerConfiguration.DEFAULT_MAX_INTERVAL_TO_CHECK_PENDING_TRANSFERS;

  /**
   * Lista de operaes.
   */
  final private List<Exchange> exchangeList = Collections
    .synchronizedList(new ArrayList<Exchange>());

  /**
   * Painel de operaes.
   */
  final private TableListPanel tablePanel;

  /**
   * Painel de detalhes.
   */
  final private DetailPanel detailPanel;

  /**
   * SplitPane do dilogo principal.
   */
  final private JSplitPane splitPane;

  /**
   * Indicativo de colunas avanadas em exibio.
   */
  private boolean advancedColumnsVisible = false;

  /**
   * Configurao avanada do programa.
   */
  final private FileExchangerConfiguration configuration =
    new FileExchangerConfiguration(this);

  /**
   * File chooser nico para a importao para manter o path do ltimo diretrio
   * escolhido.
   */
  final private JFileChooser importationFileChooser = new JFileChooser();

  /**
   * Mantm sempre o path do ltimo diretrio escolhido em uma exportao bem
   * sucedida.
   */
  static private String exportationCurrentDirectory = new String();

  /**
   * Adiciona uma operao.
   * 
   * @param exchange a operao.
   */
  final synchronized public void addToExchangeList(final Exchange exchange) {
    exchangeList.add(exchange);
    exchange.signalAdded();
  }

  /**
   * Adiciona uma operao com erro.
   * 
   * @param exchange a operao.
   */
  final synchronized public void addExchangeErrorToExchangeList(
    final Exchange exchange) {
    exchangeList.add(exchange);
    exchange.signalError();
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu de arquivos.
   */
  private JMenu buildActionsMenu() {
    final JMenu menu = new JMenu();
    menu.setText(getMenuString("actions"));
    menu.add(new ClearEndedAction(this));
    menu.addSeparator();
    menu.add(new ShowErrorAction(this));
    menu.add(new DeleteAction(this));
    menu.add(new StartAction(this));
    menu.add(new InterruptAction(this));
    return menu;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu de arquivos.
   */
  private JMenu buildAdvancedMenu() {
    final JMenu menu = new JMenu();
    final String menuText = getMenuString("advanced");
    menu.setText(menuText);

    final ToggleDetailVisibilityAction detAction =
      new ToggleDetailVisibilityAction(this);
    final JCheckBoxMenuItem detItem = new JCheckBoxMenuItem(detAction);
    menu.add(detItem);

    final ToggleAdvancedColumnsVisibilityAction advAction =
      new ToggleAdvancedColumnsVisibilityAction(this);
    final JCheckBoxMenuItem advItem = new JCheckBoxMenuItem(advAction);
    menu.add(advItem);

    menu.addSeparator();
    menu.add(buildTransferModeMenu());
    menu.add(buildBlockSizeMenu());
    return menu;
  }

  /**
   * @return o menu
   */
  private JMenu buildBlockSizeMenu() {
    final JMenu menu = new JMenu();
    menu.setText(getMenuString("advanced.block.size"));
    final ButtonGroup grp = new ButtonGroup();
    for (final BlockSize value : BlockSize.values()) {
      final BlockSizeAction action = new BlockSizeAction(this, value);
      final JRadioButtonMenuItem it = new JRadioButtonMenuItem(action);
      menu.add(it);
      grp.add(it);
      if (value == getConfiguration().getBlockSize()) {
        it.setSelected(true);
      }
    }
    return menu;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu de arquivos.
   */
  private JMenu buildFileMenu() {
    final JMenu menu = new JMenu();
    menu.setText(getMenuString("file"));
    menu.add(new ImportAction(this));
    menu.add(new ExportAction(this));
    menu.addSeparator();
    menu.add(new ApplicationExitAction(this));
    return menu;
  }

  /**
   * Montagem do menu.
   * 
   * @return o menu do programa.
   */
  private JMenuBar buildMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(buildFileMenu());
    menuBar.add(buildActionsMenu());
    menuBar.add(buildAdvancedMenu());
    return menuBar;
  }

  /**
   * Montagem da toolbar.
   * 
   * @return o painel.
   */
  private JToolBar buildToolBar() {
    final String appName = getName();
    final JToolBar toolBar = new JToolBar(appName);
    toolBar.setFloatable(false);

    toolBar.add(new ImportAction(this));
    toolBar.add(new ExportAction(this));
    toolBar.addSeparator();
    toolBar.add(new ClearEndedAction(this));
    toolBar.addSeparator();
    toolBar.add(new DeleteAction(this));
    toolBar.add(new StartAction(this));
    toolBar.add(new InterruptAction(this));
    toolBar.addSeparator();
    return toolBar;
  }

  /**
   * @return menu
   */
  private JMenu buildTransferModeMenu() {
    final JMenu menu = new JMenu();
    menu.setText(getMenuString("advanced.transfer.mode"));
    final ButtonGroup grp = new ButtonGroup();
    for (final ExchangeMode tp : ExchangeMode.values()) {
      final TransferModeAction action = new TransferModeAction(this, tp);
      final JRadioButtonMenuItem it = new JRadioButtonMenuItem(action);
      menu.add(it);
      grp.add(it);
      if (tp == configuration.getTransferMode()) {
        it.setSelected(true);
      }
    }
    return menu;
  }

  /**
   * Cria a thread de atualizao da interface grfica.
   * 
   * @return a thread
   */
  private Thread buildInterfaceUpdateThread() {
    final Thread thread = new Thread() {
      @Override
      final public void run() {
        try {
          while (true) {
            final long ms = configuration.getGuiUpdateInterval();
            Thread.sleep(ms);
            // Segundo parmetro indica que a estrutura da tabela no deve ser
            // mudada.
            FileExchanger.this.updateExchangePanel(false, false);
            markPointsOfRunningExchanges();
          }
        }
        catch (final InterruptedException e) {
        }
      }
    };
    return thread;
  }

  /**
   * Cria a thread de atualizao da interface grfica.
   * 
   * @return a thread
   */
  private Thread buildCheckPendingExchangesThread() {
    final Thread thread = new Thread() {
      @Override
      final public void run() {
        try {
          while (true) {
            checkPending();
            Thread.sleep(currenIntervalToCheckPendingTransfers);
          }
        }
        catch (InterruptedException e) {
        }
      }
    };
    return thread;
  }

  /**
   * Marca os pontos das operaes em andamento.
   */
  protected void markPointsOfRunningExchanges() {
    // Transforma em array para poder iterar sem problemas.
    Exchange[] exchangeArray =
      exchangeList.toArray(new Exchange[exchangeList.size()]);
    for (final Exchange e : exchangeArray) {
      if (e.getState() == ExchangeState.RUNNING) {
        // Aproveita e marca pontos de operaes em andamento.
        e.ping();
      }
    }
  }

  /**
   * Coloca para rodar operaes pendentes.
   */
  protected void checkPending() {
    boolean thereIsPendingExchange = false;
    int running_threads = 0;
    // Transforma em array para poder iterar sem problemas.
    Exchange[] exchangeArray =
      exchangeList.toArray(new Exchange[exchangeList.size()]);
    for (final Exchange e : exchangeArray) {
      if (running_threads >= configuration.getMaxSimultaneousTransfers()) {
        thereIsPendingExchange = true;
        break;
      }
      if (e.getState() == ExchangeState.RUNNING) {
        running_threads++;
      }
      else if (e.getState() == ExchangeState.QUEUED) {
        running_threads++;
        e.start();
      }
    }
    // No h mais transferncias na fila, considerando que o ltimo exchange da
    // lista ou est RUNNING ou est QUEUED. Neste ltimo caso, a transferncia
    // foi iniciada e no h mais exchanges na fila de espera.
    if (thereIsPendingExchange) {
      currenIntervalToCheckPendingTransfers =
        configuration.getMinIntervalToCheckPendingTransfers();
    }
    else {
      currenIntervalToCheckPendingTransfers =
        configuration.getMaxIntervalToCheckPendingTransfers();
    }
  }

  /**
   * Apaga operaes terminadas.
   */
  final synchronized public void clearEndedExchanges() {
    final ArrayList<Exchange> list = new ArrayList<Exchange>();
    for (final Exchange e : exchangeList) {
      final ExchangeState state = e.getState();
      if (state != ExchangeState.RUNNING && state != ExchangeState.QUEUED) {
        list.add(e);
      }
    }

    for (final Exchange e : list) {
      delFromExchangeList(e);
    }
    // Segundo parmetro indica que a estrutura da tabela no deve ser
    // mudada.
    updateExchangePanel(true, false);
  }

  /**
   * Remove uma operao.
   * 
   * @param exchange a operao.
   */
  final synchronized public void delFromExchangeList(final Exchange exchange) {
    exchangeList.remove(exchange);
    exchange.signalRemoved();
    // Segundo parmetro indica que a estrutura da tabela no deve ser
    // mudada.
    updateExchangePanel(true, false);
  }

  /**
   * Retorna o nmero de colunas exibidas.
   * 
   * @return o nmero
   */
  final public int getNumShownColumns() {
    final int n = 13;
    if (!isAdvancedColumnsVisible()) {
      return n - 2;
    }
    return n;
  }

  /**
   * Faz atualizao.
   * 
   * @param structureChanged indicativo de mudana da estrutura da tabela.
   */
  private void doUpdate(boolean structureChanged) {
    tablePanel.updateTable(structureChanged);
    detailPanel.updateData(getSelectedExchangeList());
  }

  /**
   * Ativa a exportao de um arquivo/diretrio para um diretrio do disco
   * local. O arquivo a ser exportado pode ter um novo nome (save as).
   * 
   * @param remoteFile arquivo remoto.
   * @param localDir diretrio local.
   * @param localName nome local.
   */
  final private void exportFileToLocalDirectory(
    final ClientProjectFile remoteFile, final File localDir, String localName) {
    if (localName != null && !localName.isEmpty()) {
      localName = localDir.getAbsolutePath() + File.separator + localName;
    }
    else {
      localName =
        localDir.getAbsolutePath() + File.separator + remoteFile.getName();
    }
    final File localFile = new File(localName);
    if (localFile.exists()) {
      final ApplicationFrame frame = getApplicationFrame();
      final String title =
        FileExchangerUI.getString("FileExchanger.export.title");
      final String[] args = new String[] { localName, localDir.getName() };
      final String tag = "export.overwrite.msg";
      final String msg = FileExchangerUI.getClassString(getClass(), tag, args);
      final int option = StandardDialogs.showYesNoDialog(frame, title, msg);
      if (option == 1) {
        return;
      }
    }
    transferFile(remoteFile, localFile);
  }

  /**
   * Ativa a exportao de um ou mais arquivos/diretrios para um diretrio do
   * disco local.
   * 
   * @param remoteFiles a lista de arquivos remotos.
   * @param localDir a diretrio local.
   */
  final public void exportFilesToLocalDirectory(
    final List<ClientProjectFile> remoteFiles, final File localDir) {
    final ApplicationFrame frame = getApplicationFrame();
    int option = OPTION_YES; /* Opo inicial de sobrescrever */
    for (ClientProjectFile remoteFile : remoteFiles) {
      final String localName =
        localDir.getAbsolutePath() + File.separator + remoteFile.getName();
      final File localFile = new File(localName);
      boolean shouldTranferFile = true;
      if (localFile.exists()) {
        /* Arquivo j existe */
        final String[] args = new String[] { localName, localDir.getName() };
        String tag = "FileExchanger.export.overwrite.msg";
        if (remoteFile.isDirectory()) {
          tag = "FileExchanger.export.overwrite_directory.msg";
        }
        final String msg = FileExchangerUI.getString(tag, args);
        final String title =
          FileExchangerUI.getString("FileExchanger.export.title");

        /* Um nico arquivo */
        if (remoteFiles.size() == 1) {
          int singleFileOption =
            StandardDialogs.showYesNoDialog(frame, title, msg);
          if (singleFileOption == 1) {
            shouldTranferFile = false;
          }
        }
        else { /* Vrios arquivos */
          if (option != OPTION_YES_ALL && option != OPTION_NO_ALL) {
            /* Exibe janela 'Sim', 'Sim para todos', 'No', 'No para todos' */
            final Object[] options =
              { LNG.get("UTIL_YES"), LNG.get("UTIL_YES_TO_ALL"),
                  LNG.get("UTIL_NO"), LNG.get("UTIL_NO_TO_ALL") };
            option =
              StandardDialogs.showOptionDialog(frame, title, msg, options);
          }
          if (option == OPTION_NO || option == OPTION_NO_ALL) {
            shouldTranferFile = false;
          }
        }
      } /* Fim do IF da sobrescrita de arquivo local */
      if (shouldTranferFile) {
        transferFile(remoteFile, localFile);
      }
    } /* Fim do FOR de arquivos remotos. */
  }

  /**
   * Ativa a exportao de um ou mais arquivos da rea de projetos para um
   * diretrio local. Os arquivos/diretrios do projeto podem ser passados se
   * eles j for conhecidos (quando a exportao  feita pela rvore de
   * projetos) ou desconhecido (quando a exportao  feita pelo transferidor de
   * arquivos).
   * 
   * @param remoteFiles os arquivos/diretrios remoto
   * @param wasRunning indica se a aplicao j estava executando.
   * @throws Exception se houver falha
   */
  final public void exportToDirectory(List<ClientProjectFile> remoteFiles,
    final Boolean wasRunning) throws Exception {
    final ApplicationFrame frame = this.getApplicationFrame();

    // Verifica se os arquivos remotos podem ser exportados de acordo com os
    // valores configurados para a aplicao.
    if (remoteFiles != null) {
      try {
        isValidFilesToExport(remoteFiles);
      }
      catch (InvalidFilesToExportException e) {
        final String title =
          FileExchangerUI.getString("FileExchanger.export.title");
        StandardDialogs.showErrorDialog(frame, title, e.getMessage());
        // Se a aplicao foi aberta para exportao, ela deve ser fechada no 
        // cancelamento.
        if (wasRunning != null && !wasRunning) {
          this.closeApplication();
        }
        // Se a aplicao j estava executando antes, no  necessrio fech-la.
        return;
      }
    }

    boolean exportMultipleFiles = configuration.isExportMultipleFiles();
    boolean exportDirectories = configuration.isExportDirectories();
    int mode = ProjectFileChooser.FILE_ONLY;
    if (exportDirectories == true) {
      mode = ProjectFileChooser.FILE_AND_DIRECTORY;
    }
    else {
      mode = ProjectFileChooser.FILE_ONLY;
    }

    // Verifica se a lista de arquivos/diretrios remotos a serem exportados
    // j foi fornecida. Caso contrrio, abre-se um dilogo para a escolha dos
    // arquivos/diretrios remotos.
    if (remoteFiles == null) {
      remoteFiles = new Vector<ClientProjectFile>();
      // Exporta mltiplos arquivos e/ou diretrios. Se o mode est setado 
      // para ProjectFileChooser.FILE_AND_DIRECTORY, pode-se exportar tanto
      // arquivos, quanto diretrios. Se estiver setado para 
      // ProjectFileChooser.FILE_ONLY, s  possvel exportar arquivos.
      if (exportMultipleFiles == true) {
        // Exibe um dilogo para a escolha de mltiplos arquivos e/ou 
        // diretrios
        ProjectFileChooserOpen chooser = null;
        chooser =
          new ProjectFileChooserOpen(frame, DesktopFrame.getInstance()
            .getProject(), exportMultipleFiles, mode);
        ProjectTreePath[] projectTreePaths = chooser.getSelectedPaths();
        if (projectTreePaths == null) {
          return;
        }
        for (ProjectTreePath projectTreePath : projectTreePaths) {
          remoteFiles.add(projectTreePath.getFile());
        }
        // Exporta s um nico arquivo ou um nico diretrio. Se o mode est setado 
        // para ProjectFileChooser.FILE_AND_DIRECTORY, pode-se exportar um arquivo 
        // ou um diretrio. Se estiver setado para ProjectFileChooser.FILE_ONLY, 
        //  s  possvel exportar arquivo.
      }
      else {
        // Exibe um dilogo para a escolha do arquivos e/ou do diretrio a
        // ser exportado
        ProjectFileChooserOpen chooser = null;
        chooser =
          new ProjectFileChooserOpen(frame, DesktopFrame.getInstance()
            .getProject(), exportMultipleFiles, mode);
        ProjectTreePath projectTreePath = chooser.getSelectedPath();
        if (projectTreePath == null) {
          return;
        }
        remoteFiles.add(projectTreePath.getFile());

      }
    }

    // Dilogo para a seleo do diretrio de destino a ser exportado os 
    // arquivos/diretrios. Se apenas um nico arquivo foi selecionado para
    // exportao,  possvel dar um novo nome para o arquivo local. 
    JFileChooser exportationFileChooser = new JFileChooser();
    exportationFileChooser.setMultiSelectionEnabled(false);
    exportationFileChooser.setDialogTitle(FileExchangerUI
      .getString("FileExchanger.export.title"));
    exportationFileChooser.setApproveButtonText(FileExchangerUI
      .getString("FileExchanger.export.filechooser.approvebuttontext"));
    exportationFileChooser.setApproveButtonToolTipText(FileExchangerUI
      .getString("FileExchanger.export.filechooser.approvebuttontooltiptext"));
    exportationFileChooser.setCurrentDirectory(new File(
      exportationCurrentDirectory));

    int result = JFileChooser.CANCEL_OPTION;
    if (remoteFiles.size() == 1 && !remoteFiles.get(0).isDirectory()) {
      ClientProjectFile file = remoteFiles.get(0);
      exportationFileChooser.setSelectedFile(new File(file.getName()));
      exportationFileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
      result =
        exportationFileChooser.showDialog(frame, FileExchangerUI
          .getString("FileExchanger.export.filechooser.approvebuttontext"));
    }
    else {
      exportationFileChooser
        .setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
      result = exportationFileChooser.showOpenDialog(frame);
    }

    String localName = null;
    File localDir = null;
    if (result == JFileChooser.CANCEL_OPTION) {
      // Se a aplicao foi aberta para exportao, ela deve ser fechada no 
      // cancelamento.
      if (wasRunning != null && !wasRunning) {
        this.closeApplication();
      }
      // Se a aplicao j estava executando antes, no  necessrio fech-la.
      return;
    }
    else if (result == JFileChooser.APPROVE_OPTION) {
      // Caso de um nico arquivo.
      if (remoteFiles.size() == 1 && !remoteFiles.get(0).isDirectory()) {
        localName = exportationFileChooser.getSelectedFile().getName();
        if (localName != null) {
          localDir = exportationFileChooser.getSelectedFile().getParentFile();
        }
      }
      // Um ou mais arquivos (incluindo diretrios).
      else {
        localDir = exportationFileChooser.getSelectedFile();
      }
    }

    // Definindo o diretrio de destino.
    if (localDir == null || !localDir.exists()) {
      final String title =
        FileExchangerUI.getString("FileExchanger.export.title");
      final String[] args = new String[] { localDir.getName() };
      final String err = "FileExchanger.invalid.directory.error";
      final String msg = FileExchangerUI.getString(err, args);
      final DesktopFrame dsk = DesktopFrame.getInstance();
      StandardDialogs.showErrorDialog(dsk.getDesktopFrame(), title, msg);
    }

    // Somento aps a confirmao de que o diretrio selecionado  vlido, 
    //  que podemos salvar o path do ltimo diretrio selecionado.
    exportationCurrentDirectory = localDir.getAbsolutePath();

    // Exporta os arquivos/diretrios selecionados.
    if (remoteFiles.size() == 1) {
      this.exportFileToLocalDirectory(remoteFiles.get(0), localDir, localName);
    }
    else {
      this.exportFilesToLocalDirectory(remoteFiles, localDir);
    }
  }

  /**
   * Verifica se os arquivos remotos passados podem ser exportados. Isso vai
   * depender dos valores configurados para a aplicao. Por exemplo, se a
   * aplicao s permite exportar um nico arquivo e a lista de arquivos
   * remotos for maior que um, a exportao no  permitida. Se a aplicao s
   * permite exportar arquivos e na lista de arquivos remotos conter diretrios,
   * a exportao tambm no  permitida.
   * 
   * @param remoteFiles arquivos remotos.
   * @throws InvalidFilesToExportException se ocorrer erro
   */
  public void isValidFilesToExport(List<ClientProjectFile> remoteFiles)
    throws InvalidFilesToExportException {

    boolean exportMultipleFiles = configuration.isExportMultipleFiles();
    boolean exportDirectories = configuration.isExportDirectories();
    int mode = ProjectFileChooser.FILE_ONLY;
    if (exportDirectories == true) {
      mode = ProjectFileChooser.FILE_AND_DIRECTORY;
    }
    else {
      mode = ProjectFileChooser.FILE_ONLY;
    }

    // Verifica se a seleo simples foi obedecida.
    if (exportMultipleFiles == false) {
      if (remoteFiles.size() > 1) {
        if (mode == ProjectFileChooser.FILE_ONLY) {
          final String msg =
            FileExchangerUI.getString("FileExchanger.export.file_only");
          throw new InvalidFilesToExportException(msg);
        }
        final String msg =
          FileExchangerUI.getString("FileExchanger.export.file_or_directory");
        throw new InvalidFilesToExportException(msg);
      }
      else if (mode == ProjectFileChooser.FILE_ONLY) {
        for (ClientProjectFile remoteFile : remoteFiles) {
          if (remoteFile.isDirectory()) {
            final String msg =
              FileExchangerUI.getString("FileExchanger.export.file_only");
            throw new InvalidFilesToExportException(msg);
          }
        }
      }
    }
    // Verifica se a seleo de arquivos foi obedecida.
    if (mode == ProjectFileChooser.FILE_ONLY) {
      for (ClientProjectFile remoteFile : remoteFiles) {
        if (remoteFile.isDirectory()) {
          final String msg =
            FileExchangerUI.getString("FileExchanger.export.files_only");
          throw new InvalidFilesToExportException(msg);
        }
      }
    }
  }

  /**
   * Consulta o valor de configuration
   * 
   * @return o valor
   */
  public final FileExchangerConfiguration getConfiguration() {
    return configuration;
  }

  /**
   * Retorna a operao da lista definida pela posio dada.
   * 
   * @param index posio na lista da operao desejada.
   * @return a operao
   */
  final synchronized public Exchange getExchangeItem(final int index) {
    if ((index < 0) || (index >= exchangeList.size())) {
      return null;
    }
    return exchangeList.get(index);
  }

  /**
   * Retorna a lista de operaes em um determinado estado
   * 
   * @param state estado
   * @return a lista
   */
  final synchronized public List<Exchange> getExchangeList(
    final ExchangeState state) {
    if (state == null) {
      return exchangeList;
    }
    final ArrayList<Exchange> list = new ArrayList<Exchange>();
    for (final Exchange e : exchangeList) {
      if (e.getState() == state) {
        list.add(e);
      }
    }
    return list;
  }

  /**
   * Retorna o tamanho da lista de operaes.
   * 
   * @return a lista
   */
  final synchronized public int getExchangeListSize() {
    return exchangeList.size();
  }

  /**
   * Busca texto de menu.
   * 
   * @param tag tag
   * @return o texto
   */
  final private String getMenuString(final String tag) {
    return FileExchangerUI.getString("FileExchanger." + tag + ".menu");
  }

  /**
   * Consulta a lista de operaes selecionadas.
   * 
   * @return a lista
   */
  final synchronized public ArrayList<Exchange> getSelectedExchangeList() {
    final int[] sels = tablePanel.getSelectedIndexes();
    final ArrayList<Exchange> list = new ArrayList<Exchange>();
    for (final int sel : sels) {
      list.add(exchangeList.get(sel));
    }
    return list;
  }

  /**
   * Ativa a importao de um arquivo do disco local para a rea de projetos
   * 
   * @param localFile arquivo local.
   * @param remoteDir o diretrio remoto (arquivo remoto ter o mesmo nome que o
   *        arquivo local).
   * @throws Exception se houver falha
   */
  final private void importFileToDirectory(final File localFile,
    final ClientProjectFile remoteDir) throws Exception {

    final String localFileName = localFile.getName();
    // No importa arquivos de controle do csbase.
    if (localFileName.matches("^\u002E.*csbase$")
      || localFileName.matches("^\u002E.*csbase_description$")) {
      addInvalidFiles(localFile, remoteDir, new InvalidReservedNameException(
        localFileName));
      return;
    }

    // No importa arquivos com nome invlido.
    if (!ClientUtilities.isValidFileName(localFileName)) {
      addInvalidFiles(localFile, remoteDir, new InvalidNameException());
      return;
    }

    // Verifica se um arquivo com o mesmo nome do arquivo a ser importado j
    // existe no servidor. Se sim, uma mensagem  mostrado ao usurio para
    // solicitar confirmao de sobrescrita do arquivo.
    final ClientProjectFile remoteFile =
      GetChildFromNameTask.runTask(remoteDir, localFileName);
    final ApplicationFrame frame = getApplicationFrame();
    if (remoteFile != null) {
      final String[] args = new String[] { localFileName, remoteDir.getName() };
      String tag = "FileExchanger.import.overwrite.msg";
      if (remoteFile.isDirectory()) {
        tag = "FileExchanger.import.overwrite_directory.msg";
      }
      final String msg = FileExchangerUI.getString(tag, args);
      final String title =
        FileExchangerUI.getString("FileExchanger.import.title");
      final int option = StandardDialogs.showYesNoDialog(frame, title, msg);
      if (option == 1) {
        return;
      }
    }
    transferFile(localFile, remoteFile, remoteDir);
  }

  /**
   * Ativa a importao de um ou mais arquivos do disco local para a rea de
   * projetos
   * 
   * @param localFileList Lista de arquivos local.
   * @param remoteDir o diretrio remoto (arquivo remoto ter o mesmo nome que o
   *        arquivo local).
   * @throws Exception se houver falha
   */
  final private void importFileToDirectory(final ArrayList<File> localFileList,
    final ClientProjectFile remoteDir) throws Exception {

    final ClientProjectFile[] children = GetChildrenTask.runTask(remoteDir);
    int option = OPTION_YES; /* Opo inicial de sobrescrever */
    for (File localFile : localFileList) {
      final String fileName = localFile.getName();
      ClientProjectFile remoteFile = null;

      /* Verifica se o nome do arquivo  vlido. */
      if (!ClientUtilities.isValidFileName(fileName)) {
        addInvalidFiles(localFile, remoteDir, new InvalidNameException());
        continue;
      }

      /* No importa arquivos de controle do csbase. */
      if (fileName.matches("^\u002E.*csbase$")
        || fileName.matches("^\u002E.*csbase_description$")) {
        addInvalidFiles(localFile, remoteDir, new InvalidReservedNameException(
          fileName));
        continue;
      }

      /* O arquivo a ser importado  vlido. */
      final ApplicationFrame frame = getApplicationFrame();
      boolean shouldTranferFile = true;

      /* Verificando se h algum arquivo no diretrio destino com o mesmo nome */
      for (int i = 0; i < children.length && remoteFile == null; i++) {
        ClientProjectFile child = children[i];
        if (fileName.equals(child.getName())) {
          remoteFile = child;
          if (option != OPTION_YES_ALL && option != OPTION_NO_ALL) {
            /* Vai questionar se deve sobrescrever */
            final String[] args =
              new String[] { fileName, remoteDir.getName() };
            String tag = "FileExchanger.import.overwrite.msg";
            if (remoteFile.isDirectory()) {
              tag = "FileExchanger.import.overwrite_directory.msg";
            }
            final String msg = FileExchangerUI.getString(tag, args);
            final String title =
              FileExchangerUI.getString("FileExchanger.import.title");
            final Object[] options =
              { LNG.get("UTIL_YES"), LNG.get("UTIL_YES_TO_ALL"),
                  LNG.get("UTIL_NO"), LNG.get("UTIL_NO_TO_ALL") };
            option =
              StandardDialogs.showOptionDialog(frame, title, msg, options);
          }
          if (option == OPTION_NO || option == OPTION_NO_ALL) {
            shouldTranferFile = false;
          }
        }
      } /* Fim da verificao de mesmo nome */
      if (shouldTranferFile) {
        transferFile(localFile, remoteFile, remoteDir);
      }
    } /* Fim do FOR de arquivos vlidos */
  }

  /**
   * Ativa a importao de um ou mais arquivos do disco local para a rea de
   * projetos. O diretrio remoto pode ser passado se ele j for conhecido
   * (quando a importao  feita pela rvore de projetos) ou desconhecido
   * (quando a importao  feita pelo transferidor de arquivos).
   * 
   * @param remoteDir o diretrio remoto
   * @param wasRunning indica se a aplicao j estava executando.
   * @throws Exception se houver falha
   */
  final public void importToDirectory(ClientProjectFile remoteDir,
    Boolean wasRunning) throws Exception {
    final ApplicationFrame frame = this.getApplicationFrame();

    importationFileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
    boolean importDirectories = configuration.isImportDirectories();
    if (importDirectories) {
      importationFileChooser
        .setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
    }
    boolean importMultipleFiles = configuration.isImportMultipleFiles();
    if (importMultipleFiles == true) {
      importationFileChooser.setMultiSelectionEnabled(true);

      // Exibe um dilogo para a escolha dos arquivos a serem importados.
      final int ret = importationFileChooser.showOpenDialog(frame);
      if (ret == JFileChooser.CANCEL_OPTION) {
        // Se a aplicao foi aberta para importao, ela deve ser fechada no 
        // cancelamento.
        if (wasRunning != null && !wasRunning) {
          this.closeApplication();
        }
        // Se a aplicao j estava executando antes, no  necessrio fech-la.
        return;
      }

      // Obtm um ou mais arquivos do disco local selecionados para importao.
      File[] localFiles = importationFileChooser.getSelectedFiles();
      if (localFiles == null || localFiles.length == 0) {
        //TODO Lembrar de capturar esse tipo de erro no transferidor de arquivos.
        final String title =
          FileExchangerUI.getString("FileExchanger.import.title");
        final String err = getString("FileExchanger.invalid.selection.error");
        final DesktopFrame dsk = DesktopFrame.getInstance();
        StandardDialogs.showErrorDialog(dsk.getDesktopFrame(), title, err);
        return;
      }

      if (remoteDir == null) {
        // Abre um dilogo para a escolha de um diretrio para a importao dos
        // arquivos locais selecionados.
        ProjectDirectoryChooser chooser = null;
        chooser =
          new ProjectDirectoryChooser(DesktopFrame.getInstance().getProject(),
            frame);
        ProjectTreePath projectTreePath = chooser.getSelectedDirectory();
        if (projectTreePath == null) {
          return;
        }
        // Obtm o arquivo que representa o diretrio selecionado.
        remoteDir = projectTreePath.getFile();
      }

      ArrayList<File> localFileList = new ArrayList<File>();
      for (File localFile : localFiles) {
        localFileList.add(localFile);
      }

      // Realiza a importao dos arquivos do disco local selecionados  para um
      // diretrio da rea de projetos.
      if (localFileList.size() == 1) {
        this.importFileToDirectory(localFileList.get(0), remoteDir);
      }
      else {
        this.importFileToDirectory(localFileList, remoteDir);
      }
    }
    else {
      importationFileChooser.setMultiSelectionEnabled(false);

      // Exibe um dilogo para a escolha dos arquivos a serem importados.
      final int ret = importationFileChooser.showOpenDialog(frame);
      File localFile = null;
      if (ret == JFileChooser.CANCEL_OPTION) {
        // Se a aplicao foi aberta para importao, ela deve ser fechada no 
        // cancelamento.
        if (wasRunning != null && !wasRunning) {
          this.closeApplication();
        }
        // Se a aplicao j estava executando antes, no  necessrio fech-la.
        return;
      }

      localFile = importationFileChooser.getSelectedFile();
      if (localFile == null || !localFile.exists()) {
        final String title =
          FileExchangerUI.getString("FileExchanger.import.title");
        final String err =
          FileExchangerUI.getString("FileExchanger.import.invalid.file.error");
        StandardDialogs.showErrorDialog(frame, title, err);
        return;
      }

      if (remoteDir == null) {
        if (localFile.isDirectory()) {
          // Abre um dilogo para a escolha de um diretrio para a importao 
          // dos arquivos locais selecionados.
          ProjectDirectoryChooser chooser = null;
          chooser =
            new ProjectDirectoryChooser(
              DesktopFrame.getInstance().getProject(), frame);
          ProjectTreePath projectTreePath = chooser.getSelectedDirectory();
          if (projectTreePath == null) {
            return;
          }
          // Obtm o arquivo que representa o diretrio selecionado.
          remoteDir = projectTreePath.getFile();
        }
        else {
          final String localFileName = localFile.getName();
          final String localFileExtension =
            FileUtils.getFileExtension(localFileName);
          final ProjectFileType pft =
            ProjectFileType.getProjectFileTypeFromExtension(localFileExtension,
            false);
          final String fileType = (pft == null ? "" : pft.getCode());

          ClientProjectFile remoteFile =
            this.browseFileSave(fileType, localFileExtension, localFileName,
              frame);
          if (remoteFile == null) {
            //TODO Lembrar de capturar esse tipo de erro no transferidor de 
            // arquivos.
            return;
          }
          remoteDir = remoteFile.getParent();
          this.transferFile(localFile, remoteFile, remoteDir);
          return;
        }
      }

      this.importFileToDirectory(localFile, remoteDir);
    } // Fim do ELSE de seleo nica

  }

  /**
   * Indica visibilidade do painel de detalhes
   * 
   * @return o indicativo
   */
  final public boolean isDetailVisible() {
    return detailPanel.isVisible();
  }

  /**
   * Indica visibilidade de colunas avanadas
   * 
   * @return o indicativo
   */
  final public boolean isAdvancedColumnsVisible() {
    return advancedColumnsVisible;
  }

  /**
   * Ajusta o valor de advancedColumnsVisible
   * 
   * @param visible o valor a ser ajustado.
   */
  public final void setAdvancedColumnsVisible(final boolean visible) {
    advancedColumnsVisible = visible;
    // Segundo parmetro indica que a estrutura da tabela deve ser mudada, pois
    // novas colunas devem aparecer na tabela.
    updateExchangePanel(true, true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public void killApplication() {
    interfaceUpdateThread.interrupt();
    checkPendingExchangesThread.interrupt();
  }

  /**
   * Callback de seleo na lista
   */
  final public void selectionChanged() {
    detailPanel.updateData(getSelectedExchangeList());
  }

  /**
   * Indica visibilidade do painel de detalhes.
   * 
   * @param visible indicativo
   */
  public void setDetailVisible(final boolean visible) {
    detailPanel.setVisible(visible);
    splitPane.setOneTouchExpandable(visible);
    final double divider = 0.5;
    splitPane.setResizeWeight(divider);
    splitPane.setDividerLocation(divider);
  }

  /**
   * Ativa a exportao de um arquivo (ou diretrio) da rea de projetos para o
   * disco local. Arquivos ocultos filho de um diretrio s sero exportados se
   * a opo 'exibir arquivos ocultos' estiver habilitada no cliente.
   * 
   * @param sourceFile arquivo remoto.
   * @param targetFile arquivo local.
   */
  final public void transferFile(final ClientProjectFile sourceFile,
    final File targetFile) {
    ProjectFileFilter filter = null;
    if (sourceFile.isDirectory()) {
      final boolean showHiddenFiles =
        DesktopFrame.getInstance().shouldShowHiddenFiles();
      if (!showHiddenFiles) {
        filter = NoHiddenFileFilter.getInstance();
      }
    }
    final Exchange exchange =
      new ExchangeExport(configuration.getTransferMode(),
        configuration.getBlockSize(), targetFile, sourceFile,
        getApplicationProject(), filter);
    treatExchange(exchange);
  }

  /**
   * Ativa a importao de um arquivo do disco local para a rea de projetos
   * 
   * @param sourceFile arquivo local.
   * @param targetFile arquivo remoto.
   * @param remoteDir diretrio remoto.
   */
  final public void transferFile(final File sourceFile,
    final ClientProjectFile targetFile, final ClientProjectFile remoteDir) {
    final Exchange exchange =
      new ExchangeImport(configuration.getTransferMode(),
        configuration.getBlockSize(), sourceFile, targetFile, remoteDir,
        getApplicationProject());
    treatExchange(exchange);
  }

  /**
   * Ativa a importao de um arquivo do disco local para a rea de projetos
   * 
   * @param invalidFile Arquivo invlido
   * @param remoteDir Diretrio remoto.
   * @param exception Exceo.
   */
  final private void addInvalidFiles(final File invalidFile,
    ClientProjectFile remoteDir, Exception exception) {
    final Exchange exchange =
      new ExchangeImport(configuration.getTransferMode(),
        configuration.getBlockSize(), invalidFile, null, remoteDir,
        getApplicationProject());
    exchange.signalEnded(exception);
    treatInvalidExchange(exchange);
  }

  /**
   * Adiciona a operao a lista, atualiza o painel e ativa o dilogo principal
   * 
   * @param exchange a operao.
   */
  private void treatExchange(final Exchange exchange) {
    addToExchangeList(exchange);
    // Segundo parmetro indica que a estrutura da tabela no deve ser
    // mudada.
    updateExchangePanel(true, false);
    final ApplicationFrame frame = getApplicationFrame();
    frame.toFront();
    frame.setState(Frame.NORMAL);
  }

  /**
   * Adiciona a operao com erro a lista, atualiza o painel e ativa o dilogo
   * principal.
   * 
   * @param exchange Operao.
   */
  private void treatInvalidExchange(final Exchange exchange) {
    addExchangeErrorToExchangeList(exchange);
    // Segundo parmetro indica que a estrutura da tabela no deve ser
    // mudada.
    updateExchangePanel(true, false);
    final ApplicationFrame frame = getApplicationFrame();
    frame.toFront();
    frame.setState(Frame.NORMAL);
  }

  /**
   * Atualizao da tabela.
   * 
   * @param isOnSwing indicativo de execuo dentro da thread do Swing.
   * @param structureChanged indicativo de mudana da estrutura da tabela.
   */
  private void updateExchangePanel(final boolean isOnSwing,
    final boolean structureChanged) {
    if (isOnSwing) {
      doUpdate(structureChanged);
    }
    else {
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        final public void run() {
          doUpdate(structureChanged);
        }
      });
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    final List<Exchange> running = getExchangeList(ExchangeState.RUNNING);
    final List<Exchange> queued = getExchangeList(ExchangeState.QUEUED);
    final int size = running.size() + queued.size();
    if (size > 0) {
      final String[] args = new String[] { String.valueOf(size) };
      final String msg =
        FileExchangerUI.getString(
          "FileExchanger.close.transmissionsInProgress", args);
      final String title = this.getName();
      final ApplicationFrame frame = getApplicationFrame();
      final int option = StandardDialogs.showYesNoDialog(frame, title, msg);
      if (option == 1) {
        return false;
      }
      // Interrompe as transmisses em andamento e as que esto na fila.
      interruptTransmissions(running);
      interruptTransmissions(queued);
    }
    return true;
  }

  /**
   * Interrompe a transmisso de uma lista de exchanges (em andamento ou em fila
   * de espera).
   * 
   * @param list lista de exchange
   */
  private void interruptTransmissions(List<Exchange> list) {
    for (Exchange exchange : list) {
      exchange.interrupt();
    }
  }

  /**
   * Construtor
   * 
   * @param id id da aplicao.
   */
  public FileExchanger(final String id) {
    super(id);
    tablePanel = new TableListPanel(this);
    detailPanel = new DetailPanel(this);
    detailPanel.setVisible(false);

    final ApplicationFrame frame = getApplicationFrame();
    final JMenuBar menuBar = buildMenuBar();
    frame.setJMenuBar(menuBar);

    splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
    splitPane.add(tablePanel);
    splitPane.add(detailPanel);
    splitPane.setOneTouchExpandable(false);

    final Container panel = frame.getContentPane();
    panel.setLayout(new BorderLayout());
    panel.add(buildToolBar(), BorderLayout.NORTH);
    panel.add(splitPane, BorderLayout.CENTER);

    frame.setSize(new Dimension(700, 400));

    interfaceUpdateThread = buildInterfaceUpdateThread();
    interfaceUpdateThread.start();

    checkPendingExchangesThread = buildCheckPendingExchangesThread();
    checkPendingExchangesThread.start();
  }

}