package csbase.client.applications.flowapplication.graph;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Stroke;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import csbase.logic.algorithms.parameters.AbstractFileParameter;
import csbase.logic.algorithms.parameters.FileParameterListener;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.OutputURLParameter;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.SimpleParameterListener;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.vix.TypeVO;
import tecgraf.vix.VO;

/**
 * <p>
 * Viso para descritores de parmetros do tipo arquivo.
 * </p>
 * <p>
 * Representa os arquivos do configurador de algoritmo no n que representa o
 * configurador.
 * </p>
 *
 * @author lmoreira
 */
public final class GraphFileDescriptor extends VO {

  /**  */
  private static final Color BORDER_COLOR = Color.BLACK;
  /**  */
  private static final Color DISABLED_COLOR = Color.LIGHT_GRAY;
  /**  */
  private static final Color ERROR_COLOR = Color.RED;
  /**  */
  private static final Color EMPTY_BACKGROUND_COLOR = Color.WHITE;
  /**  */
  private static final int HEIGHT = 6;
  /**  */
  private static final Color HIGHLIGHTED_COLOR = GraphElement.HIGHT_LIGHT_COLOR;
  /**  */
  private static final int HORIZONTAL_MARGIN = 2;
  /**  */
  private static final Stroke DEFAULT_STROKE = new BasicStroke(1f);
  /**  */
  private static final Stroke THICKER_STROKE = new BasicStroke(2f);
  /**  */
  private static final int VERTICAL_MARGIN = 2;
  /**  */
  private static final int WIDTH = 12;
  /**  */
  private static final Color BYPASSED_COLOR = new Color(150, 150, 150);
  /**  */
  private Color backgroundColor;
  /**  */
  private Color borderColor;
  /**  */
  private Color defaultColor;
  /**  */
  private Stroke stroke;
  /**  */
  private AbstractFileParameter fileParameter;
  /**  */
  private boolean isInput;
  /**  */
  private boolean isOutput;
  /**  */
  private final Set<GraphLink> linkFromSet;
  /**  */
  private GraphLink linkTo;
  /**  */
  private final String parameterLabel;
  /**  */
  private final String parameterName;
  /**  */
  private GeneralPath path;
  /**  */
  private boolean isVisible;

  /**
   * Construtor
   *
   * @param parameterName nome do parmetro
   * @param parameterLabel rtulo do parmetro
   * @param isOutput indicativo de sada
   * @param node n
   */
  public GraphFileDescriptor(final String parameterName,
    final String parameterLabel, final boolean isOutput, final GraphNode node) {
    if (parameterName == null) {
      throw new IllegalArgumentException(
        "O parmetro parameterName est nulo.");
    }
    if (parameterLabel == null) {
      throw new IllegalArgumentException(
        "O parmetro parameterLabel est nulo.");
    }
    if (node == null) {
      throw new IllegalArgumentException("O parmetro node est nulo.");
    }
    this.isOutput = isOutput;
    this.isInput = !isOutput;
    this.parameterName = parameterName;
    this.parameterLabel = parameterLabel;
    this.linkFromSet = new HashSet<GraphLink>();
    createDefaultColor();
    updateColor();
    changeVS(null, node);
    createShape(new Point());
    node.addListener(new NodeListener());
    updateVisibility();
  }

  /**
   * Construtor (Pode receber qualquer parmetro que represente um arquivo)
   *
   * @param fileParameter parmetro de tipo arquivo.
   * @param node n.
   */
  public GraphFileDescriptor(final AbstractFileParameter fileParameter,
    final GraphNode node) {

    if (fileParameter == null) {
      throw new IllegalArgumentException(
        "O parmetro fileParameter est nulo.");
    }
    if (node == null) {
      throw new IllegalArgumentException("O parmetro node est nulo.");
    }

    this.fileParameter = fileParameter;
    this.isOutput = fileParameter.isOuput();

    this.parameterName = this.fileParameter.getName();
    this.parameterLabel = this.fileParameter.getLabel();
    this.fileParameter.addParameterListener(new FileParameterListener() {
      @Override
      public void hasLinkWasChanged(final AbstractFileParameter parameter) {
      }

      @Override
      public void typeWasChanged(final AbstractFileParameter parameter) {
        createDefaultColor();
        updateColor();
        GraphFileDescriptor.this.vs.repaint();
      }
    });

    this.fileParameter.addSimpleParameterListener(
      new SimpleParameterListener<FileURLValue>() {

        @Override
        public void capabilityWasChanged(
          final SimpleParameter<FileURLValue> parameter) {
        }

        @Override
        public void defaultValueWasChanged(
          final SimpleParameter<FileURLValue> parameter) {
        }

        @Override
        public void labelWasChanged(
          final SimpleParameter<FileURLValue> parameter) {
        }

        @Override
        public void valueWasChanged(
          final SimpleParameter<FileURLValue> parameter) {
          updateVisibility();
        }

        @Override
        public void visibilityWasChanged(
          final SimpleParameter<FileURLValue> parameter) {
        }
      });

    this.linkFromSet = new HashSet<GraphLink>();
    createDefaultColor();
    updateColor();
    changeVS(null, node);
    createShape(new Point());
    node.addListener(new NodeListener());
    updateVisibility();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackRepaint(final Graphics2D g) {
    if (isVisible()) {
      g.setPaint(this.backgroundColor);
      g.fill(this.path);
      g.setPaint(this.borderColor);
      g.setStroke(stroke);
      g.draw(this.path);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object obj) {
    if (obj == null) {
      return false;
    }
    if (!obj.getClass().equals(getClass())) {
      return false;
    }
    final GraphFileDescriptor graphFileType = (GraphFileDescriptor) obj;
    if (!getParameterName().equals(graphFileType.getParameterName())) {
      return false;
    }
    if (!this.vs.equals(graphFileType.vs)) {
      return false;
    }
    return super.equals(obj);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Rectangle2D getBounds2D() {
    final Rectangle2D bounds = this.path.getBounds2D();
    bounds.setFrameFromDiagonal(bounds.getMinX() - HORIZONTAL_MARGIN, bounds
      .getMinY() - VERTICAL_MARGIN, bounds.getMaxX() + HORIZONTAL_MARGIN, bounds
        .getMaxY() + VERTICAL_MARGIN);
    return bounds;
  }

  /**
   * Busca parmetro do tipo file.
   *
   * @return parmetro
   */
  public AbstractFileParameter getFileParameter() {
    return this.fileParameter;
  }

  /**
   * Busca lnik com base em lista
   *
   * @return lista
   */
  public Collection<GraphLink> getLinkFromCollection() {
    return Collections.unmodifiableCollection(this.linkFromSet);
  }

  /**
   * Consulta link "para"
   *
   * @return link
   */
  public GraphLink getLinkTo() {
    return this.linkTo;
  }

  /**
   * Verifica se um link pode ser criado.
   * 
   * @return Indicativo.
   */
  public boolean canAddLink() {
    if (!isOutput()) {
      return false;
    }
    boolean allowMultipleOutput = true;
    if (fileParameter instanceof OutputURLParameter) {
      allowMultipleOutput = ((OutputURLParameter) fileParameter)
        .allowMultipleOutput();
    }
    if (fileParameter instanceof OutputFileParameter) {
      allowMultipleOutput = ((OutputFileParameter) fileParameter)
        .allowMultipleOutput();
    }
    if (allowMultipleOutput) {
      return true;
    }
    if (linkFromSet.isEmpty()) {
      return true;
    }
    return false;
  }

  /**
   * Consulta n
   *
   * @return n
   */
  public GraphNode getNode() {
    return (GraphNode) this.vs;
  }

  /**
   * Consulta rtulo do parmetro.
   *
   * @return rtulo
   */
  public String getParameterLabel() {
    return this.parameterLabel;
  }

  /**
   * Consulta nome do parmetro.
   *
   * @return nome
   */
  public String getParameterName() {
    return this.parameterName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.vs.hashCode();
  }

  /**
   * Indica se tipo  de entrada.
   *
   * @return indicativo
   */
  public boolean isInput() {
    return this.isInput;
  }

  /**
   * Indica se tipo  de sada.
   *
   * @return indicativo
   */
  public boolean isOutput() {
    return this.isOutput;
  }

  /**
   * Faz validao, indicando corretude.
   *
   * @return indicativo
   */
  public boolean validate() {
    if (!isWellFormed()) {
      return false;
    }
    if (isVisible && fileParameter.isEnabled()) {
      if (!fileParameter.isOptional() && fileParameter
        .usesPipe() == FileParameterPipeAcceptance.ALWAYS) {
        if (isInput) {
          if (getLinkTo() == null) {
            return false;
          }
        }
        else if (isOutput) {
          if (getLinkFromCollection().isEmpty()) {
            return false;
          }
        }
      }
    }
    return true;
  }

  /**
   * Indica formao correta.
   *
   * @return indicativo
   */
  public boolean isWellFormed() {
    if (fileParameter == null) {
      return false;
    }
    if (!isVisible) {
      if (getLinkTo() != null) {
        return false;
      }
      if (!getLinkFromCollection().isEmpty()) {
        return false;
      }
    }
    return true;
  }

  /**
   * Indica visibilidade.
   *
   * @return indicativo
   */
  public boolean isVisible() {
    return isVisible || !isWellFormed();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    String[] fileTypes = null;
    if (getFileParameter() != null) {
      fileTypes = getFileParameter().getFileTypes();
    }
    String typeDescription;
    if (fileTypes == null || fileTypes.length == 0) {
      String unknownType = LNG.get(GraphFileDescriptor.class.getName()
        + ".unknownFileType");
      typeDescription = "[" + unknownType + "]";
    }
    else {
      typeDescription = Arrays.toString(fileTypes);
    }

    return getParameterLabel() + " " + typeDescription;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TypeVO pick(final Point2D pt) {
    if (!isVisible()) {
      return null;
    }
    return super.pick(pt);
  }

  /**
   * Adiciona uma aresta que se origina desta viso de descritor.
   *
   * @param link A aresta.
   */
  void addLinkFrom(final GraphLink link) {
    this.linkFromSet.add(link);
    if (getFileParameter() != null) {
      getFileParameter().setHasLink(true);
    }
  }

  /**
   * Verifica se esta viso de descritor contm o ponto dado.
   *
   * @param point O ponto.
   * @return <code>true</code> se contm ou <code>false</code> caso contrrio.
   */
  boolean contains(final Point2D point) {
    return getBounds2D().contains(point);
  }

  /**
   * Obtm a dica desta viso de descritor.
   *
   * @return A dica.
   */
  String getHint() {
    return toString();
  }

  /**
   * Obtm o ponto que podemos ligar uma aresta a esta viso.
   *
   * @return O ponto.
   */
  Point2D getLinkablePoint() {
    final Rectangle2D bounds = this.path.getBounds2D();
    final Point2D point = new Point();
    double y;
    if (isOutput()) {
      y = bounds.getMaxY();
    }
    else {
      y = bounds.getMinY();
    }
    point.setLocation(bounds.getCenterX(), y);
    return point;
  }

  /**
   * Remove a aresta das coleo de arestas que se originam desta viso.
   *
   * @param link A aresta.
   */
  void removeLinkFrom(final GraphLink link) {
    this.linkFromSet.remove(link);
    if (this.linkFromSet.isEmpty()) {
      if (getFileParameter() != null) {
        getFileParameter().setHasLink(false);
      }
    }
  }

  /**
   * Mofidica a aresta que se destina a essa viso.
   *
   * @param link A aresta.
   */
  void setLinkTo(final GraphLink link) {
    this.linkTo = link;
    if (this.linkTo != null) {
      if (getFileParameter() != null) {
        getFileParameter().setHasLink(true);
      }
    }
    else {
      if (getFileParameter() != null) {
        getFileParameter().setHasLink(false);
      }
    }
  }

  /**
   * Modifica a localizao desta viso.
   *
   * @param pt Nova localizao.
   */
  void setLocation(final Point2D pt) {
    createShape(pt);
    updateLinks();
    updateColor();
  }

  /**
   * Ajusta a forma da viso para que ela se alinhe a grade.
   */
  void adjustGrid() {
    Rectangle2D bounds2d = path.getBounds2D();
    double centerX = bounds2d.getCenterX();
    double centerY = bounds2d.getCenterY();
    Point2D newPoint = Grid.adjustPoint(new Point2D.Double(centerX, centerY));
    setLocation(newPoint);
  }

  /**
   * Desliga o destaque.
   */
  void turnOffHightLight() {
    updateColor();
  }

  /**
   * Liga o destaque.
   */
  void turnOnHightLight() {
    this.borderColor = this.backgroundColor = HIGHLIGHTED_COLOR;
  }

  /**
   * Cria a cor-padro.
   */
  private void createDefaultColor() {
    if (!isWellFormed()) {
      this.defaultColor = Color.RED;
    }
    else {
      final FileTypeColorManager fileTypeManager = FileTypeColorManager
        .getInstance();
      if (fileTypeManager == null) {
        this.defaultColor = Color.BLACK;
      }
      else {
        String[] fileTypes = this.fileParameter.getFileTypes();
        if (fileTypes != null) {
          if (fileTypes.length == 1) {
            this.defaultColor = fileTypeManager.getColor(fileTypes[0]);
          }
          else {
            this.defaultColor = Color.BLACK;
          }
        }
      }
    }
  }

  /**
   * Cria a forma desta viso.
   *
   * @param point O ponto.
   */
  private void createShape(final Point2D point) {
    final float centerX = (float) point.getX();
    float lowerY = (float) point.getY();
    if (isOutput()) {
      lowerY += HEIGHT;
    }
    final float x = centerX - WIDTH / 2;
    final float y = lowerY - HEIGHT;
    this.path = new GeneralPath();
    this.path.moveTo(x + WIDTH / 2, y + HEIGHT);
    this.path.lineTo(x + WIDTH, y);
    this.path.lineTo(x, y);
    this.path.closePath();
  }

  /**
   * Atualizao de cor.
   */
  private void updateColor() {
    this.stroke = DEFAULT_STROKE;
    if (!isWellFormed()) {
      this.backgroundColor = ERROR_COLOR;
      this.borderColor = ERROR_COLOR;
    }
    else if (isBypassed()) {
      this.backgroundColor = BYPASSED_COLOR;
      this.borderColor = BYPASSED_COLOR;
    }
    else if (getFileParameter().isEnabled()) {
      if (fileParameter.usesPipe() == FileParameterPipeAcceptance.ALWAYS) {
        this.borderColor = this.defaultColor;
        this.backgroundColor = EMPTY_BACKGROUND_COLOR;
        this.stroke = THICKER_STROKE;
      }
      else {
        this.backgroundColor = this.defaultColor;
        this.borderColor = BORDER_COLOR;
      }
    }
    else {
      this.backgroundColor = DISABLED_COLOR;
      this.borderColor = DISABLED_COLOR;
    }
  }

  /**
   * Determina se o descritor est desativado, de acordo com o estado do n.
   *
   * @return verdadeiro se o descritor est desativado ou falso, caso contrrio.
   */
  private boolean isBypassed() {
    GraphNode node = getNode();
    if (node != null) {
      return node.isBypassed();
    }
    return false;
  }

  /**
   * Atualizao de visibilidade.
   */
  private void updateVisibility() {
    if (fileParameter != null) {
      switch (fileParameter.usesPipe()) {
        case ALWAYS:
          isVisible = fileParameter.isVisible();
          break;
        case FALSE:
          isVisible = false;
          break;
        default:
          if (fileParameter.getValue() != null) {
            isVisible = false;
          }
          else {
            isVisible = fileParameter.isVisible();
          }
      }
    }
    else {
      isVisible = true;
    }
    updateColor();
  }

  /**
   * Atualiza as arestas que se originam desta viso ou que se destina a essa
   * viso.
   */
  private void updateLinks() {
    if (this.linkTo != null) {
      this.linkTo.updateEndPoint();
    }
    else {
      final Iterator<GraphLink> linkFromIterator = this.linkFromSet.iterator();
      while (linkFromIterator.hasNext()) {
        final GraphLink link = linkFromIterator.next();
        link.updateStartPoint();
      }
    }
  }

  /**
   * Listener de n
   *
   * @author Tecgraf/PUC-Rio
   */
  private final class NodeListener implements GraphElementListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void wasAnchored(final GraphLink link,
      final GraphFileDescriptor fd) {
      updateColor();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasDragged(final GraphElement element, final Point2D startPoint,
      final Point2D endPoint) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasDragged(final GraphElement element, final double tx,
      final double ty) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasDropped(final GraphElement element, final Point2D point) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasIncreased(final GraphLink link) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasParameterSetVisible(final GraphNode node,
      final String paramName, final boolean visible) {
      updateVisibility();
      vs.repaint();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasParameterSetEnabled(final GraphNode n,
      final String paramName, final boolean isEnabled) {
      updateColor();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasParametrized(final GraphElement element) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasLinkStatusChanged(final GraphLink link) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasRenamed(final GraphNode node) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasResized(final GraphNode n) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasSelected(final GraphElement element) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasUnanchored(final GraphLink link,
      final GraphFileDescriptor fd) {
      updateColor();
    }
  }
}
