package csbase.client.applications.preferenceviewer;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashSet;
import java.util.Set;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.preferenceviewer.tree.PreferenceTreePanel;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.util.PreferencesUtil;
import csbase.client.util.gui.CardPanel;

/**
 * Painel que exibe a rvore de categorias e os valores das suas preferncias.
 * 
 * @author Tecgraf
 */
public class PreferencePanel extends JPanel {

  /** Referncia para a aplicao. */
  private PreferenceViewer app;

  /** Boto que restaura os valores default. */
  private JButton restore;

  /** Painel que exibe os valores das preferncias da categoria selecionada. */
  private CardPanel valuesPanel;

  /** Painel que exibe a rvore de categorias de preferncias. */
  private PreferenceTreePanel treePanel;

  /** Define se uma configurao de preferncias locais ser salva. */
  private JCheckBox saveLocalPreferences;

  /** Conjunto com todas as categorias que possuem preferncias modificadas. */
  private Set<PreferenceCategory> modifiedCategories;

  /**
   * Construtor padro.
   * 
   * @param app - referncia para a aplicao.
   */
  public PreferencePanel(PreferenceViewer app) {
    super(new GridBagLayout());

    this.app = app;

    restore = createRestoreButton();
    saveLocalPreferences = createSaveLocalPreferencesCheckBox();
    valuesPanel = new CardPanel(true, true);
    treePanel = createTreePanel();
    modifiedCategories = new HashSet<PreferenceCategory>();

    buildInterface(true);
  }

  /**
   * Mtodo que reconstri a interface do painel principal.
   * 
   * @param pc - categoria que ser exibida como raz.
   */
  public void rebuidInterface(PreferenceCategory pc) {
    removeAll();
    treePanel.rebuildTree(pc);
    saveLocalPreferences.setVisible(pc.isCopy());

    boolean hasChildren = !pc.getCategories().isEmpty();
    buildInterface(hasChildren);

    updateUI();
  }

  /**
   * Retorna o conjunto das categorias que possuem preferncias modificadas.
   * 
   * @return conjunto das categorias que possuem preferncias modificadas.
   */
  public Set<PreferenceCategory> getModifiedCategories() {
    return modifiedCategories;
  }

  /**
   * Adiciona uma categoria que teve algum valor de preferncia modificado.
   * 
   * @param pc - categoria modificada.
   */
  public void addModifiedCategory(PreferenceCategory pc) {
    modifiedCategories.add(pc);
  }

  /**
   * Exibe o painel com todas as preferncias da categoria.
   * 
   * @param pc - categoria de preferncia.
   */
  public void showPreferencePanel(PreferenceCategory pc) {
    if (!valuesPanel.has(pc.toString())) {
      addPreferencePanel(pc, false);
    }

    valuesPanel.show(pc.toString());

    ApplicationFrame frame = app.getApplicationFrame();

    Dimension preferred = frame.getPreferredSize();
    Dimension current = frame.getSize();
    if (preferred.width > current.width) {
      current.width = preferred.width;
    }
    if (preferred.height > current.height) {
      current.height = preferred.height;
    }
    frame.setSize(current);
  }

  /** Remove todos os ouvintes cadastrados. */
  public void removeAllListeners() {
    treePanel.removeAllListeners();
  }

  /**
   * Mtodo pacote que retorna a referncia para o boto que restaura os valores
   * default. Este mtodo serve apenas para que a aplicao possa setar o
   * tamanho do boto de acordo com o tamanho dos botes 'ok' e 'cancel'.
   * 
   * @return boto que restaura os valores default.
   */
  JButton getRestoreButton() {
    return restore;
  }

  /**
   * True se for para salvar as configuraes das preferncias locais
   * definitivamente, false caso contrrio.
   * 
   * @return true se for para salvar as configuraes das preferncias locais
   *         definitivamente, false caso contrrio.
   */
  boolean saveLocalPreferences() {
    return saveLocalPreferences.isSelected();
  }

  /**
   * Constroi a interface do painel.
   * 
   * @param showTree - true se for para exibir a rvore, false caso contrrio.
   */
  private void buildInterface(boolean showTree) {
    JPanel rightPanel = new JPanel(new GridBagLayout());
    rightPanel
      .add(new JScrollPane(valuesPanel), new GBC(0, 0).both().insets(5));
    rightPanel.add(new JLabel(), new GBC(0, 1).horizontal());
    rightPanel.add(createInfPanel(), new GBC(0, 2).horizontal().insets(5));

    if (showTree) {
      JSplitPane mainPanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
      mainPanel.add(treePanel);
      mainPanel.add(rightPanel);
      mainPanel.setOneTouchExpandable(true);

      add(mainPanel, new GBC(0, 0).both());
    }
    else {
      add(rightPanel, new GBC(0, 0).both());
    }
  }

  /**
   * Adiciona painel de preferncia de uma determinada categoria.
   * 
   * @param pc - categoria de preferncia.
   * @param showDefaultValues - true se for para criar o painel com os valores
   *        default, false caso contrrio.
   */
  private void addPreferencePanel(PreferenceCategory pc,
    boolean showDefaultValues) {

    JPanel titlePanel = createTitlePanel(pc);
    JPanel prefsPanel =
      PreferencesUtil.createPreferencePanel(pc, showDefaultValues);

    JPanel panel = new JPanel(new GridBagLayout());
    panel.add(titlePanel, new GBC(0, 0).horizontal());
    panel.add(new JSeparator(JSeparator.HORIZONTAL), new GBC(0, 1).horizontal()
      .left(10).right(10));
    panel.add(prefsPanel, new GBC(0, 2).both());

    valuesPanel.add(panel, pc.toString());
  }

  /**
   * Cria o painel que exibe a rvore.
   * 
   * @return painel que exibe a rvore.
   */
  private PreferenceTreePanel createTreePanel() {
    PreferenceTreePanel panel = new PreferenceTreePanel(this, app);
    panel.defaultSelection();
    return panel;
  }

  /**
   * Cria o painel que contm o ttulo de uma categoria.
   * 
   * @param pc - categoria de uma preferncia.
   * @return painel.
   */
  private JPanel createTitlePanel(PreferenceCategory pc) {
    JLabel label = new JLabel(pc.getLabel());

    JPanel panel = new JPanel(new GridBagLayout());
    panel.add(label, new GBC(0, 0).top(10).left(10).bottom(5).west());
    panel.add(new JLabel(), new GBC(1, 0).horizontal());

    return panel;
  }

  /**
   * Cria o painel que possui operaes aplicveis apenas para a categoria
   * selecionada, por exemplo, o boto 'restaurar default'.
   * 
   * @return painel com operaes aplicveis apenas para a categoria
   *         selecionada.
   */
  private JPanel createInfPanel() {
    JPanel infPanel = new JPanel(new GridBagLayout());
    infPanel.add(saveLocalPreferences, new GBC(0, 0).horizontal());
    infPanel.add(new JLabel(), new GBC(1, 0).horizontal());
    infPanel.add(restore, new GBC(2, 0));

    return infPanel;
  }

  /**
   * Cria o boto que restaura os valores default de uma categoria.
   * 
   * @return boto que restaura os valores default de uma categoria.
   */
  private JButton createRestoreButton() {
    JButton restore = new JButton(app.getString("button.restore"));

    restore.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        PreferenceCategory lastSelected = treePanel.getLastSelected();
        valuesPanel.remove(lastSelected.toString());
        addPreferencePanel(lastSelected, true);
        showPreferencePanel(lastSelected);
        addModifiedCategory(lastSelected);
        updateUI();
      }
    });

    return restore;
  }

  /**
   * Cria o checkbox que define se a configurao das preferncias locais sero
   * persistidas.
   * 
   * @return checkbox.
   */
  private JCheckBox createSaveLocalPreferencesCheckBox() {
    JCheckBox checkBox = new JCheckBox(app.getString("save.as.default"));
    checkBox.setFont(new Font(Font.SANS_SERIF, Font.PLAIN, 13));
    checkBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        addModifiedCategory(treePanel.getLastSelected());
      }
    });
    return checkBox;
  }

}
