/*
 * $Id: AdminFrame.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.ias.server.ServerInfoManagerPanel;

/**
 * A classe <code>AdminFrame</code> representa a janela principal da interface
 * de administrao do sistema. As opes que o administrador pode selecionar
 * so:
 * 
 * <ul>
 * <li>
 * gerncia de usurios</li>
 * <li>
 * gerncia de grupos de usurios</li>
 * <li>
 * gerncia de plataformas de execuo</li>
 * <li>
 * gerncia de perfis de usurios</li>
 * <li>
 * gerncia de permisses de usurios</li>
 * </ul>
 * 
 * @author Tecgraf
 */
public class AdminFrame {

  /**
   * Frame principal de administrao do sistema.
   */
  protected DesktopComponentFrame mainFrame;

  /**
   * Painel com as opes de gerncia do sistema.
   */
  protected JTabbedPane tabbedPane;

  /**
   * Array de painis de gerncia.
   */
  protected Component[] panes;

  /**
   * Constri o painel principal da janela com seus botes.
   * 
   * @return o painel principal da janela de administrao do sistema
   */
  private JPanel makePanel() {
    JPanel panel = new JPanel(new BorderLayout());
    panel.add(makeTabbedPane(), BorderLayout.CENTER);
    panel.add(makeCloseButton(), BorderLayout.SOUTH);
    return panel;
  }

  /**
   * Constri uma interface tabulada com as diversas opes de administrao do
   * sistema.
   * 
   * @return um painel com as opes de administrao
   */
  protected JComponent makeTabbedPane() {
    JPanel ptb = new JPanel(new GridLayout(1, 1));
    tabbedPane = new JTabbedPane();
    panes = new Component[6];
    panes[0] = new UserManagerPanel(mainFrame);
    tabbedPane.addTab(LNG.get("IAS_USERS"), panes[0]);
    panes[1] = new UserGroupManagerPanel(mainFrame);
    tabbedPane.addTab(LNG.get("IAS_USERGROUPS"), panes[1]);
    panes[2] = new RoleManagerPanel(mainFrame);
    tabbedPane.addTab(LNG.get("IAS_ROLES"), panes[2]);
    panes[3] = new PermissionManagerPanel(mainFrame);
    tabbedPane.addTab(LNG.get("IAS_PERMISSIONS"), panes[3]);
    panes[4] = new PlatformManagerPanel(mainFrame);
    tabbedPane.addTab(LNG.get("IAS_PLATFORMS"), panes[4]);
    panes[5] = new ServerInfoManagerPanel(mainFrame);
    tabbedPane.addTab(LNG.get("IAS_SERVERS"), panes[5]);
    ptb.add(tabbedPane);
    return ptb;
  }

  /**
   * Constri o boto "Fechar" da janela. Esse boto  reponsvel por fechar a
   * janela e terminar a aplicao.
   * 
   * @return um painel com o boto "Fechar".
   */
  private JComponent makeCloseButton() {
    JPanel buttonPanel = new JPanel();
    JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        close();
      }
    });
    buttonPanel.add(closeButton);
    return buttonPanel;
  }

  /**
   * Traz para a frente a janela de administrao do sistema.
   */
  public void bringToFront() {
    mainFrame.setVisible(true);
    mainFrame.toFront();
  }

  /**
   * Encerra a exibio da IAS.
   */
  public void close() {
    for (int i = 0; i < panes.length; i++) {
      if (panes[i] instanceof ManagerPanel) {
        ((ManagerPanel) panes[i]).beforeClose();
      }
    }
    if (mainFrame != null) {
      mainFrame.close();
      mainFrame = null;
    }
  }

  /**
   * Cria uma janela de administrao do sistema. A janela  centralizada na
   * tela.
   */
  public AdminFrame() {
    makeMainFrame();
  }

  /**
   * Montagem do frame.
   */
  public void makeMainFrame() {
    mainFrame = new DesktopComponentFrame(LNG.get("IAS_ADMINISTRATION_TITLE"));
    mainFrame.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        close();
      }
    });
    mainFrame.getContentPane().add(makePanel(), BorderLayout.CENTER);
    mainFrame.pack();
    mainFrame.setMinimumSize(new Dimension(800, 500));
    if (DesktopFrame.getInstance() == null) {
      mainFrame.setLocationRelativeTo(null);
    }
    else {
      mainFrame.center(DesktopFrame.getInstance().getDesktopFrame());
    }
  }

  /**
   * Mostra a janela de administrao do sistema, abrindo na pgina de gerncia
   * de usurios.
   */
  public void showUserManagement() {
    if (mainFrame == null) {
      makeMainFrame();
    }
    tabbedPane.setSelectedIndex(0);
    mainFrame.setVisible(true);
  }

  /**
   * Mostra a janela de administrao do sistema, abrindo na pgina de gerncia
   * de grupos de usurio.
   */
  public void showUserGroupManagement() {
    if (mainFrame == null) {
      makeMainFrame();
    }
    tabbedPane.setSelectedIndex(1);
    mainFrame.setVisible(true);
  }

  /**
   * Mostra a janela de administrao do sistema, abrindo na pgina de gerncia
   * de perfis.
   */
  public void showRoleManagement() {
    if (mainFrame == null) {
      makeMainFrame();
    }
    tabbedPane.setSelectedIndex(2);
    mainFrame.setVisible(true);
  }

  /**
   * Mostra a janela de administrao do sistema, abrindo na pgina de gerncia
   * de permisses.
   */
  public void showPermissionManagement() {
    if (mainFrame == null) {
      makeMainFrame();
    }
    tabbedPane.setSelectedIndex(3);
    mainFrame.setVisible(true);
  }

  /**
   * Mostra a janela de administrao do sistema, abrindo na pgina de gerncia
   * de plataformas de execuo.
   */
  public void showPlatformManagement() {
    if (mainFrame == null) {
      makeMainFrame();
    }
    tabbedPane.setSelectedIndex(4);
    mainFrame.setVisible(true);
  }

  /**
   * Mostra a janela de administrao do sistema, abrindo na pgina de gerncia
   * de servidores.
   */
  public void showServerManagement() {
    if (mainFrame == null) {
      makeMainFrame();
    }
    tabbedPane.setSelectedIndex(5);
    mainFrame.setVisible(true);
  }

  /**
   * Obtm a janela principal.
   * 
   * @return JFrame a janela principal.
   */
  public JFrame getMainFrame() {
    return this.mainFrame;
  }
}
