/*
 * PermissionProxy.java $Author: fpina $ $Revision: 176168 $ - $Date:
 * 2008-02-11 18:35:52 -0200 (Mon, 11 Feb 2008) $
 */
package csbase.client.remote.srvproxies;

import java.awt.Window;
import java.util.List;
import java.util.Vector;

import javax.swing.JOptionPane;

import csbase.client.desktop.RemoteTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.administration.AdministrationDeleteException;
import csbase.logic.Permission;
import csbase.logic.Role;
import csbase.logic.RoleInfo;
import csbase.logic.User;
import csbase.logic.UserInfo;
import csbase.logic.Utilities;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Classe que modela um proxy para Permission.
 * 
 * @author Julia
 */
public class PermissionProxy {
  /**
   * Obtem a lista de Permisses cadastradas. TODO - Mudar tipo de retorno para
   * {@code List<Permission>}
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * 
   * @return o vetor de Permisses.
   */
  public static Vector<Permission> getAllPermissions(Window owner, String title,
    String message) {
    RemoteTask<Vector<Permission>> task = new RemoteTask<Vector<Permission>>() {
      @Override
      public void performTask() throws Exception {
        setResult(new Vector<Permission>(Permission.getAllPermissions()));
      }
    };
    if (task.execute(owner, title, message)) {
      Vector<Permission> unsVector = task.getResult();
      if ((unsVector == null) || (unsVector.size() <= 0)) {
        return new Vector<Permission>();
      }
      return unsVector;
    }
    return new Vector<Permission>();
  }

  /**
   * Obtm uma Permisso com uma dada identificao.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param permissionId o identificador da Permisso.
   * 
   * @return a Permisso requisitada.
   */
  public static Permission getPermission(Window owner, String title,
    String message, final Object permissionId) {
    RemoteTask<Permission> task = new RemoteTask<Permission>() {
      @Override
      public void performTask() throws Exception {
        setResult(Permission.getPermission(permissionId));
      }
    };
    if (task.execute(owner, title, message)) {
      return task.getResult();
    }
    return null;
  }

  /**
   * Cria uma Permisso.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param permission a permisso a ser criada (sem o id).
   * 
   * @return a permisso criada (com o id).
   */
  public static Permission createPermission(Window owner, String title,
    String message, final Permission permission) {
    RemoteTask<Permission> task = new RemoteTask<Permission>() {
      @Override
      public void performTask() throws Exception {
        setResult(Permission.createPermission(permission));
      }
    };
    if (task.execute(owner, title, message)) {
      Permission perm = task.getResult();
      if (perm == null) {
        // Se no ocorreu nenhuma exceo mas a task retornou nulo,  porque a
        // j existe outra permisso com o mesmo nome
        StandardErrorDialogs.showErrorDialog(owner, title, LNG.get(
          "IAS_PERMISSION_ALREADY_EXISTS_ERROR"));
      }
      return perm;
    }
    return null;
  }

  /**
   * Modifica uma Permisso.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param permissionId o identificador da Permisso a ser alterada.
   * @param modifiedPermission As informaes para modificao da Permisso.
   * 
   * @return A Permisso modificada.
   */
  public static Permission modifyPermission(Window owner, String title,
    String message, final Object permissionId,
    final Permission modifiedPermission) {
    RemoteTask<Permission> task = new RemoteTask<Permission>() {
      @Override
      public void performTask() throws Exception {
        setResult(Permission.modifyPermission(permissionId,
          modifiedPermission));
      }
    };
    if (task.execute(owner, title, message)) {
      Permission perm = task.getResult();
      if (perm == null) {
        // Se no ocorreu nenhuma exceo mas a task retornou nulo,  porque est
        // tentando-se modificar o nome de uma permisso, mas j existe outra com
        // o mesmo nome
        StandardErrorDialogs.showErrorDialog(owner, title, LNG.get(
          "IAS_PERMISSION_ALREADY_EXISTS_ERROR"));
      }
      return perm;
    }
    return null;
  }

  /**
   * Remove uma Permisso.
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * @param permissionId o identificador da Permisso a ser removida.
   */
  public static void deletePermission(final Window owner, final String title,
    String message, final Object permissionId) {
    class DeletePermissionRemoteTask extends RemoteTask<Void> {
      boolean removeUsersPermission = false;

      @Override
      public void performTask() throws Exception {
        Permission.deletePermission(permissionId);
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof AdministrationDeleteException) {
          String errorMsg = LNG.get("IAS_CONFIRM_PERMISSION_REMOVAL");
          int res = StandardDialogs.showYesNoDialog(owner, title, errorMsg);
          if (res == JOptionPane.YES_OPTION) {
            removeUsersPermission = true;
          }
        }
        else {
          super.handleError(error);
        }
      }
    }
    DeletePermissionRemoteTask deletetask = new DeletePermissionRemoteTask();

    if (!deletetask.execute(owner, title, message)
      && deletetask.removeUsersPermission) {

      RemoteTask<Void> task = new RemoteTask<Void>() {
        @Override
        public void performTask() throws Exception {
          // Procura por perfis que tenham essa permisso
          List<Role> roles = Role.getAllRoles();
          for (int i = 0; i < roles.size(); i++) {
            Role role = roles.get(i);
            Object[] permissionIds = role.getPermissionIds();
            for (int j = 0; j < permissionIds.length; j++) {
              if (permissionIds[j].equals(permissionId)) {
                // Remove essa permisso de conjunto de permisses do
                // perfil
                RoleInfo roleInfo = role.getRoleInfo();
                roleInfo.permissionIds = Utilities.removeItemFromArray(
                  permissionIds, j);
                Role.modifyRole(role.getId(), roleInfo);
                break;
              }
            }
          }

          // Procura por usurios que tenham esta permisso
          List<User> users = User.getAllUsers();
          for (int i = 0; i < users.size(); i++) {
            User user = users.get(i);
            Object[] permissionIds = user.getPermissionIds();
            for (int j = 0; j < permissionIds.length; j++) {
              if (permissionIds[j].equals(permissionId)) {
                // Remove esse perfil do conjunto de perfis do usurio
                Object[] remainingPermissionIds = Utilities.removeItemFromArray(
                  permissionIds, j);
                UserInfo userInfo = user.getUserInfo();
                userInfo.setAttribute(User.PERMISSION_IDS,
                  remainingPermissionIds);
                User.modifyUser(user.getId(), userInfo);
                user.setPermissionIds(remainingPermissionIds);
                break;
              }
            }
          }

          // Como no h mais perfis ou usurios com essa permisso,
          // pode-se remov-la
          Permission.deletePermission(permissionId);
        }
      };
      task.execute(owner, title, message);
    }
  }

  /**
   * Obtm a lista de Classes de Permisses cadastradas. TODO - Mudar tipo de
   * retorno para {@code List<String>}
   * 
   * @param owner janela pai para efeito de hierarquia de dilogos.
   * @param title titulo da janela da <code>DesktopTask</code> associada.
   * @param message mensagem exibida na janela da <code>DesktopTask</code>
   *        associada.
   * 
   * @return o vetor de Classes de Permisses.
   */
  public static Vector<String> getPermissionClasses(Window owner, String title,
    String message) {
    RemoteTask<Vector<String>> task = new RemoteTask<Vector<String>>() {
      @Override
      public void performTask() throws Exception {
        setResult(new Vector<String>(Permission.getPermissionClasses()));
      }
    };
    if (task.execute(owner, title, message)) {
      Vector<String> classesVector = task.getResult();
      if ((classesVector == null) || (classesVector.size() <= 0)) {
        return new Vector<String>();
      }
      return classesVector;
    }
    return new Vector<String>();
  }
}
