/*
 * $Id:$
 */

package csbase.client.util.filechooser;

import java.awt.Window;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileView;

import csbase.client.ClientLocalFile;
import csbase.client.project.ClientProjectFileType;
import csbase.client.util.filechooser.filters.ClientFileAllFilter;
import csbase.client.util.filechooser.filters.ClientFileFilterInterface;
import csbase.client.util.filechooser.filters.ClientFileMultipleTypesFilter;
import csbase.client.util.filechooser.filters.ClientFileSingleTypeFilter;
import csbase.client.util.filechooser.util.ClientFileChooserUtils;
import csbase.logic.ClientFile;
import csbase.logic.ProjectFileType;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Utilitrio para busca de arquivos locais.
 *
 * @author Tecgraf
 */
public class ClientLocalFileChooserUtil {

  /**
   * Filtro para acesso a arquivos com base em extenses.
   *
   * @author Tecgraf
   */
  private static final class LocalFileFilter extends FileFilter {
    /**
     * Indicativo de uso de todos os tipos
     */
    private final String fileCode;

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean accept(final File f) {
      if (f.isDirectory()) {
        return true;
      }

      final ProjectFileType fileType = ProjectFileType.getFileType(fileCode);
      final List<String> extensions = fileType.getExtensions();
      final String fileName = f.getName();
      final String extension = FileUtils.getFileExtension(fileName);
      if (extensions.contains(extension)) {
        return true;
      }
      return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getDescription() {
      final ProjectFileType fileType = ProjectFileType.getFileType(fileCode);
      String desc = fileType.getDescription();
      desc = desc + " - ";

      final List<String> extensions = fileType.getExtensions();
      for (final String extension : extensions) {
        desc = desc + " (*." + extension + ") ";
      }
      return desc;
    }

    /**
     * Construtor.
     *
     * @param fileCode tipo do arquivo.
     */
    private LocalFileFilter(final String fileCode) {
      this.fileCode = fileCode;
    }
  }

  /**
   *
   *
   * @author Tecgraf
   */
  static class LocalFileView extends FileView {

    /**
     * {@inheritDoc}
     *
     * Deixa para o L&F descobrir
     */
    @Override
    public String getDescription(final File file) {
      return null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Icon getIcon(final File file) {
      final String path = file.getAbsolutePath();
      final String extension = FileUtils.getFileExtension(path);
      final ProjectFileType fileType =
        ProjectFileType.getProjectFileTypeFromExtension(extension,
        file.isDirectory());
      final String code = fileType.getCode();
      final ClientProjectFileType projFileType =
        ClientProjectFileType.getFileType(code);
      final ImageIcon icon = projFileType.getBaseIcon();
      return icon;
    }

    /**
     * {@inheritDoc}
     *
     * Deixa para o L&F descobrir
     */
    @Override
    public String getName(final File file) {
      return null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getTypeDescription(final File file) {
      final String path = file.getAbsolutePath();
      final String extension = FileUtils.getFileExtension(path);
      final ProjectFileType fileType =
        ProjectFileType.getProjectFileTypeFromExtension(extension,
        file.isDirectory());
      final String desc = fileType.getDescription();
      return desc;
    }

    /**
     * {@inheritDoc}
     *
     * Deixa para o L&F descobrir
     */
    @Override
    public Boolean isTraversable(final File file) {
      return null;
    }
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis.
   *
   *
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public ClientLocalFile browseSingleFileInOpenMode(
    final Window window, final Collection<String> fileTypes,
    final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory) {
    final boolean useNewChooser = useNewChooser();
    if (useNewChooser) {
      return newBrowseSingleFileInOpenMode(window, fileTypes, title, allowAll,
        currentDirectory, ClientFileChooserSelectionMode.FILES_ONLY);
    }
    return oldBrowseSingleFileInOpenMode(window, fileTypes, title, allowAll,
      currentDirectory, JFileChooser.FILES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis.
   *
   *
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public List<ClientLocalFile> browseMultipleFilesInOpenMode(
    final Window window, final Collection<String> fileTypes,
    final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory) {
    final boolean useNewChooser = useNewChooser();
    if (useNewChooser) {
      return newBrowseMultipleFilesInOpenMode(window, fileTypes, title,
        allowAll, currentDirectory, ClientFileChooserSelectionMode.FILES_ONLY);
    }
    return oldBrowseMultipleFilesInOpenMode(window, fileTypes, title, allowAll,
      currentDirectory, JFileChooser.FILES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de diretrios da locais com 
   * base em lista de tipos aceitveis.
   *
   *
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public List<ClientLocalFile> browseMultipleDirectoriesInOpenMode(
    final Window window, final Collection<String> fileTypes,
    final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory) {
    final boolean useNewChooser = useNewChooser();
    if (useNewChooser) {
      return newBrowseMultipleFilesInOpenMode(window, fileTypes, title,
        allowAll, currentDirectory, ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
    }
    return oldBrowseMultipleFilesInOpenMode(window, fileTypes, title, allowAll,
      currentDirectory, JFileChooser.DIRECTORIES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis.
   *
   * @param window a janela-me (owner).
   * @param fileCodes a lista de tipos a serem aceitos
   * @param defaultFileType tipo default.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public ClientLocalFile browseSingleFileInSaveMode(
    final Window window, final Collection<String> fileCodes,
    final String defaultFileType, final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory) {
    final boolean useNewChooser = useNewChooser();
    if (useNewChooser) {
      return newBrowseSingleFileInSaveMode(window, fileCodes, defaultFileType,
        title, allowAll, currentDirectory,
        ClientFileChooserSelectionMode.FILES_ONLY);
    }
    return oldBrowseSingleFileInSaveMode(window, fileCodes, defaultFileType,
      title, allowAll, currentDirectory, JFileChooser.FILES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos locais com base
   * em lista de tipos aceitveis.
   *
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param defaultFileType tipo default.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public ClientLocalFile browseSingleDirectoryInSaveMode(
    final Window window, final Collection<String> fileTypes,
    final String defaultFileType, final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory) {
    final boolean useNewChooser = useNewChooser();
    if (useNewChooser) {
      return newBrowseSingleFileInSaveMode(window, fileTypes, defaultFileType,
        title, allowAll, currentDirectory,
        ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
    }
    return oldBrowseSingleFileInSaveMode(window, fileTypes, defaultFileType,
      title, allowAll, currentDirectory, JFileChooser.DIRECTORIES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de diretrios locais com base
   * em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public ClientLocalFile browseSingleDirectoryInOpenMode(
    final Window window, Collection<String> fileTypes, final String title,
    final boolean allowAll, final ClientLocalFile currentDirectory) {
    final boolean useNewChooser = useNewChooser();
    if (useNewChooser) {
      return newBrowseSingleFileInOpenMode(window, fileTypes, title, allowAll,
        currentDirectory, ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
    }
    return oldBrowseSingleFileInOpenMode(window, fileTypes, title, allowAll,
      currentDirectory, JFileChooser.DIRECTORIES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis. final ProjectFileType fileType =
   * ProjectFileType.getFileType(fileCode); final List<String> extensions =
   * fileType.getExtensions();
   *
   *
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @param selectionMode modo de seleo (arquivo ou diretrio).
   *
   * @return o arquivo escolhido ou {@code null}.
   */
  private static ClientLocalFile newBrowseSingleFileInOpenMode(
    final Window window, final Collection<String> fileTypes,
    final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory,
    ClientFileChooserSelectionMode selectionMode) {
    final ClientFileChooser chooser = new ClientFileChooser(window);
    chooser.setTitle(title);
    chooser.setSelectionMode(selectionMode);
    chooser.setSelectionType(ClientFileChooserType.OPEN);
    chooser.setLocalHomeButtonVisible(true);
    chooser.setLocalRootButtonVisible(true);
    chooser.setProjectHomeButtonVisible(false);
    chooser.setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);

    final ClientLocalFile currentDir = getCurrentDirectory(currentDirectory);
    chooser.setCurrentDirectory(currentDir);

    final List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();

    if (fileTypes != null) {
      if (fileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(fileTypes));
      }
      for (final String fileType : fileTypes) {
        final ClientFileSingleTypeFilter flt =
          new ClientFileSingleTypeFilter(fileType);
        filters.add(flt);
      }
    }

    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }

    chooser.setViewFilters(filters);
    if (!filters.isEmpty()) {
      chooser.setSelectedFilter(filters.get(0));
    }
    chooser.setVisible(true);
    final List<ClientFile> selected = chooser.getChosenItens();
    if (selected == null || selected.size() == 0) {
      return null;
    }
    final ClientLocalFile first = (ClientLocalFile) selected.get(0);
    return first;
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis. final ProjectFileType fileType =
   * ProjectFileType.getFileType(fileCode); final List<String> extensions =
   * fileType.getExtensions();
   *
   *
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @param selectionMode modo de seleo (arquivo ou diretrio).
   * @return o arquivo escolhido ou {@code null}.
   */
  private static List<ClientLocalFile> newBrowseMultipleFilesInOpenMode(
    final Window window, final Collection<String> fileTypes,
    final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory,
    final ClientFileChooserSelectionMode selectionMode) {
    final ClientFileChooser chooser = new ClientFileChooser(window);
    chooser.setTitle(title);
    chooser.setSelectionMode(selectionMode);
    chooser.setSelectionType(ClientFileChooserType.OPEN);
    chooser.setLocalHomeButtonVisible(true);
    chooser.setLocalRootButtonVisible(true);
    chooser.setProjectHomeButtonVisible(false);
    chooser.setCardinality(ClientFileChooserCardinality.MULTIPLE_CHOOSE);

    final ClientLocalFile currentDir = getCurrentDirectory(currentDirectory);
    chooser.setCurrentDirectory(currentDir);

    final List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();

    if (fileTypes != null) {
      if (fileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(fileTypes));
      }

      for (final String fileType : fileTypes) {
        final ClientFileSingleTypeFilter flt =
          new ClientFileSingleTypeFilter(fileType);
        filters.add(flt);
      }
    }

    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }

    chooser.setViewFilters(filters);
    if (!filters.isEmpty()) {
      chooser.setSelectedFilter(filters.get(0));
    }
    chooser.setVisible(true);
    final List<ClientFile> selected = chooser.getChosenItens();
    if (selected == null) {
      return null;
    }

    final List<ClientLocalFile> localSelected =
      new ArrayList<ClientLocalFile>();
    for (ClientFile clientFile : selected) {
      final ClientLocalFile clientLocalFile = (ClientLocalFile) clientFile;
      localSelected.add(clientLocalFile);
    }
    return localSelected;
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis.
   *
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param defaultFileType tipo default
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da escolha.
   * @param mode modo entrada ou sada.
   * @return o arquivo escolhido ou {@code null}.
   */
  private static ClientLocalFile newBrowseSingleFileInSaveMode(Window window,
    Collection<String> fileTypes, String defaultFileType, String title,
    boolean allowAll, ClientLocalFile currentDirectory,
    ClientFileChooserSelectionMode mode) {

    ClientFileChooser chooser = new ClientFileChooser(window);
    chooser.setTitle(title);
    chooser.setSelectionMode(mode);
    chooser.setSelectionType(ClientFileChooserType.SAVE);
    chooser.setLocalHomeButtonVisible(true);
    chooser.setLocalRootButtonVisible(true);
    chooser.setProjectHomeButtonVisible(false);
    chooser.setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);

    ClientLocalFile currentDir = getCurrentDirectory(currentDirectory);
    chooser.setCurrentDirectory(currentDir);

    List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();
    List<String> allFileTypes = new ArrayList<String>();
    if (defaultFileType != null) {
      allFileTypes.add(defaultFileType);
    }
    for (String fileType : fileTypes) {
      if (!allFileTypes.contains(fileType)) {
        allFileTypes.add(fileType);
      }
    }
    ClientFileFilterInterface defaultFilter = null;
    for (final String fileType : allFileTypes) {
      ClientFileSingleTypeFilter filter =
        new ClientFileSingleTypeFilter(fileType);
      filters.add(filter);
      if (defaultFileType != null && defaultFileType.equals(fileType)) {
        defaultFilter = filter;
      }
    }
    if (allFileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(allFileTypes));
    }
    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }
    chooser.setViewFilters(filters);
    if (defaultFilter != null) {
      chooser.setSelectedFilter(defaultFilter);
    }
    chooser.setVisible(true);
    String[] selectedSavePath = chooser.getSelectedSavePath();
    if (selectedSavePath == null || selectedSavePath.length <= 0) {
      return null;
    }
    String path = FileUtils.joinPath(selectedSavePath);
    File file = new File(path);
    ClientLocalFile clientLocalFile = new ClientLocalFile(file);
    return clientLocalFile;
  }

  /**
   * Obtm o diretrio inicial de busca do dilogo de escolha de diretrios.
   *
   * @param currentDir o diretrio sugerido.
   * @return O diretrio inicial de busca. Ser retornado, em ordem de
   *         preferncia, o diretrio sugerido, o home do usurio, o root do
   *         sistema local ou o diretrio corrente. Caso nenhum desses
   *         diretrios estejam disponveis, retorna nulo.
   */
  private static ClientLocalFile getCurrentDirectory(ClientLocalFile currentDir) {
    List<ClientLocalFile> dirs = new ArrayList<ClientLocalFile>();
    dirs.add(currentDir);
    dirs.add(ClientFileChooserUtils.getLocalHomeDirectory());
    dirs.addAll(ClientFileChooserUtils.getLocalRootDirectories());
    dirs.add(new ClientLocalFile(new File(".")));
    for (ClientLocalFile dir : dirs) {
      if (dir != null && dir.isDirectory() && dir.exists()) {
        return dir;
      }
    }
    return null;
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis. final ProjectFileType fileType =
   * ProjectFileType.getFileType(fileCode); final List<String> extensions =
   * fileType.getExtensions();
   *
   *
   * @param window a janela-me (owner).
   * @param fileCodes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @param selectionMode modo de seleo ({@link JFileChooser#FILES_ONLY},
   *        {@link JFileChooser#FILES_AND_DIRECTORIES} ou
   *        {@link JFileChooser#DIRECTORIES_ONLY}).
   * @return o arquivo escolhido ou {@code null}.
   */
  private static ClientLocalFile oldBrowseSingleFileInOpenMode(
    final Window window, final Collection<String> fileCodes,
    final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory, final int selectionMode) {

    final JFileChooser fc =
      createOldLocalFileChooser(fileCodes, title, allowAll, currentDirectory,
        selectionMode);

    fc.setDialogType(JFileChooser.OPEN_DIALOG);
    fc.setMultiSelectionEnabled(false);

    final int returnVal = fc.showOpenDialog(window);
    if (returnVal != JFileChooser.APPROVE_OPTION) {
      return null;
    }

    final File file = fc.getSelectedFile();
    if (file == null || !file.exists() || !file.canRead()) {
      final String cls = ClientLocalFileChooserUtil.class.getSimpleName();
      final String err = LNG.get(cls + ".file.not.found.error");
      StandardDialogs.showErrorDialog(window, "", err);
      return null;
    }
    final ClientLocalFile localFile = new ClientLocalFile(file);
    return localFile;
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis. final ProjectFileType fileType =
   * ProjectFileType.getFileType(fileCode); final List<String> extensions =
   * fileType.getExtensions();
   *
   *
   * @param window a janela-me (owner).
   * @param fileCodes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @param selectionMode modo de seleo ({@link JFileChooser#FILES_ONLY},
   *        {@link JFileChooser#FILES_AND_DIRECTORIES} ou
   *        {@link JFileChooser#DIRECTORIES_ONLY}).
   * @return o arquivo escolhido ou {@code null}.
   */
  private static List<ClientLocalFile> oldBrowseMultipleFilesInOpenMode(
    final Window window, final Collection<String> fileCodes,
    final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory, final int selectionMode) {

    final JFileChooser fc =
      createOldLocalFileChooser(fileCodes, title, allowAll, currentDirectory,
        selectionMode);

    fc.setDialogType(JFileChooser.OPEN_DIALOG);
    fc.setMultiSelectionEnabled(true);

    final int returnVal = fc.showOpenDialog(window);
    if (returnVal != JFileChooser.APPROVE_OPTION) {
      return null;
    }

    final File[] files = fc.getSelectedFiles();
    if (files == null || files.length == 0) {
      final String cls = ClientLocalFileChooserUtil.class.getSimpleName();
      final String err = LNG.get(cls + ".file.not.found.error");
      StandardDialogs.showErrorDialog(window, "", err);
      return null;
    }

    List<ClientLocalFile> localFiles = new ArrayList<ClientLocalFile>();
    for (File file : files) {
      final ClientLocalFile localFile = new ClientLocalFile(file);
      localFiles.add(localFile);
    }
    return localFiles;
  }

  /**
   * Cria um JFileChooser com incializaces tpicas
   *
   * @param fileCodes cdigos de arquivos
   * @param title ttlo
   * @param allowAll indicativo de todos os tipos
   * @param currentDirectory diretrio corrente.
   * @param selectionMode modo de seleo ({@link JFileChooser#FILES_ONLY},
   *        {@link JFileChooser#FILES_AND_DIRECTORIES} ou
   *        {@link JFileChooser#DIRECTORIES_ONLY}).
   * @return o objeto {@link JFileChooser}.
   */
  private static JFileChooser createOldLocalFileChooser(
    final Collection<String> fileCodes, final String title,
    final boolean allowAll, final ClientLocalFile currentDirectory,
    final int selectionMode) {
    final JFileChooser fc = new JFileChooser();
    if (fileCodes != null) {
      for (final String fileCode : fileCodes) {
        fc.addChoosableFileFilter(new LocalFileFilter(fileCode));
      }
    }

    if (allowAll) {
      fc.setAcceptAllFileFilterUsed(true);
    }

    fc.setFileView(new LocalFileView());
    fc.setFileSelectionMode(selectionMode);

    fc.setDialogTitle(title);

    final ClientLocalFile dir = getCurrentDirectory(currentDirectory);
    if (dir != null) {
      fc.setCurrentDirectory(new File(dir.getStringPath()));
    }
    return fc;
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da locais com base
   * em lista de tipos aceitveis.
   *
   * @param window a janela-me (owner).
   * @param fileCodes a lista de tipos a serem aceitos
   * @param defaultFileCode tipo default
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory o diretrio inicial da janela de escolha.
   * @param mode modo entrada ou sada.
   * @return o arquivo escolhido ou {@code null}.
   */
  private static ClientLocalFile oldBrowseSingleFileInSaveMode(
    final Window window, final Collection<String> fileCodes,
    final String defaultFileCode, final String title, final boolean allowAll,
    final ClientLocalFile currentDirectory, int mode) {
    final JFileChooser fc =
      createOldLocalFileChooser(fileCodes, title, allowAll, currentDirectory,
        mode);
    fc.setFileView(new LocalFileView());

    final ClientLocalFile dir = getCurrentDirectory(currentDirectory);
    if (dir != null) {
      fc.setCurrentDirectory(new File(dir.getStringPath()));
    }

    fc.setMultiSelectionEnabled(false);
    fc.setDialogType(JFileChooser.SAVE_DIALOG);

    final int returnVal = fc.showSaveDialog(window);
    if (returnVal != JFileChooser.APPROVE_OPTION) {
      return null;
    }

    final File file = fc.getSelectedFile();
    if (file == null) {
      return null;
    }
    return new ClientLocalFile(file);
  }

  /**
   * Indica uso de novo chooser.
   *
   * @return indicativo
   */
  static private boolean useNewChooser() {
    try {
      final ConfigurationManager cnfManager =
        ConfigurationManager.getInstance();
      final Configuration cnf =
        cnfManager.getConfiguration(ClientLocalFileChooserUtil.class);
      final String propName = "new.chooser";
      final Boolean newChooser =
        cnf.getOptionalBooleanProperty(propName, false);
      return newChooser;
    }
    catch (final Exception e) {
      return false;
    }
  }
}
