package csbase.client.algorithms.parameters;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;

import csbase.client.ClientLocalFile;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.resourcehelpers.ApplicationUtils;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.client.project.tasks.GetFileTask;
import csbase.client.remote.srvproxies.SGAProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.SGASet;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.URLParameter;
import csbase.logic.algorithms.parameters.URLProtocol;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;

/**
 * A Viso para {@link URLParameter Parmetro do Tipo URL}.
 *
 * @author Tecgraf
 */
public abstract class URLParameterView extends SimpleParameterView<FileURLValue> {

  /**
   * Cria a viso.
   *
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so:
   *        {@link csbase.client.algorithms.parameters.ParameterView.Mode#CONFIGURATION}
   *        ou
   *        {@link csbase.client.algorithms.parameters.ParameterView.Mode#REPORT}
   *        .
   */
  public URLParameterView(URLParameter parameter, Mode mode) {
    super(parameter, mode);

    updateViewContents();
    updateCapabilityView();
    updateVisibilyView();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URLParameter getParameter() {
    return (URLParameter) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createConfigurationComponent(Object... componentArgs) {
    return new FileConfigurationParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createReportComponent(Object... componentArgs) {
    return new FileReportParameter();
  }

  /**
   * Abre um navegador de arquivos para um determinado protocolo.
   *
   * @param protocol protocolo.
   * @param args argumentos extras.
   * @return URL do arquivo selecionado.
   */
  protected abstract FileURLValue askForFile(URLProtocol protocol, Object... args);

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setEnabled(boolean isEnabled) {
    getComponent().setEnabled(isEnabled);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateVisibilyView() {
    boolean isVisible =
      getParameter().isVisible() && !getParameter().hasLink()
      && getParameter().usesPipe() != FileParameterPipeAcceptance.ALWAYS;
    if (isVisible() != isVisible) {
      setVisible(isVisible);
      fireVisibilityWasChanged();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateViewContents() {
    ((IFileParameterComponent) getComponent()).updateViewContents();
  }

  /**
   * Interface comum s diferentes vises do parmetro.
   */
  private interface IFileParameterComponent {
    /**
     * Atualiza o contedo exibido pela viso.
     */
    void updateViewContents();
  }

  /**
   * Viso do parmetro em modo
   * {@link csbase.client.algorithms.parameters.ParameterView.Mode#REPORT}.
   */
  private class FileReportParameter extends JPanel implements
  IFileParameterComponent {

    /**
     * O campo de texto.
     */
    private JTextField textField;

    /**
     * Construtor.
     */
    FileReportParameter() {
      setLayout(new GridBagLayout());

      textField = new JTextField();
      URLParameter parameter = getParameter();
      textField.setToolTipText(parameter.getDescription());
      ComponentProperties.setProperties(textField, Mode.REPORT, true);
      textField.setEditable(false);

      add(textField, new GBC(0, 0).horizontal());

      FileURLValue value = parameter.getValue();
      if (value != null) {
        if (value.getProtocol() == URLProtocol.PROJECT) {
          JButton button = createShowButton(textField);
          if (button != null) {
            add(button, new GBC(1, 0));
          }
        }
      }

      updateViewContents();
    }

    /**
     * Cria o boto de mostrar arquivo de log.
     *
     * @param field o campo com o nome do arquivo.
     * @return o boto.
     */
    private JButton createShowButton(JTextField field) {
      final FileURLValue value = getParameter().getValue();

      if (value == null || !value.getType().equals("LOG")) {
        return null;
      }

      JButton browseButton =
        GUIUtils.createImageButton(ApplicationImages.ICON_VIEWLOG_16);
      browseButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent ev) {
          try {
            ClientProjectFile file = getClientProjectFile(value);
            ApplicationUtils.runPreferredApp(getWindow(), file);
          }
          catch (ApplicationException e) {
            DesktopFrame desktop = DesktopFrame.getInstance();
            DesktopComponentFrame frame = desktop.getDesktopFrame();
            StandardErrorDialogs.showErrorDialog(frame, e);
          }
        }
      });
      browseButton.setToolTipText(getParameter().getDescription());
      return browseButton;
    }

    /**
     * Obtm o arquivo de projeto a partir do valor de um parmetro do tipo
     * arquivo.
     *
     * @param paramValue o valor do parmetro.
     * @return o arquivo de projeto.
     */
    private ClientProjectFile getClientProjectFile(FileURLValue paramValue) {
      CommonClientProject project = DesktopFrame.getInstance().getProject();
      String[] path = paramValue.getPathAsArray();
      return GetFileTask.runTask(getWindow(), project, path);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      FileURLValue value = getParameter().getValue();
      String text = null;
      if (value != null) {
        URLProtocol selectedProtocol = value.getProtocol();
        if (selectedProtocol == FileURLValue.DEFAULT_PROTOCOL) {
          text = value.getPath();
        }
        else {
          text = selectedProtocol.getDisplayName() + value.getPath();
        }
      }
      textField.setText(text);
    }
  }

  /**
   * Viso do parmetro em modo
   * {@link csbase.client.algorithms.parameters.ParameterView.Mode#CONFIGURATION}
   * .
   */
  private class FileConfigurationParameter extends JPanel implements
  IFileParameterComponent {

    /**
     * O campo de texto.
     */
    private JTextField textField;

    /**
     * O campo de seleo de protocolos.
     */
    private JComboBox protocolCombo;

    /**
     * O campo de seleo de SGAs. (usado no caso do protocolo ser
     * {@link URLProtocol#SGA})
     */
    private JComboBox sgaCombo;

    /**
     * O boto Navegar.
     */
    private JButton browseButton;

    /**
     * Construtor.
     */
    FileConfigurationParameter() {
      super(new GridBagLayout());
      URLParameter parameter = getParameter();
      EnumSet<URLProtocol> allowedProtocols = parameter.getAllowedProtocols();
      protocolCombo = new JComboBox();
      for (URLProtocol protocol : allowedProtocols) {
        protocolCombo.addItem(protocol);
      }
      protocolCombo.addItemListener(new ItemListener() {
        @Override
        public void itemStateChanged(ItemEvent e) {
          if (e.getStateChange() == ItemEvent.SELECTED) {
            URLProtocol item = (URLProtocol) e.getItem();
            protocolChanged(item);
          }
        }
      });
      if (allowedProtocols.size() == 1) {
        protocolCombo.setVisible(false);
      }

      browseButton =
        GUIUtils.createImageButton(ApplicationImages.ICON_BROWSEFILE_16);
      ActionListener browseAction = new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent ev) {
          URLProtocol protocol = (URLProtocol) protocolCombo.getSelectedItem();
          FileURLValue file = askForFile(protocol, getSGASelected());
          if (file != null) {
            getParameter().setValue(file);
          }
        }
      };
      browseButton.addActionListener(browseAction);
      browseButton.setToolTipText(getParameter().getDescription());
      textField = new JTextField();
      textField.addFocusListener(new FocusListener() {
        @Override
        public void focusGained(FocusEvent e) {
          // Ignora o evento.
        }

        @Override
        public void focusLost(FocusEvent e) {
          updateModel();
        }
      });
      textField.setToolTipText(getParameter().getDescription());

      sgaCombo = new JComboBox();
      updateSGAModel();
      add(protocolCombo, new GBC(0, 0));
      add(sgaCombo, new GBC(1, 0));
      add(textField, new GBC(2, 0).both().left(2).right(2));
      add(browseButton, new GBC(3, 0));

      protocolChanged(parameter.getDefaultProtocol());
    }

    /**
     * Atualiza o modelo da combo de SGAs.
     */
    private void updateSGAModel() {
      if (sgaCombo != null && sgaCombo.isVisible()) {
        List<String> sgaList = getSGAList();
        Object[] sgas = sgaList.toArray();
        if (sgaList.isEmpty()) {
          sgaCombo.setEnabled(false);
          String noSGAMessage =
            LNG.get(URLParameterView.class.getSimpleName() + ".no_sga");
          sgaCombo.setToolTipText(noSGAMessage);
        }
        else {
          sgaCombo.setEnabled(true);
          sgaCombo.setModel(new DefaultComboBoxModel(sgas));
        }
        browseButton.setEnabled(sgaCombo.isEnabled());
      }
    }

    /**
     * Indica que um novo protocolo foi selecionado.
     *
     * @param protocol o novo protocolo.
     */
    private void protocolChanged(URLProtocol protocol) {
      textField.setText(null);
      switch (protocol) {
        case LOCAL:
          sgaCombo.setVisible(false);
          browseButton.setIcon(ApplicationImages.ICON_BROWSELOCALFILE_16);
          browseButton.setVisible(true);
          browseButton.setEnabled(true);
          textField.setEditable(true);
          break;
        case PROJECT:
          sgaCombo.setVisible(false);
          browseButton.setIcon(ApplicationImages.ICON_BROWSEFILE_16);
          browseButton.setVisible(true);
          browseButton.setEnabled(true);
          textField.setEditable(true);
          break;
        case SGA:
          sgaCombo.setVisible(true);
          browseButton.setIcon(ApplicationImages.ICON_BROWSESGAFILE_16);
          browseButton.setVisible(true);
          browseButton.setEnabled(sgaCombo.isEnabled());
          textField.setEditable(sgaCombo.isEnabled());
          break;
        default:
          sgaCombo.setVisible(false);
          browseButton.setVisible(false);
          browseButton.setEnabled(false);
          textField.setEditable(true);
      }
      updateSGAModel();
      updateModel();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      FileURLValue file = getParameter().getValue();
      if (file == null) {
        textField.setText("");
      }
      else {
        if (protocolCombo != null) {
          URLProtocol protocol = file.getProtocol();
          protocolCombo.setSelectedItem(protocol);
        }
        if (sgaCombo != null) {
          updateSGAModel();
          String host = file.getHost();
          if (host != null) {
            sgaCombo.setSelectedItem(host);
          }
        }
        textField.setText(file.toString());
      }
      updateModel();
    }

    /**
     * Habilita/Desabilita a viso.
     *
     * @param isEnabled .
     */
    @Override
    public void setEnabled(boolean isEnabled) {
      super.setEnabled(isEnabled);
      textField.setEnabled(isEnabled);
      browseButton.setEnabled(isEnabled);
      if (protocolCombo != null) {
        protocolCombo.setEnabled(isEnabled);
      }
      if (sgaCombo != null) {
        sgaCombo.setEnabled(isEnabled);
      }
    }

    /**
     * Retorna o nome do SGA selecionado na combo.
     *
     * @return nome do SGA.
     */
    private String getSGASelected() {
      return (String) sgaCombo.getSelectedItem();
    }

    /**
     * Atualiza o contedo do modelo (L as informaes da viso e preenche o
     * parmetro).
     */
    private void updateModel() {
      String text = textField.getText();
      if (text.length() == 0) {
        getParameter().setValue(null);
      }
      else {
        if (text.equals("/")) {
          text = ".";
        }
        ClientFile file = null;
        String host = null;
        String type;
        URLProtocol protocol = (URLProtocol) protocolCombo.getSelectedItem();
        if (protocol == URLProtocol.PROJECT) {
          String[] path = FileUtils.splitPath(text, "/");
          CommonClientProject project = DesktopFrame.getInstance().getProject();
          ClientProjectFile rootProjectFile = project.getRoot();
          ClientProjectFile clientProjectFile = rootProjectFile;
          for (String component : path) {
            if (clientProjectFile == null) {
              break;
            }
            clientProjectFile =
              GetChildFromNameTask.runTask(clientProjectFile, component);
          }
          file = clientProjectFile;
        }
        else if (protocol == URLProtocol.LOCAL) {
          file = new ClientLocalFile(new File(text));
        }
        else if (protocol == URLProtocol.SGA) {
          host = getSGASelected();
        }
        if (file != null) {
          type = file.getType();
        }
        else {
          if (getParameter().getMode() == FileParameterMode.DIRECTORY) {
            type = ProjectFileType.DIRECTORY_TYPE;
          }
          else {
            type = ProjectFileType.UNKNOWN;
          }
        }
        FileURLValue value = new FileURLValue(text, type, protocol, host);
        getParameter().setValue(value);
      }
    }

    /**
     * Lista com todos os nomes de SGA cadastrados no servidor.
     *
     * @return lista com todos os nomes de SGA cadastrados no servidor.
     */
    private List<String> getSGAList() {
      Collection<String> sgaNames = SGAProxy.getAllSGANames();
      if (sgaNames == null) {
        return new ArrayList<String>();
      }

      List<String> sgaList = new ArrayList<String>();
      for (String name : sgaNames) {
        SGASet sgaSet = SGAProxy.getSGASet(name);
        if (sgaSet.mayExecuteCommand()) {
          sgaList.add(name);
        }
      }

      return sgaList;
    }
  }
}
