package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.StatusModificationTask;

/**
 * Ao que abre dilogo para alterao no status de alocao de projetos
 * selecionados.
 * 
 * @author jnlopes
 * 
 */
public class ChangeStatusNoAllocationAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public ChangeStatusNoAllocationAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final int res =
      JOptionPane.showConfirmDialog(projectsManager.getApplicationFrame(),
        getString("ChangeStatusNoAllocationAction.warning.message"),
        getString("ChangeStatusNoAllocationAction.warning.title"),
        JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);
    if (res != JOptionPane.OK_OPTION) {
      return;
    }

    boolean ret = runTask(projects);
    if (ret) {
      updateStatusOfLocalProjects(projects);
      projectsManager.refreshProjectsTable();
    }
    else {
      String errMsg =
        getString("ChangeStatusNoAllocationAction.error.allocation.message");
      String errTitle =
        getString("ChangeStatusNoAllocationAction.error.allocation.title");
      StandardDialogs.showErrorDialog(projectsManager.getApplicationFrame(),
        errTitle, errMsg);
    }
  }

  /**
   * Define o status de alocao dos projetos validados como
   * <code>ProjectSpaceAllocation.NO_ALLOCATION</code> e remove dados de espao
   * de alocao.
   * 
   * @param projects Projetos validados.
   */
  private void updateStatusOfLocalProjects(List<ProjectsManagerData> projects) {
    for (ProjectsManagerData pmd : projects) {
      pmd.setAllocatedSpace(0);
      pmd.setRequestedSpace(0);
      pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.NO_ALLOCATION);
    }
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @return Boolean indicando se a operao foi bem sucedida.
   */
  private boolean runTask(final List<ProjectsManagerData> projects) {
    ProjectsManager projectsManager = getProjectsManager();
    ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    StatusModificationTask smt =
      new StatusModificationTask(projectsManager, projects);
    String taskTitle =
      getString("ChangeStatusNoAllocationAction.running.title");
    String taskMsg =
      getString("ChangeStatusNoAllocationAction.running.message");
    smt.execute(appFrame, taskTitle, taskMsg);
    if (smt.wasCancelled()) {
      String err =
        getString("ChangeStatusNoAllocationAction.cancelled.message");
      smt.showError(err);
      return false;
    }
    if (smt.getStatus() != true) {
      Exception exception = smt.getError();
      String err =
        getString("ChangeStatusNoAllocationAction.allocation.status.error.message");
      String msg = err + "\n" + exception.getMessage();
      StandardDialogs.showErrorDialog(appFrame, msg, "");
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    List<ProjectsManagerData> prjList = new ArrayList<ProjectsManagerData>();

    prjList.addAll(projectList);
    List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    int i = 0;
    while (i < prjList.size()) {
      ProjectsManagerData pmd = prjList.get(i);
      ProjectSpaceAllocation spaceAllocation = pmd.getProjectSpaceAllocation();
      if (spaceAllocation != ProjectSpaceAllocation.ALLOCATED) {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
      else {
        i++;
      }
    }

    ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      String deniedMessage =
        getString("ChangeStatusNoAllocationAction.project.selection.denied.message")
          + getString("ChangeStatusNoAllocationAction.project.requirements.message");
      StatusDialog
        .showNoneOkDialog(
          projectsManager,
          deniedMessage,
          failedProjects,
          getString("ChangeStatusNoAllocationAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      String someDeniedMessage =
        getString("ChangeStatusNoAllocationAction.project.selection.some.denied.message")
          + getString("ChangeStatusNoAllocationAction.project.requirements.message");
      int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("ChangeStatusNoAllocationAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return null;
  }
}
