package csbase.client.applications.sgamonitor.columns;

import java.awt.Component;
import java.util.Comparator;

import javax.swing.ImageIcon;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;

import csbase.client.applications.ApplicationImages;
import csbase.client.facilities.configurabletable.column.AbstractConfigurableColumn;
import csbase.client.facilities.configurabletable.stringprovider.IStringProvider;
import csbase.logic.SGAInfo;

/**
 * Coluna que exibe uma imagem que representa a alocao de um n de cluster. <br>
 * - Verde - N livre <br>
 * - Vermelho - N ocupado <br>
 * - Amarelo - N parcialmente ocupado <br>
 * - Cinza - N desligado
 * 
 * A ordenao dessa coluna respeita a seguinte poltica: <br>
 * - {@code desligado < ocupado < parcialmente ocupado < livre}
 * 
 * @author Tecgraf
 */
public class NodeAllocationColumn extends AbstractConfigurableColumn<SGAInfo> {

  /**
   * Construtor padro.
   * 
   * @param id - identificador da coluna.
   * @param visible - visibilidade da coluna.
   * @param stringProvider - utilizado para obter o nome da coluna
   *        internacionalizada.
   * @param align - alinhamento da coluna.
   */

  public NodeAllocationColumn(String id, Boolean visible,
    IStringProvider stringProvider, Integer align) {
    super(id, visible, stringProvider, align);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass() {
    return Allocation.class;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName() {
    return getString(getId());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValue(SGAInfo obj) {
    return Allocation.getAllocation(obj);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Comparator<Allocation> getComparator() {

    return new Comparator<Allocation>() {
      @Override
      public int compare(Allocation first, Allocation second) {
        // off < busy < partially < free
        return first.compareTo(second);
      }
    };

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellRenderer createTableCellRenderer() {
    return new DefaultTableCellRenderer() {
      @Override
      public Component getTableCellRendererComponent(JTable table,
        Object value, boolean isSelected, boolean hasFocus, int row, int column) {
        super.getTableCellRendererComponent(table, value, isSelected, hasFocus,
          row, column);

        Allocation alloc = (Allocation) value;
        setText(null);
        setIcon(alloc.getImage());
        setToolTipText(getString("tooltip." + alloc.toString().toLowerCase()));

        return this;
      }
    };
  }

  /**
   * Enumerao que define o estado da alocao de um SGA.
   * 
   * @author Tecgraf
   */
  private enum Allocation {

    /**
     * SGA livre.
     */
    FREE(ApplicationImages.GREEN_BALL_16),

    /**
     * SGA parcialmente ocupado.
     */
    PARTIALLY_BUSY(ApplicationImages.YELLOW_BALL_16),

    /**
     * SGA ocupado.
     */
    BUSY(ApplicationImages.RED_BALL_16),

    /**
     * SGA desligado.
     */
    OFF(ApplicationImages.GRAY_BALL_16);

    /**
     * Imagem associada a cada estado de alocao.
     */
    private ImageIcon image;

    /**
     * Construtor padro.
     * 
     * @param image - imagem associada a cada estado de alocao.
     */
    Allocation(ImageIcon image) {
      this.image = image;
    }

    /**
     * Imagem associada a cada estado de alocao.
     * 
     * @return - imagem associada a cada estado de alocao.
     */
    public ImageIcon getImage() {
      return image;
    }

    /**
     * Mtodo esttico que dado um n de cluster, retorna a sua alocao.
     * 
     * @param node - n do cluster.
     * @return alocao do n do cluster.
     */
    public static Allocation getAllocation(SGAInfo node) {
      if (node.getAlive()) {

        Allocation alloc = PARTIALLY_BUSY;

        if (node.getNumberOfJobs() == 0) {
          alloc = FREE;
        }
        else if (node.getNumberOfJobs() == node.getNumProcessors()) {
          alloc = BUSY;
        }

        return alloc;
      }

      return OFF;
    }
  }
}
