/*
 * $Id: OpenBusApplicationInstanceData.java 176168 2016-09-22 21:12:51Z fpina $
 */

package csbase.client.openbus;

import java.util.Vector;

import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;

/**
 * Estrutura utilizada para identificar uma instncia de aplicao.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class OpenBusApplicationInstanceData {

  /**
   * Id do tipo da aplicao;
   */
  final private String appId;

  /**
   * Instncia da aplicao;
   */
  final private ApplicationType instance;

  /**
   * Consulta o atributo (ver documentao de {@link #appId}).
   * 
   * @return o atributo.
   */
  public final String getApplicationId() {
    return appId;
  }

  /**
   * Consulta o atributo (ver documentao de {@link #instance}).
   * 
   * @return o atributo.
   */
  public final ApplicationType getApplicationInstance() {
    return instance;
  }

  /**
   * Montagem de um valor textual para codificar uma instncia de aplicao
   * dentro de um desktop.
   * 
   * @return o valor (textual conforme sintaxe {@code
   *         <appId>:<appInstanceId>}.
   */
  final public String getEventValue() {
    return appId + ":" + instance.getInstanceIndex();
  }

  /**
   * Consulta o id da aplicao com base no valor textual de um evento.
   * 
   * @param stringValue valor textual
   * @return o id (ou {@code null}, se invlido).
   */
  public static String getApplicationIdFromEventValue(final String stringValue) {
    final String[] idsFromValue =
      OpenBusApplicationInstanceData.getIdsFromEventValue(stringValue);
    if (idsFromValue == null) {
      return null;
    }
    final String apId = idsFromValue[0];
    return apId;
  }

  /**
   * Monta um dado de instncia de aplicao com base no valor textual de um
   * evento.
   * 
   * @param stringValue valor textual
   * @return o dado (ou {@code null}, se invlido).
   */
  public static OpenBusApplicationInstanceData getDataFromEventValue(
    final String stringValue) {
    final String[] idsFromValue =
      OpenBusApplicationInstanceData.getIdsFromEventValue(stringValue);
    if (idsFromValue == null) {
      return null;
    }
    final String apId = idsFromValue[0];
    final String inId = idsFromValue[0];

    final int iId;
    try {
      iId = Integer.parseInt(inId);
    }
    catch (final NumberFormatException e) {
      return null;
    }

    final ApplicationManager appManager = ApplicationManager.getInstance();
    final Vector<ApplicationType> runningApplications =
      appManager.getRunningApplications(apId);
    if (runningApplications == null) {
      return null;
    }

    for (final ApplicationType app : runningApplications) {
      final int iIdRun = app.getInstanceIndex();
      if (iIdRun == iId) {
        final OpenBusApplicationInstanceData data =
          new OpenBusApplicationInstanceData(apId, app);
        return data;
      }
    }
    return null;
  }

  /**
   * Monta um array de strings de instncia de aplicao com base no valor
   * textual de um evento.
   * 
   * @param stringValue valor textual
   * @return o dado em formato array (id da aplicao e id da instncia) (ou
   *         {@code null}, se invlido).
   */
  public static String[] getIdsFromEventValue(final String stringValue) {
    if (stringValue == null) {
      final String err = "null value not allowed!";
      throw new IllegalArgumentException(err);
    }
    final String[] splitted = stringValue.split(":");
    if (splitted == null || splitted.length != 2) {
      return null;
    }
    final String arg0 = splitted[0];
    final String arg1 = splitted[1];
    if (arg0 == null || arg1 == null) {
      return null;
    }
    final String apId = arg0.trim();
    final String inId = arg1.trim();
    if (apId.isEmpty() || inId.isEmpty()) {
      return null;
    }

    return new String[] { apId, inId };
  }

  /**
   * @param appId identificador da aplicao.
   * @param instance instncia da aplicao.
   */
  public OpenBusApplicationInstanceData(final String appId,
    final ApplicationType instance) {
    if (appId == null || instance == null) {
      final String err = "null id/instance not allowed!";
      throw new IllegalArgumentException(err);
    }
    this.appId = appId;
    this.instance = instance;
  }
}
