/*
 * Created on Sep 27, 2004
 */
package csbase.logic;

import java.io.Serializable;
import java.util.Arrays;

/**
 * Classe que representa um n de uma rvore de arquivos e diretrios.
 *
 * @author Leonardo Barros
 */
public class FileNodeInfo implements Comparable, Serializable {
  private String name;
  private long size;
  private long lastUpdateDate;
  private FileNodeInfo parent;
  private FileNodeInfo[] children;
  private boolean isDirectory;
  private String type;
  private String renderingName;

  /**
   * Obtm os ns-filhos deste n.
   *
   * @return ns-filhos.
   */
  public FileNodeInfo[] getChildren() {
    return children;
  }

  /**
   * Substitui os ns-filhos deste n.
   *
   * @param children novos ns-filhos.
   */
  public void setChildren(FileNodeInfo[] children) {
    if (children != null) {
      Arrays.sort(children);
    }
    this.children = children;
  }

  /**
   * Obtm o n-filho localizado no ndice especificado.
   *
   * @param index ndice do n-filho a obter.
   *
   * @return n-filho ou null caso este no seja encontrado.
   */
  public Object getChild(int index) {
    if (children == null) {
      throw new IllegalStateException("children == null");
    }    
    return children[index];
  }

  /**
   * Obtm o n-filho com o nome especificado.
   *
   * @param name nome do n-filho a obter.
   *
   * @return n-filho ou null caso este no seja encontrado.
   */
  public Object getChild(String name) {
    if (name == null || name.equals("")) {
      return null;
    }
    if (children == null) {
      return null;
    }
    for (int i = 0; i < children.length; i++) {
      if (name.equals(children[i].name)) {
        return children[i];
      }
    }
    return null;
  }

  /**
   * Obtm referncia para o n (seja o prprio n ou um descendente deste) 
   * indicado pelo caminho passado. O caminho deve ser formado pelo nome deste 
   * n seguido (quando for o caso) do nome de seus descendentes, at o 
   * descendente desejado.
   *
   * @param path caminho para o n.
   *
   * @return n.
   */
  public FileNodeInfo getNode(String[] path) {
    // #TODO Esse mtodo  estranho; rever.
    if (path == null) {
      return null;
    }
    if (path.length == 0) {
      return this;
    }
    FileNodeInfo node = this;
    for (int i = 0; i < path.length; i++) {
      node = (FileNodeInfo)node.getChild(path[i]);
      if (node == null) {
        return null;
      }
    }
    return node;
  }

  /**
   * Obtm o nmero de ns-filhos deste n.
   *
   * @return nmero de ns-filhos.
   */
  public int getChildCount() {
    if (children == null) {
      return 0;
    }
    return children.length;
  }

  /**
   * Obtm o ndice do n-filho passado.
   *
   * @param child n-filho cujo ndice deseja-se obter.
   *
   * @return ndice do n-filho ou -1 caso este no seja encontrado.
   */
  public int getIndexOfChild(Object child) {
    if (children == null) {
      throw new IllegalStateException("children == null");
    }
    for (int i = 0; i < children.length; i++) {
      if (child.equals(children[i])) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Indica se o arquivo  um diretrio.
   *
   * @return true se o n for um diretrio, false se for um arquivo.
   */
  public boolean isDirectory() {
    return isDirectory;
  }

  /**
   * Substitui o indicador interno que informa se este n  um diretrio ou
   * arquivo.
   *
   * @param isDirectory true se este n for um diretrio, false se for um
   *        arquivo.
   */
  public void setDirectory(boolean isDirectory) {
    this.isDirectory = isDirectory;
  }

  /**
   * Obtm o nome do arquivo / diretrio.
   *
   * @return nome.
   */
  public String getName() {
    return name;
  }

  /**
   * Substitui o nome do arquivo / diretrio.
   *
   * @param name novo nome.
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * Obtm o tamanho do arquivo. Se for um diretrio, essa informao no 
   * significativa.
   *
   * @return tamanho do arquivo.
   */
  public long getSize() {
    return size;
  }

  /**
   * Substitui o tamanho do arquivo.
   *
   * @param size novo tamanho.
   */
  public void setSize(long size) {
    this.size = size;
  }

  /**
   * Obtm a data da ltima modificao do arquivo/diretrio.
   *
   * @return data.
   */
  public long getLastUpdateDate() {
    return lastUpdateDate;
  }

  /**
   * Substitui a data da ltima modificao do arquivo/diretrio.
   *
   * @param lastUpdateDate nova data.
   */
  public void setLastUpdateDate(long lastUpdateDate) {
    this.lastUpdateDate = lastUpdateDate;
  }

  /**
   * Obtm referncia para o n-pai.
   *
   * @return n-pai.
   */
  public FileNodeInfo getParent() {
    return parent;
  }

  /**
   * Substitui a referncia para o n-pai.
   *
   * @param parent novo n-pai.
   */
  public void setParent(FileNodeInfo parent) {
    this.parent = parent;
  }

  /**
   * Obtm o tipo do arquivo.
   *
   * @return tipo.
   */
  public String getType() {
    return type;
  }

  /**
   * Substitui o tipo do arquivo.
   *
   * @param type novo tipo.
   */
  public void setType(String type) {
    this.type = type;
  }

  /**
   * Obtm o caminho relativo para o n desde a raiz da rvore.
   *
   * @return caminho relativo para o n.
   */
  public String[] getPath() {
    if (parent == null) {
      return new String[] {};
    }
    String[] parentPath = parent.getPath();
    int parentCount = parentPath.length;
    String[] path = new String[parentCount+1];
    System.arraycopy(parentPath, 0, path, 0, parentCount);
    path[parentCount] = name;
    return path;    
  }

  /**
   * Informa se os ns-filhos deste n j foram recuperados do servidor (apenas
   * a raiz  criada com seus respectivos filhos - os demais so obtidos sob
   * demanda).
   *
   * @return true caso os ns-filhos deste n j tenham sido recuperados do
   *         servidor, false caso contrrio.
   */
  public boolean childrenWereRetrieved() {
    // Se os ns j tiverem sido recuperados, children estar ou preenchida,
    // ou vazia. Se ela estiver nula,  porque no foi feita nenhuma consulta
    // ao servidor ainda.
    return children != null;
  }

  /**
   * Obtm uma representao textual do n, incluindo apenas o seu nome.
   *
   * @return nome.
   */
  public String toString() {
    return getRenderingName();
  }

  /**
   * Descreve como comparar este n com outros objetos, para efeito de
   * ordenao. Se o outro objeto no for do mesmo tipo, ser posto no fim da
   * lista; caso contrrio, os ns sero ordenados pelo seu nome, em ordem
   * alfabtica.
   *
   * @param o objeto a ser comparado com este.
   *
   * @return -1 se este objeto for "menor" que o outro; 0 se so iguais; 1 se 
   * for "maior".
   */
  public int compareTo(Object o) {
    if (!(o instanceof FileNodeInfo)) {
      return -1;
    }
    return this.name.compareTo(((FileNodeInfo)o).name);
  }

  /**
   * Converte um caminho de nomes de ns no formato de array de Strings para um
   * array <code>FileNodeInfo</code> contendo os ns identificados pelos nomes
   * do array passado como parmetro. O array s ser retornado se todos os
   * nomes forem encontrados dentro dos descendentes deste n. Caso contrrio,
   * ser retornado null.
   *
   * @param strPath caminho de nomes de ns (array de Strings) ou 
   * <code>null</code> se o caminho no puder ser convertido.
   *
   * @return caminho de ns (array de <code>FileNodeInfo</code>).
   */
  public FileNodeInfo[] stringPathToNodePath(String[] strPath) {
    // #TODO Esse mtodo s deveria ser permitido para o n raiz. Rever.
    if (strPath == null) {
      return null;
    }
    if (strPath.length == 0) {
      return new FileNodeInfo[] {this};
    }
    FileNodeInfo[] nodePath = new FileNodeInfo[strPath.length+1];
    FileNodeInfo child = this;
    nodePath[0] = child;
    for (int i = 0; i < strPath.length; i++) {
      child = (FileNodeInfo)child.getChild(strPath[i]);
      if (child == null) {
        return null;
      }
      nodePath[i+1] = child;
    }
    return nodePath;
  }

  /**
   * Obtm a profundidade do n a partir da raiz da rvore.
   * 
   * @return 0 se for o n-raiz, 1 se for o filho imediato da raiz, e assim
   *         por diante.
   */
  public int getDepth() {
    if (parent == null) {
      return 0;
    }
    return parent.getDepth() + 1;
  }

  /**
   * @param renderingName
   */
  public void setRenderingName(String renderingName) {
    this.renderingName = renderingName;
  }
  /**
   * @return Returns the renderingName.
   */
  public String getRenderingName() {
    if (renderingName == null) {
      return name;
    }
    return renderingName;
  }
  

  /**
   * Construtor para um diretrio.
   *
   * @param name nome do diretrio.
   * @param isDirectory indica se  diretrio (true) ou arquivo (false).
   * @param type tipo do arquivo (para diretrio  irrelevante).
   * 
   */
  public FileNodeInfo(String name, boolean isDirectory, String type) {
    this(name, 0, 0, null, null, isDirectory, type);
  }

  /**
   * Construtor para um arquivo / diretrio.
   *
   * @param name nome do arquivo / diretrio.
   */
  public FileNodeInfo(String name) {
    this(name, 0, 0, null, null, true, null);
  }  
  
  /**
   * Construtor simples.
   *
   * @param name nome do arquivo / diretrio.
   * @param size tamanho do arquivo (para diretrio  irrelevante).
   * @param lastUpdateDate data da ltima modificao do arquivo / diretrio.
   * @param parent n-pai deste arquivo / diretrio.
   * @param children ns-filhos deste diretrio (para arquivos deve ser nulo).
   * @param isDirectory indica se  diretrio (true) ou arquivo (false).
   * @param type tipo do arquivo (para diretrio  irrelevante).
   */
  public FileNodeInfo(
    String name, long size, long lastUpdateDate, FileNodeInfo parent,
    FileNodeInfo[] children, boolean isDirectory, String type) {
    this.name = name;
    this.size = size;
    this.lastUpdateDate = lastUpdateDate;
    this.parent = parent;
    if (children != null) {
      Arrays.sort(children);
    }
    this.children = children;
    this.isDirectory = isDirectory;
    this.type = type;
  }
  
}
