/*
 * $Id$
 */

package csbase.logic.algorithms.commands;

import csbase.logic.Notification;

/**
 * Evento que indica que algo ocorreu no servio de persistncia de comando.
 */
public class CommandPersistenceNotification extends Notification {

  /**
   * Indica o tipo do evento.
   * 
   * @author Tecgraf / PUC-Rio
   */
  public enum Type {
    /**
     * O comando foi removido.
     */
    REMOVED,
    /**
     * O comando foi salvo.
     */
    SAVED,
    /**
     * O comando foi atualizado.
     */
    UPDATED
  }

  /**
   * O identificador do comando.
   */
  private final String commandId;

  /**
   * O identificador do projeto.
   */
  private final Object projectId;

  /**
   * Tipo do evento.
   */
  private final Type type;

  /**
   * Obtm o identificador do comando.
   * 
   * @return O identificador do comando.
   */
  public String getCommandId() {
    return commandId;
  }

  /**
   * Obtm o identificador do projeto.
   * 
   * @return O identificador do projeto.
   */
  public Object getProjectId() {
    return projectId;
  }

  /**
   * Obtm o identificador do tipo de evento.
   * 
   * @return O identificador do tipo de evento.
   */
  public Type getType() {
    return type;
  }

  /**
   * Definio da volatilidade da notificao.
   * 
   * @return Flag indicativo da volatilidade da notificao.
   */
  @Override
  public boolean isVolatile() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return CommandPersistenceNotification.class.getName()
      + "{projectId:" + projectId + ", commandId:" + commandId + ", type:"
      + type + "}";
  }

  /**
   * Cria o evento.
   * 
   * @param sender Quem est enviando a notificao (no pode ser {@code null}).
   * @param projectId O identificador do projeto (no pode ser {@code null}).
   * @param commandId O identificador do comando (no pode ser {@code null}).
   * @param type o tipo
   */
  public CommandPersistenceNotification(String sender,
    Object projectId, String commandId, Type type) {
    super(sender, false, false);
    if (commandId == null) {
      throw new IllegalArgumentException("projectId == null");
    }
    if (projectId == null) {
      throw new IllegalArgumentException("commandId == null");
    }
    if (type == null) {
      throw new IllegalArgumentException("type == null");
    }
    this.commandId = commandId;
    this.projectId = projectId;
    this.type = type;
  }
}
