/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.util.Arrays;
import java.util.List;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.openbus.algorithmservice.v1_0.parameters.OutputFileParameterHelper;
import csbase.logic.ProjectFileType;

/**
 * A classe <code>OutputFileParameter</code> representa um parmetro que  um
 * arquivo de sada.
 *
 * @author lmoreira
 */
public final class OutputFileParameter extends FileParameter {
  /** Texto que representa o tipo desse parmetro */
  public static final String TYPE_VALUE = "OUTPUT_FILE";

  /**
   * O nome do parmetro arquivo de log.
   */
  public static final String LOG_FILE_PARAMETER_NAME = "LOG_FILE";

  /** Indica se o arquivo deve ter uma das extenses pr-definidas. */
  private boolean mustForceExtension;

  /**
   * Cria um parmetro do tipo Arquivo de Sada.
   *
   * @param name O nome do parmetro (No aceita {@code null}).
   * @param label O rtulo do parmetro (No aceita {@code null}).
   * @param description A descrio do parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param fileType O tipo de arquivo que este parmetro filtra. Se ele se
   *        importa com o tipo do arquivo, ele valer {@code null}.
   * @param mode O modo de funcionamento atual (apenas arquivos, apenas
   *        diretrios ou ambos).
   * @param usesPipe Indica se este parmetro pode aceitar pipe
   *        {@value FileParameterPipeAcceptance#TRUE}, no aceita pipe
   *        {@value FileParameterPipeAcceptance#FALSE} ou *s* aceita pipe
   *        {@value FileParameterPipeAcceptance#ALWAYS}..
   * @param mustForceExtension Indica se o parmetro deve forar o uso da
   *        extenso no arquivo (Se ele no tiver, ele colocar a extenso).
   * @param usesFilter Indica se um filtro para filtro deve ser exibido.
   */
  public OutputFileParameter(String name, String label, String description,
    FileURLValue defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, String fileType, FileParameterMode mode,
    FileParameterPipeAcceptance usesPipe, boolean mustForceExtension,
    boolean usesFilter) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern, fileType, mode, usesPipe, usesFilter, false);
    if ((!mode.equals(FileParameterMode.DIRECTORY))
      && (!mode.equals(FileParameterMode.REGULAR_FILE))) {
      String errorMessage =
        String.format(
          "Modo no suportado.\nModo fornecido: %s.\nModos suportados: %s.\n",
          mode, Arrays.asList(FileParameterMode.DIRECTORY,
            FileParameterMode.REGULAR_FILE));
      throw new IllegalArgumentException(errorMessage);
    }
    this.mustForceExtension = mustForceExtension;
  }

  /**
   * Cria um parmetro que  um arquivo de log.
   *
   * @param commandLinePattern o padro da linha de comando.
   *
   * @return um parmetro que  um arquivo de log.
   */
  public static OutputFileParameter createLogFile(String commandLinePattern) {
    return new OutputFileParameter(LOG_FILE_PARAMETER_NAME, "Arquivo de Log",
      "Arquivo de Log", null, true, true, commandLinePattern, "LOG",
      FileParameterMode.REGULAR_FILE, FileParameterPipeAcceptance.FALSE, false,
      false);
  }

  /**
   * (non-Javadoc)
   *
   * @see csbase.logic.algorithms.parameters.SimpleParameter#getType()
   */
  @Override
  public String getType() {
    return TYPE_VALUE;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getIDLType() {
    return OutputFileParameterHelper.id();
  }

  /**
   * Indica se deve forar a extenso do arquivo.
   *
   * @return .
   */
  public boolean mustForceExtension() {
    return this.mustForceExtension;
  }

  /**
   * Indica se este parmetro  um parmetro do tipo arquivo de log.
   *
   * @return .
   */
  public boolean isLogFile() {
    return getName().equals(LOG_FILE_PARAMETER_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setValue(FileURLValue file) {
    checkExtension(file);
    return super.setValue(file);
  }

  /**
   * Verifica se deve forar a extenso do arquivo e a coloca caso necessrio.
   *
   * @param file arquivo de sada
   */
  private void checkExtension(FileURLValue file) {
    if ((file != null) && mustForceExtension()) {
      String filePath = file.getPath();
      String fileExtension = FileUtils.getFileExtension(filePath);
      /* Obtm a lista de possveis extenses para o arquivo de sada */
      String type = this.getFileType();
      ProjectFileType projectFileType = ProjectFileType.getFileType(type);
      if (projectFileType == null) {
        return;
      }
      List<String> extensions = projectFileType.getExtensions();
      if (extensions.isEmpty()) {
        return;
      }
      /*
       * Se arquivo no tem extenso ou a extenso no confere com o tipo do
       * arquivo, fora a extenso
       */
      if ((fileExtension == null) || !extensions.contains(fileExtension)) {
        filePath += '.' + extensions.get(0);
        /* Renomeia o arquivo. */
        file.setPath(filePath);
      }
    }
  }

  /**
   *
   * {@inheritDoc}
   */
  @Override
  public boolean isOuput() {
    return true;
  }
}
