package csbase.logic.algorithms.parsers;

import csbase.exception.BugException;
import csbase.exception.ParseException;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.InputFileListParameter;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;

/**
 * Classe abstrata que serve de base para analisadores de parmetros <T> com
 * valores que representam arquivos como {@link InputFileParameter} e
 * {@link OutputFileParameter}.
 * 
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class AbstractFileParameterParser<T extends SimpleParameter<?>>
  extends SimpleParameterParser<T> {

  /**
   * <p>
   * O atributo {@value #FILE_PARAMETER_ELEMENT_MUST_SORT_ATTRIBUTE} do elemento
   * {@link InputFileListParameter}.
   * </p>
   * 
   * Indica se o {@link InputFileParameter arquivo de entrada} se a relao de
   * arquivos ser ordenada,  opcional, o seu valor-padro 
   * {@link #FILE_PARAMETER_ELEMENT_MUST_SORT_DEFAULT_VALUE} e  do tipo
   * booleano.
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_MUST_SORT_ATTRIBUTE =
    "ordenar";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #FILE_PARAMETER_ELEMENT_MUST_SORT_ATTRIBUTE} do elemento
   * {@link InputFileListParameter}
   * </p>
   * <p>
   * O seu valor  {@value #FILE_PARAMETER_ELEMENT_MUST_SORT_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean FILE_PARAMETER_ELEMENT_MUST_SORT_DEFAULT_VALUE =
    true;

  /**
   * <p>
   * O atributo {@value #FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o se o parmetro do tipo arquivo aceita trabalhar com pipes, 
   * opcional, o seu valor-padro 
   * {@link #FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_DEFAULT_VALUE} e  do tipo
   * string e os valores vlidos para ele so:
   * <ul>
   * <li>{@link FileParameterPipeAcceptance#TRUE}</li>
   * <li>{@link FileParameterPipeAcceptance#FALSE}</li>
   * <li>{@link FileParameterPipeAcceptance#ALWAYS}</li>
   * </ul>
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE =
    "permitir_pipe";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@value #FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_DEFAULT_VALUE}.
   * </p>
   */
  protected static final FileParameterPipeAcceptance FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_DEFAULT_VALUE =
    FileParameterPipeAcceptance.TRUE;

  /**
   * <p>
   * O atributo {@value #FILE_PARAMETER_ELEMENT_CATEGORY_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o {@link FileParameterMode modo de funcionamento} do parmetros do
   * tipo arquivo},  opcional, o seu valor-padro 
   * {@link #FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE},  do tipo
   * string e o valores vlidos para ele so:
   * <ul>
   * <li>{@link #FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE}</li>
   * <li>{@link #FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY}</li>
   * <li>{@link #FILE_PARAMETER_ELEMENT_CATEGORY_BOTH}</li>
   * </ul>
   * .
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_CATEGORY_ATTRIBUTE =
    "categoria";

  /**
   * <p>
   * O valor que indica o {@link FileParameterMode#DIRECTORY modo apenas
   * diretrio} para o atributo
   * {@link #FILE_PARAMETER_ELEMENT_CATEGORY_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@value #FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY}.
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY =
    "diretorio";

  /**
   * <p>
   * O valor que indica o {@link FileParameterMode#DIRECTORY_AND_REGULAR_FILE
   * modo diretrio e/ou arquivos} para o atributo
   * {@link #FILE_PARAMETER_ELEMENT_CATEGORY_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@value #FILE_PARAMETER_ELEMENT_CATEGORY_BOTH}.
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_CATEGORY_BOTH = "ambos";

  /**
   * <p>
   * O valor que indica o {@link FileParameterMode#REGULAR_FILE modo apenas
   * arquivos} para o atributo
   * {@link #FILE_PARAMETER_ELEMENT_CATEGORY_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@value #FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE}.
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE =
    "arquivo";

  /**
   * <p>
   * O atributo
   * {@value #FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE} do
   * elemento {@link InputFileParameter}.
   * </p>
   * 
   * Indica se o {@link InputFileParameter arquivo de entrada} deve exibir como
   * valor default o diretrio raz do projeto. Se ela no for informada, 
   * opcional, o seu valor-padro 
   * {@link #FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE} e  do tipo
   * booleano.
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE =
    "usar_diretorio_raiz_como_padrao";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE} do
   * elemento {@link InputFileParameter}
   * </p>
   * <p>
   * O seu valor 
   * {@value #FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_DEFAULT_VALUE}
   * .
   * </p>
   */
  protected static final boolean FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_DEFAULT_VALUE =
    false;

  /**
   * <p>
   * O atributo {@value #FILE_PARAMETER_ELEMENT_TYPE_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * </ul>
   * </p>
   * 
   * Indica tipo de arquivo aceito pelo parmetro do tipo arquivo,  opcional e
   * o seu tipo  string.
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_TYPE_ATTRIBUTE = "tipo";

  /**
   * <p>
   * O atributo {@value #FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE} do
   * elemento {@link InputFileParameter}.
   * </p>
   * 
   * Indica se o {@link OutputFileParameter arquivo de entrada} deve exibir um
   * filtro. Se ela no for informada,  opcional, o seu valor-padro 
   * {@link #FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE} e  do tipo
   * booleano.
   * </p>
   */
  protected static final String FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE =
    "usar_filtro";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE} do elemento
   * {@link OutputFileParameter}
   * </p>
   * <p>
   * O seu valor  {@value #FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE =
    false;

  /**
   * {@inheritDoc}
   */
  @Override
  public T createSimpleParameter(XmlParser parser, String name, String label,
    String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String type = parser.extractAttributeValue(
      FILE_PARAMETER_ELEMENT_TYPE_ATTRIBUTE, null);
    String modeName = parser.extractAttributeValue(
      FILE_PARAMETER_ELEMENT_CATEGORY_ATTRIBUTE, null);

    FileParameterMode mode;
    if (modeName == null) {
      if (type != null) {
        ProjectFileType projectFileType = ProjectFileType.getFileType(type);
        if (projectFileType.isDirectory()) {
          mode = FileParameterMode.DIRECTORY;
        }
        else {
          mode = FileParameterMode.REGULAR_FILE;
        }
      }
      else {
        mode = FileParameterMode.REGULAR_FILE;
      }
    }
    else if (modeName.equals(FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY)) {
      mode = FileParameterMode.DIRECTORY;
    }
    else if (modeName.equals(
      FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE)) {
      mode = FileParameterMode.REGULAR_FILE;
    }
    else if (modeName.equals(FILE_PARAMETER_ELEMENT_CATEGORY_BOTH)) {
      if (acceptBothCategory()) {
        mode = FileParameterMode.DIRECTORY_AND_REGULAR_FILE;
      }
      else {
        throw new ParseException("A categoria informada {0} no  vlida.\n"
          + "Categorias vlidas:\n{1};\n{2}.", new Object[] { modeName,
              FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE,
              FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY });
      }
    }
    else {
      if (acceptBothCategory()) {
        throw new ParseException("A categoria informada {0} no  vlida.\n"
          + "Categorias vlidas:\n{1};\n{2};{3}.", new Object[] { modeName,
              FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE,
              FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY,
              FILE_PARAMETER_ELEMENT_CATEGORY_BOTH });
      }
      else {
        throw new ParseException("A categoria informada {0} no  vlida.\n"
          + "Categorias vlidas:\n{1};\n{2}.", new Object[] { modeName,
              FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_REGULAR_FILE,
              FILE_PARAMETER_ELEMENT_CATEGORY_VALUE_DIRECTORY });
      }
    }

    if (type != null) {
      ProjectFileType projectFileType = ProjectFileType.getFileType(type);

      if (!projectFileType.isDirectory()
        && mode == FileParameterMode.DIRECTORY) {
        throw new ParseException(
          "A categoria informada {0} no  compatvel com o tipo informado {1}, "
            + "pois o tipo  aplicavel somente a arquivos.", modeName, type);
      }

      if (projectFileType.isDirectory()
        && mode == FileParameterMode.REGULAR_FILE) {
        throw new ParseException(
          "A categoria informada {0} no  compatvel com o tipo informado {1}, "
            + "pois o tipo  aplicavel somente a diretrios.", modeName, type);
      }
    }
    else {
      switch (mode) {
        case DIRECTORY:
          type = ProjectFileType.DIRECTORY_TYPE;
          break;
        case REGULAR_FILE:
        case DIRECTORY_AND_REGULAR_FILE:
          type = ProjectFileType.UNKNOWN;
          break;
        default:
          throw new IllegalStateException(String.format("Modo no tratado %s.",
            mode));
      }
    }

    return createFileParameter(parser, name, label, description, isOptional,
      isVisible, commandLinePattern, type, mode);
  }

  /**
   * Indica se a categoria {@link FILE_PARAMETER_ELEMENT_CATEGORY_BOTH} 
   * aceita.
   * 
   * @return {@code true} se for aceita ou {@code false} caso contrrio.
   */
  protected abstract boolean acceptBothCategory();

  /**
   * Cria uma instncia do parmetro de tipo <T>, a partir dos atributos bsicos
   * do parmetro. As subclasses devem implementar esse mtodo, fazendo a
   * extrao dos demais atributos do parmetro.
   * 
   * @param parser Parser xml do configurador.
   * @param name Nome do parmetro.
   * @param label Rtulo do parmetro.
   * @param description Dica do parmetro.
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern Padro da linha de comando do parmetro.
   * @param type Tipo do arquivo do parmetro.
   * @param mode O modo de funcionamento do parmetro.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   *         preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   *         parmetro com os atributos especificados.
   */
  protected abstract T createFileParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, String type, FileParameterMode mode)
      throws ParseException;

  /**
   * Converte strings para valores da enumerao FileParameterPipeAcceptance.
   */
  protected class FileParameterPipeAcceptanceConverter implements
    StringToEnumConverter<FileParameterPipeAcceptance> {

    /**
     * Retorna as strings que correspondem a cada um dos possveis valores da
     * enumerao {@link FileParameterPipeAcceptance}.
     * 
     * @param value O valor da enumerao.
     * @return Um array com todas as strings que podem representar o valor.
     */
    public String[] getPossibleMatches(FileParameterPipeAcceptance value) {
      switch (value) {
        case TRUE:
          return XmlParser.TRUE_VALUES;
        case FALSE:
          return XmlParser.FALSE_VALUES;
        case ALWAYS:
          return new String[] { "sempre", "always" };
        default:
          throw new BugException();
      }
    }

    /**
     * {@inheritDoc}
     * 
     */
    @Override
    public FileParameterPipeAcceptance valueOf(String string)
      throws ParseException {
      if (string != null) {
        String upperCaseString = string.toUpperCase();
        for (FileParameterPipeAcceptance mode : FileParameterPipeAcceptance
          .values()) {
          String[] possibleMatches = getPossibleMatches(mode);
          for (String possibleMatch : possibleMatches) {
            if (possibleMatch.toUpperCase().equals(upperCaseString)) {
              return mode;
            }
          }
        }
      }
      throw new ParseException("A string " + string
        + " no pde ser mapeada para um item de enumerao do tipo "
        + FileParameterPipeAcceptance.class.getName());
    }
  }

}
