package csbase.logic.algorithms.parsers.columns;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.BooleanColumn;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.TableColumn;
import csbase.logic.algorithms.parsers.XmlParser;

/**
 * Classe abstrata para facilitar a criao de {@link TableColumnFactory}.
 */
public abstract class AbstractTableColumnFactory implements TableColumnFactory {
  /**
   * O atributo {@value #COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE} dos elementos
   * do tipo coluna: indica o valor-padro da coluna,  opcional e o seu tipo
   * depende do tipo da coluna (Ex.: {@link BooleanColumn coluna de booleanos} -
   * booleano).
   */
  protected static final String COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE =
    "padrao";

  /**
   * O atributo {@value #COLUMN_ELEMENT_ID_ATTRIBUTE} dos elementos do tipo
   * coluna: define um identificador para a coluna,  opcional e  do tipo
   * string.
   */
  private static final String COLUMN_ELEMENT_ID_ATTRIBUTE = "id";

  /**
   * O atributo {@value #COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE} dos elementos do
   * tipo coluna: indica se a coluna  editvel,  opcional, o valor-padro 
   * {@link #COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE} e  do tipo booleano.
   */
  private static final String COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE = "editavel";

  /**
   * <p>
   * O valor-padro do atributo {@link #COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE}
   * dos elementos do tipo coluna.
   * </p>
   * <p>
   * O seu valor  {@value #COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE}.
   * </p>
   */
  private static final boolean COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE = true;

  /**
   * O atributo {@value #COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE} dos elementos do
   * tipo coluna: indica se a coluna  editvel,  opcional, o valor-padro 
   * {@link #COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE} e  do tipo booleano.
   */
  private static final String COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE = "opcional";

  /**
   * <p>
   * O valor-padro do atributo {@link #COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE}
   * dos elementos do tipo coluna.
   * </p>
   * <p>
   * O seu valor  {@value #COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE}.
   * </p>
   */
  private static final boolean COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE = false;

  /**
   * O atributo {@value #COLUMN_ELEMENT_LABEL_ATTRIBUTE} dos elementos do tipo
   * coluna: define um rtulo para a coluna,  obrigatrio e  do tipo string.
   */
  private static final String COLUMN_ELEMENT_LABEL_ATTRIBUTE = "rotulo";

  /**
   * O nome do elemento XML que esta fbrica  capaz de analisar.
   */
  private String elementName;

  /**
   * Cria a fbrica.
   *
   * @param elementName O nome do elemento (No aceita {@code null}).
   */
  protected AbstractTableColumnFactory(String elementName) {
    this.elementName = elementName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getElementName() {
    return elementName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final TableColumn<?> createColumn(XmlParser parser, String name,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String elementName = parser.getElementName();
    if (!elementName.equals(this.elementName)) {
      throw new ParseException(
        "O elemento {0} no  suportado por esta fbrica. Tipo suportado: {1}.",
        elementName, this.elementName);
    }
    String label = parser.extractAttributeValue(COLUMN_ELEMENT_LABEL_ATTRIBUTE);
    String id = parser.extractAttributeValue(COLUMN_ELEMENT_ID_ATTRIBUTE,
      label);
    boolean isOptional = parser.extractAttributeValueAsBoolean(
      COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE,
      COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE);
    boolean isEditable = parser.extractAttributeValueAsBoolean(
      COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE,
      COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE);
    return createColumn(parser, name, label, id, isOptional, isEditable,
      configurator);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract void setCellValue(XmlParser parser, String parameterName,
    TableColumn<?> column, int rowIndex, String valueAttributeName)
      throws ParseException;

  /**
   * Cria a coluna.
   *
   * @param parser O analisador de XML (No aceita {@code null}).
   * @param parameterName O nome da tabela que est sendo processado (No aceita
   *        {@code null}).
   * @param label O rtulo da coluna.
   * @param id O identificador da coluna.
   * @param isOptional Flag que indica se a coluna  opcional {@code true} ou
   *        obrigatrio {@code false}.
   * @param isEditable Flag que indica se a coluna  editvel {@code true} ou
   *        no {@code false}.
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   *
   * @return A coluna.
   *
   * @throws ParseException Se houver um erro no XML.
   */
  protected abstract TableColumn<?> createColumn(XmlParser parser,
    String parameterName, String label, String id, boolean isOptional,
    boolean isEditable, SimpleAlgorithmConfigurator configurator)
      throws ParseException;
}
