/*
 * $Id: SharedProjectEvent.java 116055 2011-03-04 18:15:45Z cassino $
 */
package csbase.logic;

import java.util.Collections;
import java.util.Set;

import csbase.logic.ProjectPermissions.SharingType;
import csbase.remote.RemoteEvent;

/**
 * Representa um evento sobre o compartilhamento de um projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class SharedProjectEvent extends RemoteEvent {
  /**
   * Os identificadores dos usurios que devem receber este evento. Caso esteja
   * {@code null}, todos os usurios devem receber o evento.
   */
  private Set<Object> eventUsersId;
  /**
   * Informaes sobre o projeto.
   */
  private CommonProjectInfo projectInfo;
  /**
   * Usurios com direito de acesso RO ao projeto. Caso o projeto seja privado
   * ou pblico, ter o valor {@code null}.
   */
  private Set<Object> usersRO;
  /**
   * Usurios com direito de acesso RW ao projeto. Caso o projeto seja privado
   * ou pblico, ter o valor {@code null}.
   */
  private Set<Object> usersRW;
  /**
   * Tipo de compartilhamento do projeto.
   */
  private final SharingType sharingType;

  /**
   * Cria um evento de compartilhamento de projeto. O projeto ser compartilhado
   * como pblico e, portanto, todos os usurios do sistema estaro aptos a
   * receber este evento.
   * 
   * @param projectInfo - informaes sobre o projeto.
   * @param sharingType - tipo de compartilhamento
   */
  public SharedProjectEvent(CommonProjectInfo projectInfo,
    SharingType sharingType) {
    this(null, projectInfo, null, null, sharingType);
  }

  /**
   * Cria um evento de compartilhamento de projeto.
   * 
   * @param eventUsersId Os usurios que recebero o evento. Deve ser
   *        <code>null</code>, caso o projeto seja privado ou pblico.
   * @param projectInfo As informaes sobre o projeto.
   * @param usersRO conjunto com os identificadores dos usurios que tem acesso
   *        RO ao projeto. Deve ser <code>null</code>, caso o projeto seja
   *        privado ou pblico.
   * @param usersRW conjunto com os identificadores dos usurios que tem acesso
   *        RW ao projeto. Deve ser <code>null</code>, caso o projeto seja
   *        privado ou pblico.
   * @param sharingType tipo de compartilhamento
   */
  public SharedProjectEvent(Set<Object> eventUsersId,
    CommonProjectInfo projectInfo, Set<Object> usersRO, Set<Object> usersRW,
    SharingType sharingType) {
    if (sharingType != SharingType.PARTIAL) {
      if (eventUsersId != null || usersRO != null || usersRW != null) {
        throw new IllegalArgumentException(
          "conjuntos de usurios != null para compartilhamento " + sharingType);
      }
    }
    this.projectInfo = projectInfo;
    this.sharingType = sharingType;
    this.eventUsersId =
      eventUsersId == null ? null : Collections.unmodifiableSet(eventUsersId);
    this.usersRO = usersRO;
    this.usersRW = usersRW;
  }

  /**
   * Obtm as informaes sobre o projeto.
   * 
   * @return As informaes sobre o projeto.
   */
  public final CommonProjectInfo getProjectInfo() {
    return this.projectInfo;
  }

  /**
   * Obtm os usurios com direito de acesso RO ao projeto.
   * 
   * @return Os usurios com direto de acesso RO ao projeto, ou
   *         <code>null</code> caso o projeto seja pblico ou privado
   */
  public Set<Object> getUsersRO() {
    return usersRO;
  }

  /**
   * Obtm os usurios com direito de acesso RW ao projeto.
   * 
   * @return Os usurios com direto de acesso RW ao projeto, ou
   *         <code>null</code> caso o projeto seja pblico ou privado
   */
  public Set<Object> getUsersRW() {
    return usersRW;
  }

  /**
   * Verifica se o projeto  pblico.
   * 
   * @return {@code true} caso o projeto seja pblico, ou {@code false}, caso
   *         contrrio.
   */
  public boolean isPublic() {
    return sharingType == SharingType.ALL_RO
      || sharingType == SharingType.ALL_RW;
  }

  /**
   * Verifica se o projeto  pblico com acesso somente leitura.
   * 
   * @return {@code true} caso o projeto seja pblico e somente para leitura, ou
   *         {@code false}, caso contrrio.
   */
  public boolean isReadOnly() {
    return sharingType == SharingType.ALL_RO;
  }

  /**
   * Verifica se o projeto foi compartilhado para um determinado usurio.
   * 
   * @param userId O identificador do usurio.
   * 
   * @return {@code true} caso o usurio tenha acesso ao projeto, ou {@code
   *         false}, caso contrrio.
   */
  public boolean contains(Object userId) {
    if (isPublic()) {
      return true;
    }
    if (sharingType == SharingType.PRIVATE) {
      return false;
    }
    return usersRO.contains(userId) || usersRW.contains(userId);
  }

  /**
   * Obtm o conjunto de usurios que devem receber este evento. Caso esteja
   * {@code null}, todos os usurios devem receber o evento.
   * 
   * @return O conjunto de usurios que devem receber o evento.
   */
  public Set<Object> getEventUsersId() {
    return eventUsersId;
  }

  /**
   * Obtm o novo tipo de compartilhamento.
   * 
   * @return novo tipo de compartilhamento
   */
  public SharingType getSharingType() {
    return sharingType;
  }
}