/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters.validators;

import java.rmi.RemoteException;
import java.util.Arrays;

import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.ServiceFailureException;
import csbase.logic.ClientProjectFile;
import csbase.logic.algorithms.parameters.FileParameter;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.logic.algorithms.validation.ValidationSuccess;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Validador do {@link FileParameter Parmetro do Tipo Arquivo}.
 *
 * @author lmoreira
 */
public class FileParameterValidator extends SimpleParameterValidator<FileURLValue> {

  /**
   * Cria um validador de arquivo.
   *
   * @param optional Indica se o valor  opcional {@code true} ou obrigatrio
   *        {@code false}.
   */
  public FileParameterValidator(boolean optional) {
    super(optional);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Validation validateValue(SimpleParameter<?> parameter,
    FileURLValue value, ValidationContext context) throws RemoteException {
    if (!parameter.isVisible()) {
      return new ValidationSuccess();
    }
    if (!parameter.isEnabled()) {
      return new ValidationSuccess();
    }
    FileParameter fileParameter = (FileParameter) parameter;
    if (fileParameter.hasLink()) {
      return new ValidationSuccess();
    }
    else {
      if (fileParameter.usesPipe() == FileParameterPipeAcceptance.ALWAYS
        && context.getMode() == ValidationMode.FULL) {
        LocalizedMessage message =
          new LocalizedMessage(FileParameterValidator.class, "must_use_pipe",
            new Object[] { parameter.getLabel() });
        return new ValidationError(message);
      }
    }
    Validation result = super.validateValue(parameter, value, context);
    if (!result.isWellSucceded()) {
      return result;
    }
    if (value != null) {
      return validateFile(value, fileParameter.getMode(), context
        .getProjectId(), fileParameter.mustExist());
    }
    return new ValidationSuccess();
  }

  /**
   * Valida um arquivo selecionado pelo usurio.
   *
   * @param file O arquivo (No aceita {@code null}).
   * @param mode O modo do parmetro (No aceita {@code null}).
   * @param projectId Identificador do projeto no qual o algoritmo vai ser
   *        executado. Esse identificador  necessrio para validar, por
   *        exemplo, se existe o arquivo escolhido.
   * @param mustExist Indica se o arquivo deve existir.
   *
   * @return o resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  public Validation validateFile(FileURLValue file, FileParameterMode mode,
    Object projectId, boolean mustExist) throws RemoteException {
    if (projectId == null) {
      LocalizedMessage message =
        new LocalizedMessage(FileParameterValidator.class, "no_project",
          new Object[] { file.getPath() });
      return new ValidationError(message);
    }
    String[] path = file.getPathAsArray();
    Validation fileNameResult = validateFileName(path);
    if (!fileNameResult.isWellSucceded()) {
      return fileNameResult;
    }
    ClientProjectFile projectFile = getFile(path, projectId);
    if (projectFile != null) {
      return validateFileMode(mode, projectFile);
    }
    else {
      if (mustExist) {
        LocalizedMessage message =
          new LocalizedMessage(FileParameterValidator.class, "file_not_found",
            new Object[] { file.getPath() });
        return new ValidationError(message);
      }
      else {
        return validateParent(file, projectId);
      }
    }
  }

  /**
   * Verifica se o arquivo tem um nome e um caminho vlidos.
   *
   * @param path o caminho do arquivo.
   * @return o resultado da validao.
   */
  public static Validation validateFileName(String[] path) {
    for (String component : path) {
      String fixedFileName = FileUtils.fixFileName(component);
      if (!fixedFileName.equals(component)) {
        LocalizedMessage message =
          new LocalizedMessage(FileParameterValidator.class,
            "invalid_file_name", new Object[] { component });
        return new ValidationError(message);
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Valida o arquivo escolhido em relao ao modo do parmetro. Se parmetro
   * aceitar somente diretrios, por exemplo, a validao falha ao encontrar um
   * arquivo comum.
   *
   * @param mode o modo do parmetro.
   * @param projectFile o arquivo selecionado.
   *
   * @return o resultado da validao.
   */
  private Validation validateFileMode(FileParameterMode mode,
    ClientProjectFile projectFile) {
    if (mode == FileParameterMode.DIRECTORY && !projectFile.isDirectory()) {
      LocalizedMessage message =
        new LocalizedMessage(FileParameterValidator.class, "not_directory",
          new Object[] { projectFile.getStringPath() });
      return new ValidationError(message);
    }
    else if (mode == FileParameterMode.REGULAR_FILE
      && projectFile.isDirectory()) {
      LocalizedMessage message =
        new LocalizedMessage(FileParameterValidator.class, "not_file",
          new Object[] { projectFile.getStringPath() });
      return new ValidationError(message);
    }
    else {
      return new ValidationSuccess();
    }
  }

  /**
   * Valida o diretrio onde o arquivo se encontra. O diretrio-pai do arquivo
   * precisa existir, mesmo que o arquivo sendo validado no exista.
   *
   * @param file o valor do parmetro sendo validado.
   * @param projectId o identificador do projeto que contm o arquivo.
   *
   * @return o resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  private Validation validateParent(FileURLValue file, Object projectId)
    throws RemoteException {
    String[] parentPath = file.getPathAsArray();
    while (parentPath.length > 1) {
      parentPath = Arrays.copyOf(parentPath, parentPath.length - 1);
      String parentName = parentPath[parentPath.length - 1];
      if (!parentName.equals(".")) {
        ClientProjectFile parentProjectFile = getFile(parentPath, projectId);
        if (parentProjectFile == null) {
          LocalizedMessage message =
            new LocalizedMessage(FileParameterValidator.class,
              "file_not_found", new Object[] { FileUtils.joinPath(parentPath) });
          return new ValidationError(message);
        }
        if (!parentProjectFile.isDirectory()) {
          LocalizedMessage message =
            new LocalizedMessage(FileParameterValidator.class, "not_directory",
              new Object[] { parentProjectFile.getStringPath() });
          return new ValidationError(message);
        }
        else {
          return new ValidationSuccess();
        }
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Obtm um arquivo de projeto.
   *
   * @param path o caminho do arquivo.
   * @param projectId o identificador do projeto onde o arquivo se encontra.
   * @return o arquivo de projeto ou nulo, caso no exista ou no seja possvel
   *         obt-lo.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  private ClientProjectFile getFile(String[] path, Object projectId)
    throws RemoteException {
    try {
      ProjectServiceInterface projectService =
        ClientRemoteLocator.projectService;
      if (projectService.existsFile(projectId, path)) {
        return projectService.getChild(projectId, path);
      }
      else {
        return null;
      }
    }
    catch (ServiceFailureException e) {
      return null;
    }
  }
}