package csbase.logic.algorithms.serializer;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import java.rmi.RemoteException;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import csbase.exception.ParseException;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerException;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerIOException;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerParameterNotFoundException;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerParseException;

import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;

/**
 * Objeto utilizado para (des)serializar as configuraces de um configurador de
 * algortmos.
 *
 * @author Tecgraf / PUC-Rio
 */
public final class DefaultAlgorithmConfigurationSerializer implements
IAlgorithmConfigurationSerializer {

  /**
   * Enconding usado no xml.
   */
  private static final String ENCODING = "ISO-8859-1";

  /**
   * Charset usado no xml.
   */
  private static final Charset CHARSET = Charset.forName(ENCODING);

  /**
   * Template do configurador algoritmo
   */
  private static final String ALGORITHM_CONFIGURATOR_XML_TEMPLATE =
    "<?xml version='1.0' encoding='" + ENCODING + "'?>\n"
      + "<algoritmo id='%1$s' nome='%2$s' versao='%3$s'>\n" + "%4$s" + // aqui entram os parmetros no formato de PARAMETER_XML_TEMPLATE
      "%5$s" + // aqui entram o parmetro no formato de STDOUT_XML_TEMPLATE
      "%6$s" + // aqui entram o parmetro no formato de EXIT_CODE_XML_TEMPLATE
      "</algoritmo>";

  /**
   * Template da tag de parmetros do algoritmo.
   */
  private static final String PARAMETER_XML_TEMPLATE =
    "\t<parametro nome='%1$s'>\n" + "\t\t<valor>\n"
      + "\t\t\t<![CDATA[%2$s]]>\n" + "\t\t</valor>\n" + "\t</parametro>\n";

  /**
   * Template da tag de parmetros nulos do algoritmo.
   */
  private static final String PARAMETER_NULL_XML_TEMPLATE =
    "\t<parametro nome='%1$s'>\n" + "\t</parametro>\n";

  /**
   * Template da tag de sada padro do algoritmo.
   */
  private static final String STDOUT_XML_TEMPLATE = "\t<stdout tipo='%1$s'>\n"
    + "\t\t<![CDATA[%2$s]]>\n" + "\t</stdout>\n";

  /**
   * Template da tag de arquivo de cdigo de sada do algoritmo.
   */
  private static final String EXIT_CODE_XML_TEMPLATE =
    "\t<exitcode tipo='%1$s'>\n" + "\t\t<![CDATA[%2$s]]>\n" + "\t</exitcode>\n";

  /**
   * @param input fluxo de dados com a configuraco de um AlgorithmConfigurator
   *        serializada (No aceita {@code null}).
   *
   * @return Um configurador de algortmos com seus parmetros preenchidos.
   *
   * @throws AlgorithmConfigurationSerializerParseException Quando os dados do
   *         fluxo no estiverem formatados corretamente.
   * @throws AlgorithmConfigurationSerializerIOException Quando ocorrer um erro
   *         de IO no fluxo.
   * @throws AlgorithmNotFoundException Quando a verso do algoritmo utilizado
   *         para criar este configurador no existe mais.
   * @throws AlgorithmConfigurationSerializerException Demais erros que possam
   *         ocorrer na implementaco deste mtodo.
   */
  @Override
  public AlgorithmConfigurator read(InputStream input)
    throws AlgorithmConfigurationSerializerParseException,
    AlgorithmConfigurationSerializerIOException,
    AlgorithmConfigurationSerializerException, AlgorithmNotFoundException {

    if (null == input) {
      throw new IllegalArgumentException();
    }

    // Guarda o nome do algoritmo.
    String algorithmName = null;
    // Guarda identificador da verso do algoritmo.
    AlgorithmVersionId algorithmVersionId = null;
    // Guarda o nome do parmetro sendo lido.
    String parameterName = null;

    try {
      DocumentBuilder builder =
        DocumentBuilderFactory.newInstance().newDocumentBuilder();
      InputSource source = new InputSource(input);
      source.setEncoding(ENCODING);
      Document document = builder.parse(source);

      // Pegando o n representando o elemento "algoritmo" no xml
      Element algorithmElement = document.getDocumentElement();

      Object algorithmId = null;
      // Verificando se o id do algoritmo foi definido
      if(algorithmElement.hasAttribute("id")) {
        algorithmId = algorithmElement.getAttribute("id");
      }

      // Pegando o valor do atributo "nome"
      algorithmName = algorithmElement.getAttribute("nome");
      // Pegando o valor do atributo "versao"
      String algorithmVersion = algorithmElement.getAttribute("versao");

      // Cria o configurador
      algorithmVersionId = AlgorithmVersionId.create(algorithmVersion);

      AlgorithmServiceInterface algorithmService =
        ClientRemoteLocator.algorithmService;

      // Busca pelo identificador, se disponvel, para encontrar algoritmos
      // que tenham sido renomeados.
      String name = algorithmName;
      if (algorithmId != null) {
        AlgorithmInfo info = algorithmService.getInfo(algorithmId);
        if (info != null) {
          name = info.getName();
        }
      }
      AlgorithmConfigurator configurator =
        algorithmService.createAlgorithmConfigurator(name, algorithmVersionId);

      /**
       * Pega uma lista de ns representando os elementos "parametro" para
       * atribuir os valores dos parmetros no configurador.
       */
      NodeList parameters = algorithmElement.getElementsByTagName("parametro");
      for (int index = 0; index < parameters.getLength(); index++) {
        Element parameterElement = (Element) parameters.item(index);
        // Pegando o valor do atributo "nome" de um elemento "parametro".
        parameterName = parameterElement.getAttribute("nome");

        // Pegando o texto do elemento "valor" filho do elemento "parametro".
        String parameterValue = null;
        NodeList parameterValueList =
          parameterElement.getElementsByTagName("valor");
        if (null != parameterValueList && 0 < parameterValueList.getLength()) {
          Element parameterValueElement = (Element) parameterValueList.item(0);
          parameterValue = parameterValueElement.getTextContent().trim();
        }

        // Atribui no configurador o valor de um parmetro lido.
        configurator.setParameterValue(parameterName, parameterValue);
      }

      NodeList stdOut = algorithmElement.getElementsByTagName("stdout");
      if (0 < stdOut.getLength()) {
        Element stdOutElement = (Element) stdOut.item(0);
        String outputFilePath = stdOutElement.getTextContent();
        if (null != outputFilePath) {
          outputFilePath = outputFilePath.trim();
          if (!"".equals(outputFilePath)) {

            String type = stdOutElement.getAttribute("tipo");
            type =
              (null == type || "".equals(type.trim())) ? "LOG" : type.trim();

            FileURLValue outputFile = new FileURLValue(outputFilePath, type);
            configurator.setStandardOutputFile(outputFile);
          }
        }
      }

      NodeList exitCode = algorithmElement.getElementsByTagName("exitcode");
      if (0 < exitCode.getLength()) {
        Element exitCodeElement = (Element) exitCode.item(0);
        String exitCodeFilePath = exitCodeElement.getTextContent();
        if (null != exitCodeFilePath) {
          exitCodeFilePath = exitCodeFilePath.trim();
          if (!"".equals(exitCodeFilePath)) {

            String type = exitCodeElement.getAttribute("tipo");
            type =
              (null == type || "".equals(type.trim())) ? "LOG" : type.trim();

            FileURLValue exitCodeFile = new FileURLValue(exitCodeFilePath, type);
            configurator.setExitCodeLogFile(exitCodeFile);
            configurator.setHasExitCode(true);
          }
        }
      }

      return configurator;
    }
    catch (RemoteException e) {
      throw new AlgorithmConfigurationSerializerException(e);
    }
    catch (ParameterNotFoundException e) {
      throw new AlgorithmConfigurationSerializerParameterNotFoundException(e,
        algorithmName, algorithmVersionId, parameterName);
    }
    catch (ParseException e) {
      throw new AlgorithmConfigurationSerializerParseException(e);
    }
    catch (ParserConfigurationException e) {
      throw new AlgorithmConfigurationSerializerParseException(e);
    }
    catch (SAXException e) {
      throw new AlgorithmConfigurationSerializerParseException(e);
    }
    catch (IOException e) {
      throw new AlgorithmConfigurationSerializerIOException(e);
    }
  }

  /**
   * @param configurator Um configurador de algortmos (No aceita {@code null}
   *        ).
   * @param output Fluxo de dados para o qual ser serializada a configuraco do
   *        configurador de algortmos (No aceita {@code null}).
   *
   * @throws AlgorithmConfigurationSerializerIOException Quando ocorrer um erro
   *         de IO no fluxo.
   * @throws AlgorithmConfigurationSerializerException Demais erros que possam
   *         ocorrer na implementaco deste mtodo.
   */
  @Override
  public void write(AlgorithmConfigurator configurator, OutputStream output)
    throws AlgorithmConfigurationSerializerException {

    if (null == configurator) {
      throw new IllegalArgumentException();
    }
    if (null == output) {
      throw new IllegalArgumentException();
    }

    /** Guarda o nome do parmetro sendo escrito. */
    String parameterName = null;

    try {
      StringBuilder parameters = new StringBuilder();
      Set<String> parameterNames = configurator.getParameterNames();
      for (String aParameterName : parameterNames) {
        /**
         * Armazena o valor para ser utilizado na exceo caso o parmetro no
         * exista.
         */
        parameterName = aParameterName;
        String aParameterValue = configurator.getParameterValue(aParameterName);

        String parameterXml;
        if (null == aParameterValue) {
          parameterXml =
            String.format(PARAMETER_NULL_XML_TEMPLATE, aParameterName);
        }
        else {
          parameterXml =
            String.format(PARAMETER_XML_TEMPLATE, aParameterName,
              aParameterValue);
        }
        parameters.append(parameterXml);
      }

      String stdOut;
      if (null == configurator.getStandardOutputFile()) {
        stdOut = "";
      }
      else {
        stdOut =
          String.format(STDOUT_XML_TEMPLATE, configurator
            .getStandardOutputFile().getType(), configurator
            .getStandardOutputFile().getPath());
      }

      String exitCode;
      if (!configurator.hasExitCode()
        || null == configurator.getExitCodeLogFile()) {
        exitCode = "";
      }
      else {
        exitCode =
          String.format(EXIT_CODE_XML_TEMPLATE, configurator
            .getExitCodeLogFile().getType(), configurator.getExitCodeLogFile()
            .getPath());
      }

      String id = "";
      AlgorithmVersionInfo algorithmVersion= configurator.getAlgorithmVersion();
      if (algorithmVersion != null) {
        AlgorithmInfo info = algorithmVersion.getInfo();
        if (info != null) {
          id = info.getId();
        }
      }

      String algorithmConfiguratorXml = String
        .format(ALGORITHM_CONFIGURATOR_XML_TEMPLATE, id,
          configurator.getAlgorithmName(),
          configurator.getAlgorithmVersionId().toString(), parameters.toString(),
          stdOut, exitCode);

      PrintWriter writer =
        new PrintWriter(new OutputStreamWriter(output, CHARSET));
      writer.write(algorithmConfiguratorXml);
      writer.flush();
    }
    catch (ParameterNotFoundException e) {
      /**
       * J que quem nos deu o nome o parmetro para atravs deste recuperar o
       * seu label, tipo e valor foi o prprio configurador; esta exceo nunca
       * dever ser lanada. De qualquer forma, se por alguma inconsistncia dos
       * dados do configurador isso vier a ocorrer em tempo de execuo, a
       * exceo ser encapsulada por uma RuntimeException e lanada.
       */
      throw new AlgorithmConfigurationSerializerParameterNotFoundException(e,
        configurator.getAlgorithmName(), configurator.getAlgorithmVersionId(),
        parameterName);
    }
  }
}
