/*
 * AlgorithmVersionInfo.java
 * 
 * $Author: isabella $ $Revision: 172547 $ - $Date: 2009-04-28 16:38:26 -0300
 * (Tue, 28 Apr 2009) $
 */
package csbase.logic.algorithms;

import java.io.Serializable;
import java.text.DecimalFormat;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;

import csbase.logic.FileInfo;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;

/**
 * Representa os dados de uma verso de algoritmo no SSI.
 * 
 * @author clinio
 */
public final class AlgorithmVersionInfo implements
  Comparable<AlgorithmVersionInfo>, Serializable {
  /**
   * Nome do diretrio que contm os executveis para as plataformas (contido no
   * diretrio de uma verso).
   */
  public static String BIN_DIR = "bin";

  /** Nome do subdiretrio/subpacote (dentro do diretrio de uma verso) */
  public static final String CONFIGURATOR_DIR = "configurator";

  /** Nome do pacote (default) aonde se busca a documentao do algoritmo. */
  public static final String DOCUMENTATION_DIR = "html";

  /** Nome do arquivo de ajuda. */
  public static final String HTML_FILE = "index.html";

  /** Nome do arquivo que define o configurador de algoritmos. */
  public static final String CONFIG_FILE = "config.xml";

  /**
   * Nome do arquivo de configurao de uma verso de um algoritmo.
   */
  public static final String CONFIG_PROPERTIES_FILE = "config.properties";

  /**
   * Nome do arquivo de configurao de parmetros de uma verso de um
   * algoritmo.
   */
  public static final String PARAMETERS_PROPERTIES_FILE =
    "parameters.properties";

  /** Nome do arquivo que define o configurador de fluxos de algoritmo. */
  public static final String FLOW_CONFIG_FILE = "config.flx";

  /** Arquivo auxiliar com elementos extras para os configuradores. */
  private static final String PROPERTIES_FILE = "config.properties";

  /** Determina se a verso corresponde a um algoritmo simples ou fluxo **/
  private final ConfiguratorType type;

  /** Informaes sobre o algoritmo. */
  private AlgorithmInfo algorithmInfo;

  /** Arquivos de configuradores da versao */
  private List<FileInfo> configurators;

  /** Localizao (diretrio) da verso de um algoritmo. */
  private String dir;

  /** Arquivos de documentacao da versao */
  private List<FileInfo> documentation;

  /** Identificador da verso */
  private AlgorithmVersionId id;

  /** Descrio da verso do algoritmo */
  private String description;

  /**
   * Nome do arquivo de informacoes do algoritmo. Contem os VALORES atribuidos a
   * cada um dos atributos estendidos de uma dada instancia do algoritmo.
   */
  public static final String PROPERTY_VALUES_FILE = "version.properties";

  /**
   * Mapa de valores para os atributos estendidos.
   */
  private Map<String, String> propertyValues;

  /**
   * Hashtable com as plataformas suportadas. A chave da tabela e' a plataforma
   * e o valor e' uma lista de executaveis dessa plataforma. Ex: { Windows ->
   * {alg.exe, aux.exe}}
   */
  private Map<String, List<FileInfo>> supportedPlatforms = null;

  /**
   * Construtor da representao da verso de um algoritmo. Utilizado para criar
   * uma nova versao no servico. Recebe a lista de atributos estendidos.
   * 
   * @param algorithmInfo Informaes sobre o algoritmo.
   * @param id identificao da verso
   * @param supportedPlatforms Plataformas suportadas.
   * @param propertyValues Mapa de valores para os atributos estendidos.
   */
  public AlgorithmVersionInfo(AlgorithmInfo algorithmInfo,
    AlgorithmVersionId id, Map<String, List<FileInfo>> supportedPlatforms,
    Map<String, String> propertyValues, ConfiguratorType type) {
    this(algorithmInfo, supportedPlatforms, type);
    this.id = id;
    int[] numbers = id.getNumbers();
    this.dir = getDirectoryFor(numbers[0], numbers[1], numbers[2]);
    this.propertyValues = propertyValues;
  }

  /**
   * Construtor da representao da verso de um algoritmo. Utilizado para
   * incluir uma versao no algoritmo (carrega atributos estendidos do arquivo de
   * propriedades)
   *
   * @param algorithmInfo Informaes sobre o algoritmo.
   * @param versionDirName O node do diretrio da verso.
   * @param supportedPlatforms Plataformas suportadas.
   * @param propertyValues Mapa de valores para os atributos estendidos
   * @param type tipo da verso (simples ou fluxo).
   */
  public AlgorithmVersionInfo(AlgorithmInfo algorithmInfo, String versionDirName,
    Map<String, List<FileInfo>> supportedPlatforms,
    Map<String, String> propertyValues, ConfiguratorType type) {
    this(algorithmInfo, supportedPlatforms, type);
    this.id = getVersionIdFromDirName(versionDirName);
    int[] numbers = this.id.getNumbers();
    this.dir = getDirectoryFor(numbers[0], numbers[1], numbers[2]);
    this.propertyValues = propertyValues;
  }

  /**
   * Parte comum do processo de instanciacao de um algoritmo. Os outros dois
   * construtores fazem uso deste.
   *
   * @param algorithmInfo Informacoes sobre o algoritmo.
   * @param supportedPlatforms plataformas suportadas.
   * @param type tipo da verso (simples ou fluxo).
   */
  private AlgorithmVersionInfo(AlgorithmInfo algorithmInfo,
    Map<String, List<FileInfo>> supportedPlatforms, ConfiguratorType type) {
    this.algorithmInfo = algorithmInfo;
    this.supportedPlatforms = supportedPlatforms;
    this.documentation = new Vector<FileInfo>();
    this.configurators = new Vector<FileInfo>();
    this.type = type;
  }

  /**
   * Constri as informaes da verso de um algoritmo durante o processo de
   * leitura das verses em um pacote de algoritmos.
   * 
   * @param id identificador de uma verso do algoritmo
   * @param description descrio  do algoritmo
   * @param propertyValues valores das propriedades
   * @param type tipo da verso (simples ou fluxo).
   */
  public AlgorithmVersionInfo(AlgorithmVersionId id, String description,
    Map<String, String> propertyValues, ConfiguratorType type) {
    this.id = id;
    this.setDescription(description);
    this.propertyValues = propertyValues;
    this.type = type;
  }

  /**
   * Obtem o nome do diretrio de uma verso partir de seus nmeros.
   * 
   * @param major numero da verso
   * @param minor numero da reviso
   * @param patch numero da correo
   * 
   * @return o identificador como uma String
   */
  public static String getDirectoryFor(int major, int minor, int patch) {
    char sep = '_';
    DecimalFormat df = new DecimalFormat("000");
    StringBuffer buf = new StringBuffer("v_");
    buf.append(df.format(major));
    buf.append(sep);
    buf.append(df.format(minor));
    buf.append(sep);
    buf.append(df.format(patch));
    return buf.toString();
  }

  /**
   * Obtm o identificador da verso do algoritmo associando ao diretrio
   * fornecido.
   * 
   * @param dirName O nome do diretrio.
   * 
   * @return O identificador da verso do algoritmo.
   */
  public static Object getIdFromDirectory(String dirName) {
    String[] version = dirName.split("_");
    int major = Integer.parseInt(version[1]);
    int minor = Integer.parseInt(version[2]);
    int patch = Integer.parseInt(version[3]);
    return new AlgorithmVersionId(major, minor, patch);
  }

  /**
   * Obtem um identificador de verso a partir de seus nmeros.
   * 
   * @param major numero da verso
   * @param minor numero da reviso
   * @param patch numero da correo
   * 
   * @return o identificador da verso
   */
  public static Object getIdFor(int major, int minor, int patch) {
    return new AlgorithmVersionId(major, minor, patch);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(AlgorithmVersionInfo o) {
    return -1 * this.id.compareTo(o.id);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!(obj instanceof AlgorithmVersionInfo)) {
      return false;
    }
    AlgorithmVersionInfo version = (AlgorithmVersionInfo) obj;
    return this.id.equals(version.id);
  }

  /**
   * Obtm as informaes sobre o algoritmo.
   * 
   * @return As informaes sobre o algoritmo.
   */
  public AlgorithmInfo getInfo() {
    return this.algorithmInfo;
  }

  /**
   * Atribui os arquivos de documentao da verso.
   * 
   * @param documentation Lista de arquivos de documentao.
   */
  public void setDocumentation(List<FileInfo> documentation) {
    this.documentation = documentation;
  }

  /**
   * Remove o arquivo de documentao da verso.
   * 
   * @param file O arquivo a ser removido.
   */
  public void removeDocumentation(FileInfo file) {
    this.documentation.remove(file);
  }

  /**
   * Adiciona um arquivo de documentao da verso.
   * 
   * @param file Arquivo a ser adicionado.
   */
  public void addDocumentation(FileInfo file) {
    if (!this.documentation.contains(file)) {
      this.documentation.add(file);
    }
  }

  /**
   * Obtm os arquivos de documentao da verso.
   * 
   * @return Lista de arquivos de documentao.
   */
  public List<FileInfo> getDocumentation() {
    return this.documentation;
  }

  /**
   * Atribui os arquivos de configurao da verso.
   * 
   * @param configurators A lista de arquivos de configurao.
   */
  public void setConfigurators(List<FileInfo> configurators) {
    this.configurators = configurators;
  }

  /**
   * Remove o arquivo de configurao da verso.
   * 
   * @param file O arquivo a ser removido.
   */
  public void removeConfigurator(FileInfo file) {
    configurators.remove(file);
  }

  /**
   * Adiciona um arquivo de configurao da verso.
   * 
   * @param file Arquivo a ser adicionado.
   */
  public void addConfigurator(FileInfo file) {
    if (!this.configurators.contains(file)) {
      this.configurators.add(file);
    }
  }

  /**
   * Obtm os arquivos de configurao da verso.
   * 
   * @return Lista de arquivos de documentao.
   */
  public List<FileInfo> getConfigurators() {
    return this.configurators;
  }

  /**
   * Consulta ao nome do diretrio de executveis para as plataformas.
   * 
   * @return o nome do diretrio
   */
  public String getBinDirName() {
    return BIN_DIR;
  }

  /**
   * Consulta ao nome do sub-pacote do configurador.
   * 
   * @return o nome do sub-pacote como uma string.
   */
  public String getConfiguratorDirName() {
    return CONFIGURATOR_DIR;
  }

  /**
   * Consulta ao nome do sub-pacote da documentao.
   * 
   * @return o nome do sub-pacote como uma string.
   */
  public String getDocumentationDirName() {
    return DOCUMENTATION_DIR;
  }

  /**
   * Obtm o subdiretrio da verso. <B>Nao</b> inclui o caminho ate ele!
   * 
   * @return o nome do subdiretrio desejado
   */
  public String getDirectory() {
    return this.dir;
  }

  /**
   * Obtm o identificador.
   * 
   * @return a identificao
   */
  public AlgorithmVersionId getId() {
    return this.id;
  }

  /**
   * Consulta ao nome do pacote do algoritmo e verso.
   * 
   * @return o nome do pacote como uma string.
   */
  public String getPackageName() {
    return this.algorithmInfo.getDirectory() + "." + AlgorithmInfo.VERSIONS_DIR
      + "." + this.dir + "." + CONFIGURATOR_DIR;
  }

  /**
   * Obtm as plataformas suportadas.
   * 
   * @return vetor contendo os identificadores das plataformas
   */
  public Vector<String> getSupportedPlatforms() {
    Vector<String> result = new Vector<String>();
    for (String supportedPlatform : this.supportedPlatforms.keySet()) {
      result.add(supportedPlatform);
    }
    return result;
  }

  /**
   * Obtm as plataformas suportadas com os executaveis.
   * 
   * @return Hashtable com as plataformas e os executaveis correspondentes.
   */
  public Map<String, List<FileInfo>> getPlatforms() {
    return this.supportedPlatforms;
  }

  /**
   * Obtm os nomes das plataformas suportadas.
   * 
   * @return Lista com as plataformas.
   */
  public Set<String> getPlatformsNames() {
    return this.supportedPlatforms.keySet();
  }

  /**
   * Obtm os arquivos executveis de uma plataforma da verso.
   * 
   * @param platform Plataforma dos executveis.
   * 
   * @return Lista de arquivos executveis.
   */
  public List<FileInfo> getExecutables(String platform) {
    return this.supportedPlatforms.get(platform);
  }

  /**
   * Coloca os executaveis de uma plataforma.
   * 
   * @param platform .
   * @param execs .
   */
  public void setPlatformExecutables(String platform, List<FileInfo> execs) {
    this.supportedPlatforms.put(platform, execs);
  }

  /**
   * Atribui as plataformas dos executveis da verso.
   * 
   * @param platforms Lista de plataformas da verso, mapeadas pelo nome.
   */
  public void setPlatforms(Hashtable<String, List<FileInfo>> platforms) {
    this.supportedPlatforms = platforms;
  }

  /**
   * Adiciona nova plataforma de executveis da verso.
   * 
   * @param platform Platafroma de executveis.
   */
  public void setPlatform(String platform) {
    this.supportedPlatforms.put(platform, new Vector<FileInfo>());
  }

  /**
   * Consulta ao nome do sub-pacote de verses.
   * 
   * @return o nome do sub-pacote como uma string.
   */
  public String getVersionsDirName() {
    return AlgorithmInfo.VERSIONS_DIR;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.id.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.id.toString();
  }

  /**
   * Remove o executvel de uma plataforma da verso.
   * 
   * @param platform Plataforma do executvel.
   * @param executable Arquico executvel.
   * @return Verdadeiro se o executvel foi removido ou falso, caso contrrio.
   */
  public boolean removeExecutable(String platform, FileInfo executable) {
    Collection<FileInfo> execs = this.supportedPlatforms.get(platform);
    if (execs != null) {
      return execs.remove(executable);
    }
    return false;
  }

  /**
   * Obtem o diretorio completo para a versao (repositorio+algoritmo+versao)
   * 
   * @return caminho para a versao
   */
  public String getDirPath() {
    String dirPath =
      this.algorithmInfo.getAlgorithmRepositoryPath() + "/"
        + this.algorithmInfo.getDirectory() + "/" + AlgorithmInfo.VERSIONS_DIR
        + "/" + this.dir;
    return dirPath;
  }

  /**
   * Obtm o caminho para o diretrio de configurao da verso.
   * 
   * @return O caminho para o diretrio de configurao.
   */
  public String getConfiguratorDirPath() {
    String configurationPath = getDirPath() + "/" + CONFIGURATOR_DIR;
    return configurationPath;
  }

  /**
   * Obtm o caminho para o diretrio de executveis da verso.
   * 
   * @return O caminho para o diretrio de executveis.
   */
  public String getExecutableDirPath() {
    String executablePath = getDirPath() + "/" + BIN_DIR;
    return executablePath;
  }

  /**
   * Obtm o caminho para o arquivo de configurao da verso.
   * 
   * @return O caminho para o arquivo de configurao.
   */
  public String getConfiguratorPath() {
    String configurationPath = getConfiguratorDirPath() + "/" + CONFIG_FILE;
    return configurationPath;
  }

  /**
   * Obtm o caminho para o arquivo de configurao do fluxo.
   * 
   * @return O caminho para o arquivo do fluxo.
   */
  public String getFlowConfiguratorPath() {
    String configurationPath =
      getConfiguratorDirPath() + "/" + FLOW_CONFIG_FILE;
    return configurationPath;
  }

  /**
   * Obtm o caminho para o arquivo de propriedades da verso.
   * 
   * @return O caminho para o arquivo de propriedades.
   */
  public String getPropertiesPath() {
    String propertiesPath = getConfiguratorDirPath() + "/" + PROPERTIES_FILE;
    return propertiesPath;
  }

  /**
   * Obtm o caminho para o diretrio de documentao da verso.
   * 
   * @return O caminho para o diretrio de documentao.
   */
  public String getDocDirPath() {
    String htmlPath = getDirPath() + "/" + DOCUMENTATION_DIR;
    return htmlPath;
  }

  /**
   * Obtm o caminho para o diretrio da plataforma de execuo da verso.
   * 
   * @param platform Plataforma de execuo.
   * 
   * @return O caminho para o diretrio da plataforma de execuo.
   */
  public String getPlatformPath(String platform) {
    String platPath = getDirPath() + "/" + BIN_DIR + "/" + platform;
    return platPath;
  }

  /**
   * Obtm o caminho para o executvel de uma plataforma da verso.
   * 
   * @param platform Plataforma do executvel.
   * @param exec Nome do executvel.
   * 
   * @return O caminho para o executvel de uma plataforma.
   */
  public String getExecFilePath(String platform, String exec) {
    String execPath = getPlatformPath(platform) + "/" + exec;
    return execPath;
  }

  /**
   * Extrai os nmeros de uma verso do nome de seu diretrio.
   * 
   * @param dirName diretrio da verso
   * 
   * @return um array com os trs nmeros (major, minor, patch) ou null se o
   *         nome do arquivo  invlido
   */
  private static AlgorithmVersionId getVersionIdFromDirName(String dirName) {
    try {
      StringTokenizer stok = new StringTokenizer(dirName, "_");
      String prefix = stok.nextToken();
      if (!prefix.equals("v")) {
        return null;
      }
      int major = Integer.parseInt(stok.nextToken());
      int minor = Integer.parseInt(stok.nextToken());
      int patch = Integer.parseInt(stok.nextToken());
      return new AlgorithmVersionId(major, minor, patch);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Retorna uma lista com os valores dos atributos estendidos do algoritmo
   * 
   * @return vector com os valores dos atributos estendidos do algoritmo
   */
  public Map<String, String> getPropertyValues() {
    return this.propertyValues;
  }

  /**
   * Modifica o valor dos atributos estendidos. O mtodo garante a persistencia
   * da modificacao.
   * 
   * @param propertyValues vector com os novos valores dos atributos
   */
  public void setPropertyValues(Hashtable<String, String> propertyValues) {
    /* Seta em memoria a lista de atributos */
    this.propertyValues = propertyValues;
  }

  /**
   * Atribui uma descrio para essa verso do algoritmo.
   * 
   * @param description descrio da verso do algoritmo
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Obtm a descrio da verso do algoritmo.
   * 
   * @return description a descrio da verso do algoritmo
   */
  public String getDescription() {
    return description;
  }

  /**
   * Obtm o tipo da verso.
   * @return o tipo.
   */
  public ConfiguratorType getType() {
    return type;
  }

  /**
   * Atualiza os dados do algoritmo.
   *
   * @param info os dados do algoritmo.
   */
  void setInfo(AlgorithmInfo info) {
    this.algorithmInfo = info;
  }
}
