/*
 * ProjectSynchronizationResult.java
 * $Author$
 * $Revision$ - $Date$
 */
package csbase.logic;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;

/**
 * Modela o resultado de uma operao de sincronizao entre dois projetos.
 */
public class ProjectSynchronizationResult implements Serializable {
  /** Lista de mensagens com os erros de transferncia do servidor A para B */
  private List<String> resultsA;
  /** Lista de mensagens com os erros de transferncia do servidor B para A */
  private List<String> resultsB;
  /** Messagem de resultado */
  private String message;
  /** Causa do erro, quando houver um */
  private Throwable cause;
  /** Indica se a sincronizao foi completada com sucesso. */
  private boolean success;

  /**
   * Constri o resultado da sincronizao com falha.
   * 
   * @param result mensagem de erro
   * @param cause causa da exceo
   * @param resultsA lista de resultado em um sentido
   * @param resultsB lista de resultado em outro sentido
   */
  public ProjectSynchronizationResult(
    String result, Throwable cause, List<String> resultsA, List<String> resultsB) {
    this.success = false;
    this.message = result;
    this.resultsA = resultsA;
    this.resultsB = resultsB;
    this.cause = cause;
  }
  
  /**
   * Constri o resultado da sincronizao com sucesso.
   * 
   * @param resultsA lista de resultado em um sentido
   * @param resultsB lista de resultado em outro sentido
   */
  public ProjectSynchronizationResult(List<String> resultsA,
    List<String> resultsB) {
    if (resultsA.size() + resultsB.size() <= 0) {
      this.success = true;
    } else {
      this.success = false;
    }
    this.resultsA = resultsA;
    this.resultsB = resultsB;
  }

  /**
   * Retorna o resultado geral ocorrido na operao de sincronizao. O texto
   * ser null caso nenhum erro geral tenha ocorrido. Um erro geral  um erro
   * no relacionado  transferncia de algum arquivo em particular. Caso um
   * erro geral ocorra, a operao de transferncia  interrompida no ponto em
   * que ela estiver.
   * Note que, independentemente da ocorrncia ou no de um erro geral, 
   * preciso verificar o resultado da transferncia de cada arquivo.
   * @return A descrio do erro geral ocorrido ou null caso no tenha ocorrido
   *         nenhum erro geral.
   */
  public String getMessage() {
    return message;
  }

  /**
   * Retorna a lista dos resultados das operaes de transferncia de arquivos
   * do projeto A para o projeto B. Essa lista mantm a mesma indexao da
   * lista de arquivos cuja transferncia foi solicitada. Na lista, null
   * significa que a transferncia foi realizada com sucesso. No caso de falha,
   *  retornado um texto descritivo.
   * @return Uma lista com os resultados das operaes de transferncia de
   *         arquivos de A para B: null para cada transferncia realizada com
   *         sucesso ou um texto descritivo no caso de falha.
   */
  public List<String> getResultsA() {
    return Collections.unmodifiableList(resultsA);
  }

  /**
   * Retorna a lista dos resultados das operaes de transferncia de arquivos
   * do projeto B para o projeto A. Essa lista mantm a mesma indexao da
   * lista de arquivos cuja transferncia foi solicitada. Na lista, null
   * significa que a transferncia foi realizada com sucesso. No caso de falha,
   *  retornado um texto descritivo.
   * @return Uma lista com os resultados das operaes de transferncia de
   *         arquivos de B para A: null para cada transferncia realizada com
   *         sucesso ou um texto descritivo no caso de falha.
   */
  public List<String> getResultsB() {
    return Collections.unmodifiableList(resultsB);
  }
  
  /**
   * Indica se a sincronizao foi bem sucedida.
   * @return true se a sincronizao terminou com sucesso.
   */
  public boolean succeeded(){
    return success;
  }
  
  /**
   * Retorna a causa de erro na sincronizao.
   * @return a causa da sincronizao
   */
  public Throwable getCause(){
    return cause;
  }
}
